<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class PrizeAward extends Model
{
    use HasFactory;

    protected $fillable = [
        'tournament_id',
        'category',
        'position',
        'title',
        'description',
        'prize_value',
        'prize_type',
        'additional_benefits',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer',
        'prize_value' => 'decimal:2',
    ];

    // Relationships
    public function tournament()
    {
        return $this->belongsTo(Tournament::class);
    }

    // Scopes
    public function scopeActive(Builder $query): void
    {
        $query->where('is_active', true);
    }

    public function scopeOrdered(Builder $query): void
    {
        $query->orderBy('sort_order')->orderBy('position');
    }

    public function scopeByCategory(Builder $query, string $category): void
    {
        $query->where('category', $category);
    }

    public function scopeByTournament(Builder $query, int $tournamentId): void
    {
        $query->where('tournament_id', $tournamentId);
    }

    // Static methods
    public static function getCategories(): array
    {
        return [
            'team' => 'Team Awards',
            'individual' => 'Individual Awards',
            'special' => 'Special Awards',
            'performance' => 'Performance Awards',
            'fair_play' => 'Fair Play Awards',
        ];
    }

    public static function getPrizeTypes(): array
    {
        return [
            'cash' => 'Cash Prize',
            'trophy' => 'Trophy',
            'medal' => 'Medal',
            'certificate' => 'Certificate',
            'merchandise' => 'Merchandise',
            'experience' => 'Experience',
            'mixed' => 'Mixed Prize',
        ];
    }

    public static function getPositions(): array
    {
        return [
            '1st' => '1st Place',
            '2nd' => '2nd Place',
            '3rd' => '3rd Place',
            '4th' => '4th Place',
            '5th' => '5th Place',
            '6th' => '6th Place',
            '7th' => '7th Place',
            '8th' => '8th Place',
            'participation' => 'Participation',
            'special' => 'Special Recognition',
        ];
    }
}
