<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;

class Player extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'tournament_id',
        'national_id_front',     // Path to national ID front image
        'national_id_back',      // Path to national ID back image  
        'team_id',
        'display_name',
        'bio',
        'avatar',
        'country',
        'city',
        'date_of_birth',
        'gender',
        'phone',
        'discord_username',
        'steam_id',
        'twitch_username',
        'youtube_channel',
        'twitter_handle',
        'instagram_handle',
        'website',
        'player_type',
        'experience_level',
        'primary_game',
        'game_preferences',
        'football_position',
        'jersey_number',
        'is_team_captain',
        'global_rating',
        'global_rank',
        'tournament_rating',
        'tournament_rank',
        'seeding_position',
        'status',
        'verification_status',
        'verified_at',
        'registered_at',
        'last_active_at',
        'otp_code',
        'otp_expires_at',
        'otp_verified',
        'tournament_wins',
        'tournament_losses',
        'tournament_draws',
        'tournament_win_rate',
        'tournament_matches_played',
        'tournament_matches_won',
        'tournament_current_streak',
        'tournament_longest_streak',
        'match_history',
        'performance_metrics',
        'achievements',
        'tournament_stats',
        'rankings_history',
        'total_earnings',
        'tournaments_won',
        'prizes_won',
        'prize_history',
        'notification_preferences',
        'privacy_settings',
        'is_public_profile',
        'allow_messages',
        'show_contact_info',
        'metadata',
        'digital_id',
        'id_number',
        'birth_certificate_number',
        'passport_photo',
        'birth_cert_photo',
        'approval_status',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'national_id_front' => 'string',
        'national_id_back' => 'string',
        'game_preferences' => 'array',
        'global_rating' => 'decimal:2',
        'tournament_rating' => 'decimal:2',
        'tournament_win_rate' => 'decimal:2',
        'total_earnings' => 'decimal:2',
        'verified_at' => 'datetime',
        'registered_at' => 'datetime',
        'last_active_at' => 'datetime',
        'otp_expires_at' => 'datetime',
        'match_history' => 'array',
        'performance_metrics' => 'array',
        'achievements' => 'array',
        'tournament_stats' => 'array',
        'rankings_history' => 'array',
        'prize_history' => 'array',
        'notification_preferences' => 'array',
        'privacy_settings' => 'array',
        'metadata' => 'array',
        'is_public_profile' => 'boolean',
        'allow_messages' => 'boolean',
        'show_contact_info' => 'boolean',
        'is_team_captain' => 'boolean',
        'otp_verified' => 'boolean',
    ];

    protected $dates = [
        'date_of_birth',
        'verified_at',
        'registered_at',
        'last_active_at',
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function tournament(): BelongsTo
    {
        return $this->belongsTo(Tournament::class);
    }

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function matches(): HasMany
    {
        return $this->hasMany(GameMatch::class, 'player1_id')
            ->orWhere('player2_id', $this->id);
    }

    public function matchesAsPlayer1(): HasMany
    {
        return $this->hasMany(GameMatch::class, 'player1_id');
    }

    public function matchesAsPlayer2(): HasMany
    {
        return $this->hasMany(GameMatch::class, 'player2_id');
    }

    public function matchesAsWinner(): HasMany
    {
        return $this->hasMany(GameMatch::class, 'winner_id');
    }

    // Scopes
    public function scopeActive(Builder $query): void
    {
        $query->where('status', 'active');
    }

    public function scopeVerified(Builder $query): void
    {
        $query->where('verification_status', 'verified');
    }

    public function scopeByType(Builder $query, string $type): void
    {
        $query->where('player_type', $type);
    }

    public function scopeByExperience(Builder $query, string $level): void
    {
        $query->where('experience_level', $level);
    }

    public function scopeByCountry(Builder $query, string $country): void
    {
        $query->where('country', $country);
    }

    public function scopeTopRated(Builder $query, int $limit = 10): void
    {
        $query->orderBy('global_rating', 'desc')->limit($limit);
    }

    public function scopeByTournament(Builder $query, int $tournamentId): void
    {
        $query->where('tournament_id', $tournamentId);
    }

    // Advanced Methods
    public function updateLastActive(): void
    {
        $this->update(['last_active_at' => now()]);
    }

    public function calculateWinRate(): float
    {
        $totalMatches = $this->tournament_matches_played;
        return $totalMatches > 0 ? round(($this->tournament_matches_won / $totalMatches) * 100, 2) : 0;
    }

    public function updateTournamentStats(): void
    {
        $wins = $this->matchesAsWinner()->where('tournament_id', $this->tournament_id)->count();
        $totalMatches = $this->matches()->where('tournament_id', $this->tournament_id)->count();
        $losses = $totalMatches - $wins;
        $winRate = $this->calculateWinRate();

        $this->update([
            'tournament_matches_played' => $totalMatches,
            'tournament_matches_won' => $wins,
            'tournament_losses' => $losses,
            'tournament_win_rate' => $winRate,
        ]);
    }

    public function updateGlobalRating(float $newRating): void
    {
        $this->update(['global_rating' => $newRating]);
        $this->updateGlobalRank();
    }

    public function updateGlobalRank(): void
    {
        $rank = static::where('global_rating', '>', $this->global_rating)->count() + 1;
        $this->update(['global_rank' => $rank]);
    }

    public function updateTournamentRating(float $newRating): void
    {
        $this->update(['tournament_rating' => $newRating]);
        $this->updateTournamentRank();
    }

    public function updateTournamentRank(): void
    {
        $rank = static::where('tournament_id', $this->tournament_id)
            ->where('tournament_rating', '>', $this->tournament_rating)
            ->count() + 1;
        $this->update(['tournament_rank' => $rank]);
    }

    public function addAchievement(string $achievement, array $data = []): void
    {
        $achievements = $this->achievements ?? [];
        $achievements[] = [
            'name' => $achievement,
            'earned_at' => now()->toISOString(),
            'data' => $data,
        ];
        $this->update(['achievements' => $achievements]);
    }

    public function hasAchievement(string $achievement): bool
    {
        $achievements = $this->achievements ?? [];
        return collect($achievements)->contains('name', $achievement);
    }

    public function addMatchToHistory(GameMatch $match): void
    {
        $history = $this->match_history ?? [];
        $history[] = [
            'match_id' => $match->id,
            'opponent' => $match->player1_id === $this->id ? $match->player2_id : $match->player1_id,
            'result' => $match->winner_id === $this->id ? 'win' : 'loss',
            'score' => $match->player1_id === $this->id ? 
                [$match->player1_score, $match->player2_score] : 
                [$match->player2_score, $match->player1_score],
            'played_at' => $match->completed_at?->toISOString(),
        ];
        $this->update(['match_history' => $history]);
    }

    public function getPerformanceMetrics(): array
    {
        $metrics = $this->performance_metrics ?? [];
        
        // Calculate additional metrics
        $recentMatches = $this->matches()->where('completed_at', '>=', now()->subDays(30))->get();
        $recentWins = $recentMatches->where('winner_id', $this->id)->count();
        $recentTotal = $recentMatches->count();
        
        $metrics['recent_win_rate'] = $recentTotal > 0 ? round(($recentWins / $recentTotal) * 100, 2) : 0;
        $metrics['matches_last_30_days'] = $recentTotal;
        $metrics['wins_last_30_days'] = $recentWins;
        
        return $metrics;
    }

    public function getAge(): ?int
    {
        return $this->date_of_birth ? $this->date_of_birth->age : null;
    }

    public function isVerified(): bool
    {
        return $this->verification_status === 'verified';
    }

    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    public function isProfessional(): bool
    {
        return in_array($this->player_type, ['professional', 'legend']);
    }

    public function getFullLocation(): string
    {
        $parts = array_filter([$this->city, $this->country]);
        return implode(', ', $parts) ?: 'Location not specified';
    }

    public function getSocialLinks(): array
    {
        return array_filter([
            'discord' => $this->discord_username,
            'steam' => $this->steam_id,
            'twitch' => $this->twitch_username,
            'youtube' => $this->youtube_channel,
            'twitter' => $this->twitter_handle,
            'instagram' => $this->instagram_handle,
            'website' => $this->website,
        ]);
    }

    public function getDisplayStats(): array
    {
        return [
            'matches_played' => $this->tournament_matches_played,
            'matches_won' => $this->tournament_matches_won,
            'win_rate' => $this->tournament_win_rate,
            'current_streak' => $this->tournament_current_streak,
            'longest_streak' => $this->tournament_longest_streak,
            'global_rating' => $this->global_rating,
            'global_rank' => $this->global_rank,
            'tournament_rating' => $this->tournament_rating,
            'tournament_rank' => $this->tournament_rank,
            'total_earnings' => $this->total_earnings,
            'tournaments_won' => $this->tournaments_won,
        ];
    }

    // OTP Methods for team registration
    public function generateOtp(): string
    {
        $otp = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        $this->update([
            'otp_code' => $otp,
            'otp_expires_at' => now()->addHours(24),
            'otp_verified' => false,
        ]);
        return $otp;
    }

    public function verifyOtp(string $otp): bool
    {
        if ($this->otp_code === $otp && $this->otp_expires_at > now()) {
            $this->update([
                'otp_verified' => true,
                'otp_code' => null,
                'otp_expires_at' => null,
            ]);
            return true;
        }
        return false;
    }

    public function isTeamCaptain(): bool
    {
        return $this->is_team_captain;
    }

    /**
     * Generate a unique Digital ID for the player
     */
    public function generateDigitalId(): string
    {
        $digitalId = 'PID-' . str_pad($this->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($this->id . time()), 0, 8));
        
        $this->update(['digital_id' => $digitalId]);
        
        return $digitalId;
    }

    /**
     * Check if player has a Digital ID
     */
    public function hasDigitalId(): bool
    {
        return !empty($this->digital_id);
    }

    /**
     * Get the player's avatar with fallback to default image
     */
    public function getAvatarAttribute($value): string
    {
        // If no avatar is set, return default
        if (empty($value)) {
            return asset('player.jpg');
        }

        // If it's already a full URL, return it
        if (str_starts_with($value, 'http://') || str_starts_with($value, 'https://')) {
            return $value;
        }

        // Handle storage paths
        if (str_starts_with($value, 'storage/') || str_starts_with($value, '/storage/')) {
            $storagePath = str_starts_with($value, '/') ? substr($value, 1) : $value;
            $fullPath = public_path($storagePath);
            if (file_exists($fullPath)) {
                return asset($storagePath);
            }
        }

        // Handle public directory files
        $publicPath = public_path(ltrim($value, '/'));
        if (file_exists($publicPath)) {
            return asset(ltrim($value, '/'));
        }

        // If file doesn't exist, return default
        return asset('player.jpg');
    }

    /**
     * Get the player's national ID front with fallback to default image
     */
    public function getNationalIdFrontAttribute($value): string
    {
        // If no ID front is set, return default
        if (empty($value)) {
            return asset('player.jpg');
        }

        // If it's already a full URL, return it
        if (str_starts_with($value, 'http://') || str_starts_with($value, 'https://')) {
            return $value;
        }

        // Handle storage paths
        if (str_starts_with($value, 'storage/') || str_starts_with($value, '/storage/')) {
            $storagePath = str_starts_with($value, '/') ? substr($value, 1) : $value;
            $fullPath = public_path($storagePath);
            if (file_exists($fullPath)) {
                return asset($storagePath);
            }
        }

        // Handle public directory files
        $publicPath = public_path(ltrim($value, '/'));
        if (file_exists($publicPath)) {
            return asset(ltrim($value, '/'));
        }

        // If file doesn't exist, return default
        return asset('player.jpg');
    }

    /**
     * Get the player's national ID back with fallback to default image
     */
    public function getNationalIdBackAttribute($value): string
    {
        // If no ID back is set, return default
        if (empty($value)) {
            return asset('player.jpg');
        }

        // If it's already a full URL, return it
        if (str_starts_with($value, 'http://') || str_starts_with($value, 'https://')) {
            return $value;
        }

        // Handle storage paths
        if (str_starts_with($value, 'storage/') || str_starts_with($value, '/storage/')) {
            $storagePath = str_starts_with($value, '/') ? substr($value, 1) : $value;
            $fullPath = public_path($storagePath);
            if (file_exists($fullPath)) {
                return asset($storagePath);
            }
        }

        // Handle public directory files
        $publicPath = public_path(ltrim($value, '/'));
        if (file_exists($publicPath)) {
            return asset(ltrim($value, '/'));
        }

        // If file doesn't exist, return default
        return asset('player.jpg');
    }

    // Boot method for automatic updates
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($player) {
            if (!$player->registered_at) {
                $player->registered_at = now();
            }
            if (!$player->last_active_at) {
                $player->last_active_at = now();
            }
        });

        static::updating(function ($player) {
            if ($player->isDirty('status') && $player->status === 'active') {
                $player->last_active_at = now();
            }
        });
    }
}
