<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Builder;

class Permission extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'display_name',
        'description',
        'category',
        'is_system',
        'is_active',
        'sort_order',
        'metadata',
    ];

    protected $casts = [
        'is_system' => 'boolean',
        'is_active' => 'boolean',
        'metadata' => 'array',
    ];

    // Relationships
    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class, 'permission_role')
                    ->withTimestamps();
    }

    // Scopes
    public function scopeActive(Builder $query): void
    {
        $query->where('is_active', true);
    }

    public function scopeSystem(Builder $query): void
    {
        $query->where('is_system', true);
    }

    public function scopeCustom(Builder $query): void
    {
        $query->where('is_system', false);
    }

    public function scopeByCategory(Builder $query, string $category): void
    {
        $query->where('category', $category);
    }

    public function scopeOrdered(Builder $query): void
    {
        $query->orderBy('category')->orderBy('sort_order')->orderBy('name');
    }

    // Methods
    public function isSystemPermission(): bool
    {
        return $this->is_system;
    }

    public function isActive(): bool
    {
        return $this->is_active;
    }

    public function deactivate(): void
    {
        $this->update(['is_active' => false]);
    }

    public function activate(): void
    {
        $this->update(['is_active' => true]);
    }

    public function assignToRole(Role $role): void
    {
        $this->roles()->attach($role->id);
    }

    public function removeFromRole(Role $role): void
    {
        $this->roles()->detach($role->id);
    }

    public function getResourceAttribute(): string
    {
        return explode('.', $this->name)[0] ?? '';
    }

    public function getActionAttribute(): string
    {
        $parts = explode('.', $this->name);
        return $parts[1] ?? '';
    }

    public function getCategoryIconAttribute(): string
    {
        return match($this->category) {
            'dashboard' => 'layout-grid',
            'tournaments' => 'trophy',
            'players' => 'users',
            'teams' => 'users-round',
            'matches' => 'gamepad',
            'brackets' => 'list-ordered',
            'users' => 'user-cog',
            'roles' => 'shield',
            'permissions' => 'key',
            'payments' => 'credit-card',
            'settings' => 'settings',
            'media' => 'hard-drive',
            'analytics' => 'bar-chart-3',
            'system' => 'server',
            default => 'lock',
        };
    }

    public function getCategoryColorAttribute(): string
    {
        return match($this->category) {
            'dashboard' => 'text-blue-600',
            'tournaments' => 'text-yellow-600',
            'players' => 'text-green-600',
            'teams' => 'text-purple-600',
            'matches' => 'text-orange-600',
            'brackets' => 'text-indigo-600',
            'users' => 'text-gray-600',
            'roles' => 'text-blue-700',
            'permissions' => 'text-red-600',
            'payments' => 'text-emerald-600',
            'settings' => 'text-slate-600',
            'media' => 'text-pink-600',
            'analytics' => 'text-cyan-600',
            'system' => 'text-red-700',
            default => 'text-gray-500',
        };
    }

    // Static methods
    public static function getCategories(): array
    {
        return [
            'dashboard' => 'Dashboard',
            'tournaments' => 'Tournaments',
            'players' => 'Players',
            'teams' => 'Teams',
            'matches' => 'Matches',
            'brackets' => 'Brackets',
            'users' => 'Users',
            'roles' => 'Roles',
            'permissions' => 'Permissions',
            'payments' => 'Payments',
            'settings' => 'Settings',
            'media' => 'Media',
            'analytics' => 'Analytics',
            'system' => 'System',
        ];
    }

    public static function getActions(): array
    {
        return [
            'view' => 'View',
            'create' => 'Create',
            'edit' => 'Edit',
            'delete' => 'Delete',
            'manage' => 'Manage',
            'assign' => 'Assign',
            'start' => 'Start',
            'complete' => 'Complete',
            'verify' => 'Verify',
            'impersonate' => 'Impersonate',
            'upload' => 'Upload',
            'export' => 'Export',
            'backup' => 'Backup',
            'maintenance' => 'Maintenance',
        ];
    }
}