<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class NotificationSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'notification_type',
        'category',
        'enabled',
        'description',
        'is_critical',
        'channels',
    ];

    protected $casts = [
        'enabled' => 'boolean',
        'is_critical' => 'boolean',
        'channels' => 'array',
    ];

    /**
     * Check if this notification is enabled system-wide
     */
    public function isEnabled(): bool
    {
        return $this->enabled;
    }

    /**
     * Get all categories with their notification counts
     */
    public static function getCategoriesWithCounts(): array
    {
        return Cache::remember('notification_categories_counts', 3600, function () {
            $categories = self::selectRaw('category, COUNT(*) as total, SUM(CASE WHEN enabled = 1 THEN 1 ELSE 0 END) as enabled_count')
                ->groupBy('category')
                ->get()
                ->keyBy('category')
                ->toArray();

            return $categories;
        });
    }

    /**
     * Get all notifications grouped by category
     */
    public static function getGroupedByCategory(): array
    {
        return Cache::remember('notifications_grouped_by_category', 3600, function () {
            return self::orderBy('category')->orderBy('notification_type')->get()->groupBy('category')->toArray();
        });
    }

    /**
     * Check if a specific notification type is enabled
     */
    public static function isNotificationEnabled(string $notificationType): bool
    {
        return Cache::remember("notification_enabled_{$notificationType}", 3600, function () use ($notificationType) {
            $setting = self::where('notification_type', $notificationType)->first();
            return $setting ? $setting->enabled : true; // Default to enabled if not found
        });
    }

    /**
     * Clear notification cache
     */
    public static function clearCache(): void
    {
        Cache::forget('notification_categories_counts');
        Cache::forget('notifications_grouped_by_category');
        
        // Clear individual notification cache
        self::all()->each(function ($setting) {
            Cache::forget("notification_enabled_{$setting->notification_type}");
        });
    }

    /**
     * Boot method to clear cache on changes
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function () {
            self::clearCache();
        });

        static::deleted(function () {
            self::clearCache();
        });
    }
}

