<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\Storage;

class Media extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'file_name',
        'mime_type',
        'extension',
        'size',
        'path',
        'disk',
        'collection_name',
        'metadata',
        'custom_properties',
        'alt_text',
        'description',
        'user_id',
        'model_type',
        'model_id',
        'order_column',
    ];

    protected $casts = [
        'metadata' => 'array',
        'custom_properties' => 'array',
        'size' => 'integer',
    ];

    protected $appends = [
        'url',
        'human_readable_size',
        'is_image',
        'is_video',
        'is_audio',
        'is_document',
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function model(): MorphTo
    {
        return $this->morphTo();
    }

    // Scopes
    public function scopeImages($query)
    {
        return $query->where('mime_type', 'like', 'image/%');
    }

    public function scopeVideos($query)
    {
        return $query->where('mime_type', 'like', 'video/%');
    }

    public function scopeAudios($query)
    {
        return $query->where('mime_type', 'like', 'audio/%');
    }

    public function scopeDocuments($query)
    {
        return $query->whereIn('mime_type', [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'text/plain',
            'text/csv',
        ]);
    }

    public function scopeInCollection($query, $collection)
    {
        // return $query->where('collection_name', $collection);
        return $query;
    }

    public function scopeForModel($query, $model)
    {
        return $query->where('model_type', get_class($model))
                    ->where('model_id', $model->id);
    }

    // Accessors
    public function getUrlAttribute(): string
    {
        return Storage::disk($this->disk)->url($this->path);
    }

    public function getHumanReadableSizeAttribute(): string
    {
        $bytes = $this->size;
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }

    public function getIsImageAttribute(): bool
    {
        return str_starts_with($this->mime_type, 'image/');
    }

    public function getIsVideoAttribute(): bool
    {
        return str_starts_with($this->mime_type, 'video/');
    }

    public function getIsAudioAttribute(): bool
    {
        return str_starts_with($this->mime_type, 'audio/');
    }

    public function getIsDocumentAttribute(): bool
    {
        return in_array($this->mime_type, [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'text/plain',
            'text/csv',
        ]);
    }

    // Helper methods
    public function delete(): bool
    {
        // Delete the physical file
        if (Storage::disk($this->disk)->exists($this->path)) {
            Storage::disk($this->disk)->delete($this->path);
        }

        // Delete the database record
        return parent::delete();
    }

    public function move(string $newPath): bool
    {
        $oldPath = $this->path;
        
        if (Storage::disk($this->disk)->move($oldPath, $newPath)) {
            $this->update(['path' => $newPath]);
            return true;
        }
        
        return false;
    }

    public function copy(string $newPath): ?self
    {
        if (Storage::disk($this->disk)->copy($this->path, $newPath)) {
            return self::create([
                'name' => $this->name,
                'file_name' => basename($newPath),
                'mime_type' => $this->mime_type,
                'extension' => $this->extension,
                'size' => $this->size,
                'path' => $newPath,
                'disk' => $this->disk,
                'collection_name' => $this->collection_name,
                'metadata' => $this->metadata,
                'custom_properties' => $this->custom_properties,
                'alt_text' => $this->alt_text,
                'description' => $this->description,
                'user_id' => $this->user_id,
            ]);
        }
        
        return null;
    }

    public static function createFromUpload($file, array $options = []): self
    {
        $originalName = $file->getClientOriginalName();
        $extension = $file->getClientOriginalExtension();
        $mimeType = $file->getMimeType();
        $size = $file->getSize();
        
        $fileName = time() . '_' . uniqid() . '.' . $extension;
        $collection = $options['collection'] ?? 'default';
        $disk = $options['disk'] ?? 'public';
        $directory = $options['directory'] ?? 'uploads/' . $collection;
        
        $path = $file->storeAs($directory, $fileName, $disk);
        
        return self::create([
            'name' => $originalName,
            'file_name' => $fileName,
            'mime_type' => $mimeType,
            'extension' => $extension,
            'size' => $size,
            'path' => $path,
            'disk' => $disk,
            'collection_name' => $collection,
            'metadata' => $options['metadata'] ?? [],
            'custom_properties' => $options['custom_properties'] ?? [],
            'alt_text' => $options['alt_text'] ?? null,
            'description' => $options['description'] ?? null,
            'user_id' => auth()->id(),
            'model_type' => $options['model_type'] ?? null,
            'model_id' => $options['model_id'] ?? null,
        ]);
    }
}
