<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;
use Carbon\Carbon;

class GameMatch extends Model
{
    use HasFactory;

    protected $table = 'matches';   

    protected $fillable = [
        'name',
        'slug',
        'match_code',
        'match_type',
        'format',
        'best_of_games',
        'time_limit_minutes',
        'rules',
        'settings',
        'description',
        'notes',
        'tournament_id',
        'bracket_id',
        'round',
        'match_number',
        'player1_id',
        'player2_id',
        'team1_id',
        'team2_id',
        'player1_score',
        'player2_score',
        'game_scores',
        'round_scores',
        'player1_games_won',
        'player2_games_won',
        'team1_games_won',
        'team2_games_won',
        'player1_rating_change',
        'player2_rating_change',
        'winner_id',
        'winning_team_id',
        'status',
        'progress_status',
        'result_type',
        'result_notes',
        'verification_status',
        'verified_at',
        'verified_by',
        'scheduled_time',
        'estimated_duration_minutes',
        'actual_duration_minutes',
        'started_at',
        'warmup_started_at',
        'warmup_ended_at',
        'paused_at',
        'resumed_at',
        'pause_duration_minutes',
        'completed_at',
        'is_streamed',
        'stream_url',
        'twitch_url',
        'youtube_url',
        'stream_key',
        'broadcaster_notes',
        'allow_spectators',
        'spectator_count',
        'peak_spectator_count',
        'spectator_analytics',
        'match_statistics',
        'performance_metrics',
        'highlight_moments',
        'match_quality_score',
        'total_actions',
        'average_action_per_minute',
        'server_location',
        'game_server',
        'connection_quality',
        'technical_issues',
        'system_logs',
        'admin_notes',
        'referee_notes',
        'referee_id',
        'created_by',
        'updated_by',
        'metadata',
        'last_activity_at',
        'archived_at',
    ];

    protected $casts = [
        'scheduled_time' => 'datetime',
        'started_at' => 'datetime',
        'warmup_started_at' => 'datetime',
        'warmup_ended_at' => 'datetime',
        'paused_at' => 'datetime',
        'resumed_at' => 'datetime',
        'completed_at' => 'datetime',
        'verified_at' => 'datetime',
        'last_activity_at' => 'datetime',
        'archived_at' => 'datetime',
        'rules' => 'array',
        'settings' => 'array',
        'game_scores' => 'array',
        'round_scores' => 'array',
        'spectator_analytics' => 'array',
        'match_statistics' => 'array',
        'performance_metrics' => 'array',
        'highlight_moments' => 'array',
        'technical_issues' => 'array',
        'system_logs' => 'array',
        'admin_notes' => 'array',
        'referee_notes' => 'array',
        'metadata' => 'array',
        'player1_rating_change' => 'decimal:2',
        'player2_rating_change' => 'decimal:2',
        'match_quality_score' => 'decimal:2',
        'is_streamed' => 'boolean',
        'allow_spectators' => 'boolean',
    ];

    protected $dates = [
        'scheduled_time',
        'started_at',
        'warmup_started_at',
        'warmup_ended_at',
        'paused_at',
        'resumed_at',
        'completed_at',
        'verified_at',
        'last_activity_at',
        'archived_at',
    ];

    // Relationships
    public function tournament(): BelongsTo
    {
        return $this->belongsTo(Tournament::class);
    }

    public function bracket(): BelongsTo
    {
        return $this->belongsTo(Bracket::class);
    }

    public function player1(): BelongsTo
    {
        return $this->belongsTo(Player::class, 'player1_id');
    }

    public function player2(): BelongsTo
    {
        return $this->belongsTo(Player::class, 'player2_id');
    }

    public function team1(): BelongsTo
    {
        return $this->belongsTo(Team::class, 'team1_id');
    }

    public function team2(): BelongsTo
    {
        return $this->belongsTo(Team::class, 'team2_id');
    }

    public function winner(): BelongsTo
    {
        return $this->belongsTo(Player::class, 'winner_id');
    }

    public function winningTeam(): BelongsTo
    {
        return $this->belongsTo(Team::class, 'winning_team_id');
    }

    public function referee(): BelongsTo
    {
        return $this->belongsTo(User::class, 'referee_id');
    }

    public function verifiedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updatedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    // Scopes
    public function scopeScheduled(Builder $query): void
    {
        $query->where('status', 'scheduled');
    }

    public function scopeInProgress(Builder $query): void
    {
        $query->where('status', 'in_progress');
    }

    public function scopeCompleted(Builder $query): void
    {
        $query->where('status', 'completed');
    }

    public function scopeCancelled(Builder $query): void
    {
        $query->where('status', 'cancelled');
    }

    public function scopeStreamed(Builder $query): void
    {
        $query->where('is_streamed', true);
    }

    public function scopeUpcoming(Builder $query): void
    {
        $query->where('scheduled_time', '>=', now());
    }

    public function scopeLive(Builder $query): void
    {
        $query->where('progress_status', 'in_progress');
    }

    public function scopeVerified(Builder $query): void
    {
        $query->where('verification_status', 'verified');
    }

    public function scopeByTournament(Builder $query, $tournamentId): void
    {
        $query->where('tournament_id', $tournamentId);
    }

    public function scopeByPlayer(Builder $query, $playerId): void
    {
        $query->where(function($q) use ($playerId) {
            $q->where('player1_id', $playerId)
              ->orWhere('player2_id', $playerId);
        });
    }

    public function scopeByTeam(Builder $query, $teamId): void
    {
        $query->where(function($q) use ($teamId) {
            $q->where('team1_id', $teamId)
              ->orWhere('team2_id', $teamId);
        });
    }

    // Mutators
    public function setNameAttribute($value)
    {
        $this->attributes['name'] = $value;
        if (empty($this->attributes['slug'])) {
            $this->attributes['slug'] = Str::slug($value);
        }
    }

    public function setMatchCodeAttribute($value)
    {
        if (empty($value)) {
            $this->attributes['match_code'] = 'MATCH-' . strtoupper(Str::random(8));
        } else {
            $this->attributes['match_code'] = $value;
        }
    }

    // Accessors
    public function getDurationAttribute()
    {
        if ($this->started_at && $this->completed_at) {
            return $this->started_at->diffInMinutes($this->completed_at);
        }
        return null;
    }

    public function getIsLiveAttribute()
    {
        return $this->progress_status === 'in_progress';
    }

    public function getIsUpcomingAttribute()
    {
        return $this->scheduled_time && $this->scheduled_time->isFuture();
    }

    public function getIsCompletedAttribute()
    {
        return $this->status === 'completed';
    }

    public function getWinnerNameAttribute()
    {
        if ($this->winner_id) {
            return $this->winner->display_name ?? 'Unknown Player';
        }
        if ($this->winning_team_id) {
            return $this->winningTeam->name ?? 'Unknown Team';
        }
        return null;
    }

    public function getParticipantNamesAttribute()
    {
        $participants = [];
        
        if ($this->player1_id) {
            $participants[] = $this->player1->display_name ?? 'Player 1';
        }
        if ($this->player2_id) {
            $participants[] = $this->player2->display_name ?? 'Player 2';
        }
        if ($this->team1_id) {
            $participants[] = $this->team1->name ?? 'Team 1';
        }
        if ($this->team2_id) {
            $participants[] = $this->team2->name ?? 'Team 2';
        }
        
        return implode(' vs ', $participants);
    }

    // Advanced Methods
    public function startMatch()
    {
        if ($this->status !== 'scheduled') {
            throw new \Exception('Match is not scheduled');
        }

        $this->update([
            'status' => 'in_progress',
            'progress_status' => 'in_progress',
            'started_at' => now(),
            'last_activity_at' => now(),
        ]);

        return $this;
    }

    public function pauseMatch()
    {
        if ($this->progress_status !== 'in_progress') {
            throw new \Exception('Match is not in progress');
        }

        $this->update([
            'progress_status' => 'paused',
            'paused_at' => now(),
            'last_activity_at' => now(),
        ]);

        return $this;
    }

    public function resumeMatch()
    {
        if ($this->progress_status !== 'paused') {
            throw new \Exception('Match is not paused');
        }

        $pauseDuration = $this->paused_at ? $this->paused_at->diffInMinutes(now()) : 0;

        $this->update([
            'progress_status' => 'in_progress',
            'resumed_at' => now(),
            'pause_duration_minutes' => $this->pause_duration_minutes + $pauseDuration,
            'last_activity_at' => now(),
        ]);

        return $this;
    }

    public function completeMatch($winnerId = null, $winningTeamId = null, $resultType = 'normal')
    {
        if ($this->status === 'completed') {
            throw new \Exception('Match is already completed');
        }

        $this->update([
            'status' => 'completed',
            'progress_status' => 'completed',
            'winner_id' => $winnerId,
            'winning_team_id' => $winningTeamId,
            'result_type' => $resultType,
            'completed_at' => now(),
            'last_activity_at' => now(),
        ]);

        // Update participant statistics
        $this->updateParticipantStats();

        return $this;
    }

    public function updateScore($player1Score = null, $player2Score = null, $gameScores = null)
    {
        $updates = [];
        
        if ($player1Score !== null) {
            $updates['player1_score'] = $player1Score;
        }
        if ($player2Score !== null) {
            $updates['player2_score'] = $player2Score;
        }
        if ($gameScores !== null) {
            $updates['game_scores'] = $gameScores;
        }

        $updates['last_activity_at'] = now();

        $this->update($updates);

        return $this;
    }

    public function updateParticipantStats()
    {
        if (!$this->isCompleted) {
            return;
        }

        // Update player statistics
        if ($this->player1_id) {
            $this->player1->updateMatchStats($this);
        }
        if ($this->player2_id) {
            $this->player2->updateMatchStats($this);
        }

        // Update team statistics
        if ($this->team1_id) {
            $this->team1->updateMatchStats($this);
        }
        if ($this->team2_id) {
            $this->team2->updateMatchStats($this);
        }
    }

    public function canStart()
    {
        return $this->status === 'scheduled' && 
               $this->scheduled_time && 
               $this->scheduled_time->isPast();
    }

    public function canPause()
    {
        return $this->progress_status === 'in_progress';
    }

    public function canResume()
    {
        return $this->progress_status === 'paused';
    }

    public function canComplete()
    {
        return in_array($this->status, ['scheduled', 'in_progress']) && 
               $this->progress_status !== 'cancelled';
    }

    public function getDetailedStats()
    {
        return [
            'duration' => $this->duration,
            'is_live' => $this->is_live,
            'is_upcoming' => $this->is_upcoming,
            'is_completed' => $this->is_completed,
            'winner_name' => $this->winner_name,
            'participant_names' => $this->participant_names,
            'spectator_count' => $this->spectator_count,
            'peak_spectator_count' => $this->peak_spectator_count,
            'match_quality_score' => $this->match_quality_score,
            'total_actions' => $this->total_actions,
            'average_action_per_minute' => $this->average_action_per_minute,
        ];
    }

    public function generateMatchCode()
    {
        $this->match_code = 'MATCH-' . strtoupper(Str::random(8));
        $this->save();
        return $this->match_code;
    }

    public function archive()
    {
        $this->update([
            'archived_at' => now(),
        ]);
    }

    public function unarchive()
    {
        $this->update([
            'archived_at' => null,
        ]);
    }
}
