<?php

namespace App\Models\Frontend;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class VlogVideo extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'title',
        'description',
        'platform',
        'video_url',
        'embed_url',
        'thumbnail_url',
        'duration',
        'views',
        'likes',
        'category',
        'tags',
        'featured',
        'status',
        'published_at',
        'sort_order',
    ];

    protected $casts = [
        'tags' => 'array',
        'featured' => 'boolean',
        'published_at' => 'datetime',
    ];

    public function scopePublished($query)
    {
        return $query->where('status', 'published');
    }

    public function scopeFeatured($query)
    {
        return $query->where('featured', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order', 'asc')->orderBy('published_at', 'desc');
    }

    public function scopeByPlatform($query, $platform)
    {
        return $query->where('platform', $platform);
    }

    public function scopeByCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    public static function getPlatforms()
    {
        return [
            'youtube' => 'YouTube',
            'instagram' => 'Instagram',
            'facebook' => 'Facebook',
            'tiktok' => 'TikTok',
            'twitter' => 'Twitter/X',
            'linkedin' => 'LinkedIn',
            'vimeo' => 'Vimeo',
        ];
    }

    public static function getCategories()
    {
        return [
            'all' => 'All Videos',
            'matches' => 'Match Highlights',
            'interviews' => 'Player Interviews',
            'behind-scenes' => 'Behind the Scenes',
            'training' => 'Training Sessions',
            'celebrations' => 'Celebrations',
            'awards' => 'Award Ceremonies',
            'fans' => 'Fan Content',
            'announcements' => 'Announcements',
            'tutorials' => 'Tutorials',
        ];
    }

    public function getEmbedCodeAttribute()
    {
        switch ($this->platform) {
            case 'youtube':
                $videoId = $this->extractYouTubeId($this->video_url);
                return $videoId ? "https://www.youtube.com/embed/{$videoId}" : null;
            
            case 'instagram':
                return $this->embed_url;
            
            case 'facebook':
                return $this->embed_url;
            
            case 'tiktok':
                return $this->embed_url;
            
            default:
                return $this->embed_url;
        }
    }

    private function extractYouTubeId($url)
    {
        $pattern = '/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/i';
        if (preg_match($pattern, $url, $matches)) {
            return $matches[1];
        }
        return null;
    }

    public function getPlatformIconAttribute()
    {
        $icons = [
            'youtube' => '🎥',
            'instagram' => '📷',
            'facebook' => '📘',
            'tiktok' => '🎵',
            'twitter' => '🐦',
            'linkedin' => '💼',
            'vimeo' => '🎬',
        ];

        return $icons[$this->platform] ?? '📹';
    }

    public function getFormattedDurationAttribute()
    {
        if (!$this->duration) return null;
        
        $seconds = $this->duration;
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;

        if ($hours > 0) {
            return sprintf('%02d:%02d:%02d', $hours, $minutes, $secs);
        } else {
            return sprintf('%02d:%02d', $minutes, $secs);
        }
    }

    public function getFormattedViewsAttribute()
    {
        if (!$this->views) return '0';
        
        if ($this->views >= 1000000) {
            return round($this->views / 1000000, 1) . 'M';
        } elseif ($this->views >= 1000) {
            return round($this->views / 1000, 1) . 'K';
        }
        
        return number_format($this->views);
    }
} 