<?php

namespace App\Models\Frontend;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class NewsArticle extends Model
{
    use HasFactory;

    protected $table = 'frontend_news_articles';

    protected $fillable = [
        'title',
        'excerpt',
        'content',
        'category',
        'author',
        'author_image',
        'date',
        'read_time',
        'image',
        'tags',
        'views',
        'likes',
        'featured',
        'status',
        'published_at',
        'slug',
        'meta_description',
        'meta_keywords'
    ];

    protected $casts = [
        'tags' => 'array',
        'featured' => 'boolean',
        'published_at' => 'datetime',
        'date' => 'date',
        'views' => 'integer',
        'likes' => 'integer'
    ];

    protected $dates = [
        'published_at',
        'date',
        'created_at',
        'updated_at'
    ];

    // Scopes
    public function scopeFeatured($query)
    {
        return $query->where('featured', true);
    }

    public function scopePublished($query)
    {
        return $query->where('status', 'published');
    }

    public function scopeByCategory($query, $category)
    {
        if ($category && $category !== 'all') {
            return $query->where('category', $category);
        }
        return $query;
    }

    // Accessors
    public function getRouteKeyName()
    {
        return 'slug';
    }

    public function getExcerptAttribute($value)
    {
        return $value ?: \Str::limit(strip_tags($this->content), 150);
    }

    // Mutators
    public function setTitleAttribute($value)
    {
        $this->attributes['title'] = $value;
        $this->attributes['slug'] = \Str::slug($value);
    }

    // Helper methods
    public function incrementViews()
    {
        $this->increment('views');
    }

    public function incrementLikes()
    {
        $this->increment('likes');
    }

    public static function getCategories()
    {
        return [
            'all' => 'All News',
            'tournament' => 'Tournament Updates',
            'registration' => 'Registration',
            'teams' => 'Team News',
            'results' => 'Match Results',
            'announcements' => 'Announcements'
        ];
    }
} 