<?php

namespace App\Models\Frontend;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Fixture extends Model
{
    use HasFactory;

    protected $table = 'frontend_fixtures';

    protected $fillable = [
        'date',
        'time',
        'team_a_name',
        'team_a_logo',
        'team_b_name',
        'team_b_logo',
        'venue',
        'status',
        'result',
        'team_a_score',
        'team_b_score',
        'live_minute',
        'tournament_round',
        'match_type',
        'referee',
        'weather',
        'attendance'
    ];

    protected $casts = [
        'date' => 'date',
        'team_a_score' => 'integer',
        'team_b_score' => 'integer',
        'live_minute' => 'integer',
        'attendance' => 'integer'
    ];

    protected $dates = [
        'date',
        'created_at',
        'updated_at'
    ];

    // Scopes
    public function scopeUpcoming($query)
    {
        return $query->where('status', 'upcoming');
    }

    public function scopeLive($query)
    {
        return $query->where('status', 'live');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeByDate($query, $date)
    {
        return $query->whereDate('date', $date);
    }

    public function scopeByVenue($query, $venue)
    {
        return $query->where('venue', $venue);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('date')->orderBy('time');
    }

    // Accessors
    public function getTeamALogoUrlAttribute()
    {
        return $this->team_a_logo ? asset($this->team_a_logo) : null;
    }

    public function getTeamBLogoUrlAttribute()
    {
        return $this->team_b_logo ? asset($this->team_b_logo) : null;
    }

    public function getFormattedDateAttribute()
    {
        return $this->date ? $this->date->format('M d, Y') : null;
    }

    public function getFormattedTimeAttribute()
    {
        return $this->time ? Carbon::parse($this->time)->format('H:i') : null;
    }

    public function getDisplayResultAttribute()
    {
        if ($this->status === 'completed') {
            return $this->team_a_score . ' - ' . $this->team_b_score;
        } elseif ($this->status === 'live') {
            return $this->team_a_score . ' - ' . $this->team_b_score . " ({$this->live_minute}')";
        }
        return '-';
    }

    public function getStatusColorAttribute()
    {
        $colors = [
            'upcoming' => 'text-blue-600',
            'live' => 'text-green-600',
            'completed' => 'text-gray-600'
        ];

        return $colors[$this->status] ?? 'text-gray-500';
    }

    // Static methods
    public static function getStatuses()
    {
        return [
            'upcoming' => 'Upcoming',
            'live' => 'Live',
            'completed' => 'Completed'
        ];
    }

    public static function getVenues()
    {
        return self::distinct('venue')->pluck('venue')->filter()->sort()->values();
    }
} 