<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Crypt;

class EmailCredential extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'email',
        'provider',
        'smtp_host',
        'smtp_port',
        'smtp_username',
        'smtp_password',
        'smtp_encryption',
        'is_active',
        'instructions',
        'metadata',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'metadata' => 'array',
        'smtp_port' => 'integer',
    ];

    protected $hidden = [
        'smtp_password',
    ];

    // Relationships
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'email_user')
                    ->withPivot(['can_send', 'can_receive', 'permissions', 'granted_at', 'expires_at'])
                    ->withTimestamps();
    }

    // Scopes
    public function scopeActive(Builder $query): void
    {
        $query->where('is_active', true);
    }

    public function scopeByProvider(Builder $query, string $provider): void
    {
        $query->where('provider', $provider);
    }

    // Accessors & Mutators
    public function getSmtpPasswordAttribute($value)
    {
        return $value ? Crypt::decryptString($value) : null;
    }

    public function setSmtpPasswordAttribute($value)
    {
        $this->attributes['smtp_password'] = $value ? Crypt::encryptString($value) : null;
    }

    // Methods
    public function hasUserAccess(User $user): bool
    {
        return $this->users()->where('user_id', $user->id)->exists();
    }

    public function grantUserAccess(User $user, array $permissions = []): void
    {
        $this->users()->syncWithoutDetaching([
            $user->id => [
                'can_send' => $permissions['can_send'] ?? true,
                'can_receive' => $permissions['can_receive'] ?? true,
                'permissions' => $permissions['permissions'] ?? null,
                'granted_at' => now(),
            ]
        ]);
    }

    public function revokeUserAccess(User $user): void
    {
        $this->users()->detach($user->id);
    }

    public function updateUserPermissions(User $user, array $permissions): void
    {
        $this->users()->updateExistingPivot($user->id, [
            'can_send' => $permissions['can_send'] ?? true,
            'can_receive' => $permissions['can_receive'] ?? true,
            'permissions' => $permissions['permissions'] ?? null,
        ]);
    }

    public function getSetupInstructions(): array
    {
        $instructions = [
            'android' => $this->getAndroidInstructions(),
            'ios' => $this->getIOSInstructions(),
            'windows_outlook' => $this->getWindowsOutlookInstructions(),
        ];

        return $instructions;
    }

    private function getAndroidInstructions(): array
    {
        return [
            'title' => 'Android Email Setup',
            'steps' => [
                'Open the Gmail app or your default email app',
                'Tap the menu (three lines) and select "Settings"',
                'Tap "Add account" or the "+" button',
                'Select "Other" or "Manual setup"',
                'Enter your email address: ' . $this->email,
                'Enter your password: [Your email password]',
                'Configure the following settings:',
                '  - Incoming server: ' . $this->smtp_host,
                '  - Port: ' . $this->smtp_port,
                '  - Security: ' . strtoupper($this->smtp_encryption),
                '  - Username: ' . $this->smtp_username,
                'Tap "Next" and complete the setup'
            ]
        ];
    }

    private function getIOSInstructions(): array
    {
        return [
            'title' => 'iOS Email Setup',
            'steps' => [
                'Open Settings app on your iPhone/iPad',
                'Scroll down and tap "Mail"',
                'Tap "Accounts" then "Add Account"',
                'Select "Other" at the bottom',
                'Tap "Add Mail Account"',
                'Enter the following information:',
                '  - Name: Your display name',
                '  - Email: ' . $this->email,
                '  - Password: [Your email password]',
                '  - Description: ' . $this->name,
                'Tap "Next" and configure server settings:',
                '  - Incoming Mail Server:',
                '    - Host Name: ' . $this->smtp_host,
                '    - User Name: ' . $this->smtp_username,
                '    - Password: [Your email password]',
                '  - Outgoing Mail Server:',
                '    - Host Name: ' . $this->smtp_host,
                '    - User Name: ' . $this->smtp_username,
                '    - Password: [Your email password]',
                'Tap "Next" and select the items to sync',
                'Tap "Save" to complete setup'
            ]
        ];
    }

    private function getWindowsOutlookInstructions(): array
    {
        return [
            'title' => 'Windows Outlook Setup',
            'steps' => [
                'Open Microsoft Outlook',
                'Go to File > Add Account',
                'Select "Manual setup or additional server types"',
                'Choose "POP or IMAP"',
                'Enter your account information:',
                '  - Your Name: Your display name',
                '  - Email Address: ' . $this->email,
                '  - Account Type: IMAP',
                'Click "More Settings" and go to "Outgoing Server" tab',
                'Check "My outgoing server (SMTP) requires authentication"',
                'Go to "Advanced" tab and configure:',
                '  - Incoming server (IMAP): ' . $this->smtp_host,
                '  - Port: ' . $this->smtp_port,
                '  - Use the following type of encrypted connection: ' . strtoupper($this->smtp_encryption),
                '  - Outgoing server (SMTP): ' . $this->smtp_host,
                '  - Port: ' . $this->smtp_port,
                '  - Use the following type of encrypted connection: ' . strtoupper($this->smtp_encryption),
                'Click "OK" and then "Next"',
                'Enter your password when prompted',
                'Click "Finish" to complete setup'
            ]
        ];
    }

    public function getProviderIcon(): string
    {
        $icons = [
            'gmail' => 'mail',
            'outlook' => 'mail',
            'yahoo' => 'mail',
            'icloud' => 'mail',
            'exchange' => 'mail',
        ];

        return $icons[strtolower($this->provider)] ?? 'mail';
    }

    public function getProviderColor(): string
    {
        $colors = [
            'gmail' => 'bg-red-500',
            'outlook' => 'bg-blue-500',
            'yahoo' => 'bg-purple-500',
            'icloud' => 'bg-gray-500',
            'exchange' => 'bg-green-500',
        ];

        return $colors[strtolower($this->provider)] ?? 'bg-gray-500';
    }
}
