<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class OptimizeResponse
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        // Only optimize HTML responses
        if ($this->shouldOptimize($response)) {
            // Minify HTML
            if (config('performance.view.minify_html')) {
                $this->minifyHtml($response);
            }

            // Add cache headers for static assets
            if ($this->isStaticAsset($request)) {
                $this->addCacheHeaders($response);
            }

            // Add GZIP compression
            if (config('performance.response.gzip') && $this->supportsGzip($request)) {
                $this->compressResponse($response);
            }

            // Add performance headers
            $this->addPerformanceHeaders($response);
        }

        return $response;
    }

    /**
     * Determine if response should be optimized.
     */
    protected function shouldOptimize(Response $response): bool
    {
        return $response->headers->get('Content-Type') === 'text/html; charset=UTF-8'
            || str_contains($response->headers->get('Content-Type', ''), 'text/html');
    }

    /**
     * Minify HTML content.
     */
    protected function minifyHtml(Response $response): void
    {
        $content = $response->getContent();
        
        if ($content) {
            // Remove comments
            $content = preg_replace('/<!--(?!<!)[^\[>].*?-->/s', '', $content);
            
            // Remove whitespace
            $content = preg_replace('/\s+/', ' ', $content);
            
            // Remove whitespace between tags
            $content = preg_replace('/>\s+</', '><', $content);
            
            $response->setContent($content);
        }
    }

    /**
     * Check if request is for static asset.
     */
    protected function isStaticAsset(Request $request): bool
    {
        $path = $request->path();
        $staticExtensions = ['css', 'js', 'jpg', 'jpeg', 'png', 'gif', 'svg', 'woff', 'woff2', 'ttf', 'eot'];
        
        foreach ($staticExtensions as $ext) {
            if (str_ends_with($path, '.' . $ext)) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Add cache headers for static assets.
     */
    protected function addCacheHeaders(Response $response): void
    {
        $response->headers->set('Cache-Control', 'public, max-age=31536000, immutable');
        $response->headers->set('Expires', gmdate('D, d M Y H:i:s', time() + 31536000) . ' GMT');
    }

    /**
     * Check if client supports GZIP.
     */
    protected function supportsGzip(Request $request): bool
    {
        return str_contains($request->header('Accept-Encoding', ''), 'gzip');
    }

    /**
     * Compress response content.
     */
    protected function compressResponse(Response $response): void
    {
        $content = $response->getContent();
        
        if ($content && strlen($content) > 1024) { // Only compress if > 1KB
            $compressed = gzencode($content, config('performance.response.compression_level', 6));
            
            if ($compressed) {
                $response->setContent($compressed);
                $response->headers->set('Content-Encoding', 'gzip');
                $response->headers->set('Content-Length', strlen($compressed));
            }
        }
    }

    /**
     * Add performance-related headers.
     */
    protected function addPerformanceHeaders(Response $response): void
    {
        // DNS prefetch for external domains
        $response->headers->set('X-DNS-Prefetch-Control', 'on');
        
        // Preconnect to important origins
        if (config('performance.assets.cdn_url')) {
            $response->headers->set('Link', '<' . config('performance.assets.cdn_url') . '>; rel=preconnect', false);
        }

        // Remove unnecessary headers
        $response->headers->remove('X-Powered-By');
        
        // Add security headers that also improve performance
        $response->headers->set('X-Content-Type-Options', 'nosniff');
        $response->headers->set('X-Frame-Options', 'SAMEORIGIN');
    }
}

