<?php

namespace App\Http\Middleware;

use Illuminate\Foundation\Inspiring;
use Illuminate\Http\Request;
use Inertia\Middleware;
use Tighten\Ziggy\Ziggy;
use Illuminate\Support\Facades\Log;

class HandleInertiaRequests extends Middleware
{
    /**
     * The root template that's loaded on the first page visit.
     *
     * @see https://inertiajs.com/server-side-setup#root-template
     *
     * @var string
     */
    protected $rootView = 'app';

    /**
     * Determines the current asset version.
     *
     * @see https://inertiajs.com/asset-versioning
     */
    public function version(Request $request): ?string
    {
        return parent::version($request);
    }

    /**
     * Define the props that are shared by default.
     *
     * @see https://inertiajs.com/shared-data
     *
     * @return array<string, mixed>
     */
    public function share(Request $request): array
    {
        [$message, $author] = str(Inspiring::quotes()->random())->explode('-');

        $user = auth()->user(); 
        $userPermissions = $user ? $user->getAllPermissions() : [];
        $userRoles = $user ? $user->roles()->pluck('name')->toArray() : [];

        // Get impersonation data
        $impersonationData = $this->getImpersonationData();

        // Get team registration status for team managers
        $teamRegistrationStatus = $this->getTeamRegistrationStatus($user);

        return [
            ...parent::share($request),
            'name' => config('app.name'),
            'quote' => ['message' => trim($message), 'author' => trim($author)],
            'auth' => [
                'user' => $request->user(),
                'userPermissions' => $userPermissions,
                'userRoles' => $userRoles,
                'impersonation' => $impersonationData,
            ],
            'ziggy' => fn (): array => [
                ...(new Ziggy)->toArray(),
                'location' => $request->url(),
            ],
            'sidebarOpen' => ! $request->hasCookie('sidebar_state') || $request->cookie('sidebar_state') === 'true',
            'session' => [
                'success' => $request->session()->get('success'),
                'error' => $request->session()->get('error'),
                'warning' => $request->session()->get('warning'),
                'info' => $request->session()->get('info'),
                'toast' => $request->session()->get('toast'),
            ],
            'teamRegistrationStatus' => $teamRegistrationStatus,
        ];
    }

    /**
     * Get impersonation data for the frontend
     */
    private function getImpersonationData(): ?array
    {
        $originalUserId = session('impersonate_original_user');
        
        if (!$originalUserId || !auth()->check()) {
            return null;
        }

        $originalUser = \App\Models\User::find($originalUserId);
        $currentUser = auth()->user();

        if (!$originalUser || !$currentUser) {
            return null;
        }

        return [
            'is_impersonating' => true,
            'impersonated_user' => [
                'id' => $currentUser->id,
                'name' => $currentUser->name,
                'email' => $currentUser->email,
            ],
            'original_user' => [
                'id' => $originalUser->id,
                'name' => $originalUser->name,
                'email' => $originalUser->email,
            ],
            'started_at' => session('impersonation_started_at', now()->toISOString()),
        ];
    }

    /**
     * Get team registration status for team managers
     */
    private function getTeamRegistrationStatus($user): ?array
    {
        if (!$user) {
            return null;
        }

        // Check if user is a team manager or captain
        $hasTeamRole = $user->hasAnyRole(['team_manager', 'captain']);
        
        if (!$hasTeamRole) {
            return null;
        }

        // Get teams managed by this user
        $teams = \App\Models\Team::where('captain_id', $user->id)->with(['players', 'tournaments', 'payments'])->get();
        
        if ($teams->isEmpty()) {
            return null;
        }

        $warnings = [];
        
        foreach ($teams as $team) {
            $teamWarnings = [
                'team_id' => $team->id,
                'team_name' => $team->name,
                'has_pending_payment' => false,
                'has_unbilled_registrations' => false,
                'has_incomplete_players' => false,
                'pending_payments' => [],
                'unbilled_tournaments' => [],
                'incomplete_players_count' => 0,
                'total_players' => 0,
            ];

            // Check each tournament registration for payment status
            foreach ($team->tournaments as $tournament) {
                // Check if payment record exists for this tournament
                $payment = $team->payments()
                    ->where('tournament_id', $tournament->id)
                    ->latest()
                    ->first();
                
                if (!$payment) {
                    // No payment record - create one automatically
                    try {
                        $entryFee = $tournament->entry_fee ?? 0;
                        $currency = !empty($tournament->currency) ? $tournament->currency : 'KES';
                        
                        $payment = $team->payments()->create([
                            'tournament_id' => $tournament->id,
                            'team_id' => $team->id,
                            'user_id' => $team->captain_id,
                            'amount' => $entryFee,
                            'status' => 'pending',
                            'reference_number' => uniqid('PAY_'),
                            'payment_method' => 'Mpesa',
                            'payment_amount' => $entryFee,
                            'currency' => $currency,
                            'description' => "Tournament Registration Fee - {$tournament->name}",
                        ]);
                        
                        \Log::info('Auto-created payment record for unbilled tournament', [
                            'team_id' => $team->id,
                            'tournament_id' => $tournament->id,
                            'payment_id' => $payment->id,
                        ]);
                    } catch (\Exception $e) {
                        \Log::error('Failed to auto-create payment record', [
                            'team_id' => $team->id,
                            'tournament_id' => $tournament->id,
                            'error' => $e->getMessage(),
                        ]);
                        continue;
                    }
                }

                // Check payment status
                if ($payment && $payment->status !== 'completed') {
                    $teamWarnings['has_pending_payment'] = true;
                    $amount = $payment->payment_amount ?? $payment->amount ?? $tournament->entry_fee ?? 0;
                    $currency = !empty($payment->currency) ? $payment->currency : (!empty($tournament->currency) ? $tournament->currency : 'KES');
                    
                    $teamWarnings['pending_payments'][] = [
                        'payment_id' => $payment->id,
                        'tournament_id' => $tournament->id,
                        'tournament_name' => $tournament->name,
                        'amount' => (float) $amount,
                        'currency' => $currency,
                        'status' => $payment->status,
                        'reference_number' => $payment->reference_number,
                        'created_at' => $payment->created_at->toDateTimeString(),
                        'payment_method' => $payment->payment_method ?? 'Mpesa',
                    ];
                }
            }

            // Check for incomplete player profiles
            // Use the first tournament to get tournament-specific player requirements
            $primaryTournament = $team->tournaments->first();
            $completenessStatus = $team->getPlayerCompletenessStatus($primaryTournament);
            $teamWarnings['total_players'] = $completenessStatus['total_players'];
            $teamWarnings['incomplete_players_count'] = count($completenessStatus['incomplete_players']);
            $teamWarnings['has_incomplete_players'] = !$completenessStatus['is_team_complete'];
            $teamWarnings['completeness_percentage'] = $completenessStatus['team_completeness_percentage'];
            $teamWarnings['minimum_players_met'] = $completenessStatus['minimum_players_met'];
            $teamWarnings['minimum_players_required'] = $primaryTournament 
                ? ($primaryTournament->min_players_per_team ?? config('tournament.min_players_per_team', 11))
                : config('tournament.min_players_per_team', 11);

            // Only add to warnings if there are actual issues
            if ($teamWarnings['has_pending_payment'] || $teamWarnings['has_incomplete_players']) {
                $warnings[] = $teamWarnings;
            }
        }

        return [
            'has_warnings' => !empty($warnings),
            'teams_with_issues' => $warnings,
        ];
    }
}
