<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Inertia\Inertia;

class CheckPermission
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, string $permission): Response
    {
        $user = $request->user();
        
        if (!$user) {
            if ($request->expectsJson()) {
                return response()->json([
                    'message' => 'Authentication required.',
                    'error' => 'Unauthenticated',
                    'required_permission' => $permission
                ], 401);
            }
            
            return redirect()->route('login');
        }

        if (!$user->hasPermission($permission)) {
            $userRoles = $user->roles()->pluck('name')->toArray();
            $userPermissions = $user->getAllPermissions()->pluck('name')->toArray();
            
            if ($request->expectsJson()) {
                return response()->json([
                    'message' => 'You do not have permission to perform this action.',
                    'error' => 'Insufficient permissions',
                    'required_permission' => $permission,
                    'user_roles' => $userRoles,
                    'user_permissions' => $userPermissions,
                    'suggestions' => $this->getPermissionSuggestions($permission)
                ], 403);
            }

            // For web requests, render a permission error page
            if ($request->inertia()) {
                return Inertia::render('errors/PermissionDenied', [
                    'error' => [
                        'message' => 'You do not have permission to access this resource.',
                        'required_permission' => $permission,
                        'user_roles' => $userRoles,
                        'user_permissions' => $userPermissions,
                        'suggestions' => $this->getPermissionSuggestions($permission)
                    ]
                ])->toResponse($request)->setStatusCode(403);
            }

            abort(403, 'You do not have permission to access this resource.');
        }

        return $next($request);
    }

    /**
     * Get suggestions for the user based on the required permission
     */
    private function getPermissionSuggestions(string $permission): array
    {
        $suggestions = [];
        
        // Parse permission to get resource and action
        $parts = explode('.', $permission);
        $resource = $parts[0] ?? '';
        $action = $parts[1] ?? '';

        // Suggest contacting admin for sensitive permissions
        $sensitivePermissions = ['users', 'roles', 'permissions', 'settings'];
        if (in_array($resource, $sensitivePermissions)) {
            $suggestions[] = 'Contact your system administrator to request access.';
        }

        // Suggest role-based access
        $roleMapping = [
            'tournaments' => ['organizer', 'admin'],
            'players' => ['organizer', 'admin'],
            'teams' => ['organizer', 'admin'],
            'matches' => ['referee', 'organizer', 'admin'],
            'analytics' => ['organizer', 'admin'],
            'media' => ['admin'],
        ];

        if (isset($roleMapping[$resource])) {
            $roles = implode(' or ', $roleMapping[$resource]);
            $suggestions[] = "This action typically requires the {$roles} role.";
        }

        // Suggest viewing alternative
        if ($action !== 'view' && !empty($resource)) {
            $suggestions[] = "You may still be able to view {$resource} if you have view permissions.";
        }

        return $suggestions;
    }
}
