<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\NotificationSetting;
use App\Models\UserNotificationPreference;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class NotificationPreferencesController extends Controller
{
    /**
     * Display user notification preferences
     */
    public function index(): Response
    {
        $user = Auth::user();

        // Get all system-enabled notifications
        $systemNotifications = NotificationSetting::where('enabled', true)
            ->orderBy('category')
            ->orderBy('notification_type')
            ->get();

        // Filter notifications based on user roles and permissions
        $filteredNotifications = $this->filterNotificationsByRole($systemNotifications, $user);

        // Group by category
        $groupedNotifications = $filteredNotifications->groupBy('category');

        // Get user preferences
        $userPreferences = UserNotificationPreference::getUserPreferences($user->id);

        // Get category counts
        $categoryCounts = [];
        foreach ($groupedNotifications as $category => $notifications) {
            $categoryCounts[$category] = $notifications->count();
        }

        return Inertia::render('profile/NotificationPreferences', [
            'systemNotifications' => $groupedNotifications,
            'userPreferences' => $userPreferences,
            'categoryCounts' => $categoryCounts,
        ]);
    }

    /**
     * Filter notifications based on user's roles and permissions
     */
    protected function filterNotificationsByRole($notifications, $user)
    {
        $userRoles = $user->roles->pluck('name')->toArray();

        // Define which categories are relevant for which roles
        $roleCategories = [
            'admin' => ['all'], // Admins can see all
            'super_admin' => ['all'], // Super admins can see all
            'organizer' => ['all'], // Organizers can see all
            'team_manager' => ['auth', 'team', 'player', 'payment', 'tournament', 'match', 'certificate', 'invitation', 'system'],
            'player' => ['auth', 'team', 'player', 'payment', 'tournament', 'match', 'certificate', 'invitation', 'system'],
            'referee' => ['auth', 'match', 'official', 'tournament', 'system'],
            'spectator' => ['auth', 'tournament', 'system'],
        ];

        // Check if user has any role that sees all notifications
        $canSeeAll = false;
        foreach ($userRoles as $role) {
            if (isset($roleCategories[$role]) && in_array('all', $roleCategories[$role])) {
                $canSeeAll = true;
                break;
            }
        }

        if ($canSeeAll) {
            return $notifications;
        }

        // Build list of allowed categories for this user
        $allowedCategories = [];
        foreach ($userRoles as $role) {
            if (isset($roleCategories[$role])) {
                $allowedCategories = array_merge($allowedCategories, $roleCategories[$role]);
            }
        }
        $allowedCategories = array_unique($allowedCategories);

        // Filter notifications
        return $notifications->filter(function ($notification) use ($allowedCategories) {
            return in_array($notification->category, $allowedCategories);
        });
    }

    /**
     * Update user notification preferences
     */
    public function update(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'preferences' => 'required|array',
            'preferences.*.notification_type' => 'required|string',
            'preferences.*.preference_type' => 'required|in:category,individual',
            'preferences.*.enabled' => 'required|boolean',
        ]);

        // Bulk update preferences
        UserNotificationPreference::bulkUpdate($user->id, $validated['preferences']);

        return back()->with('success', 'Notification preferences updated successfully.');
    }

    /**
     * Toggle a category preference
     */
    public function toggleCategory(Request $request, string $category)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'enabled' => 'required|boolean',
        ]);

        UserNotificationPreference::setPreference(
            $user->id,
            $category,
            'category',
            $validated['enabled']
        );

        $status = $validated['enabled'] ? 'enabled' : 'disabled';
        return back()->with('success', "'{$category}' notifications {$status} successfully.");
    }

    /**
     * Toggle an individual notification preference
     */
    public function toggleNotification(Request $request, string $notificationType)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'enabled' => 'required|boolean',
        ]);

        UserNotificationPreference::setPreference(
            $user->id,
            $notificationType,
            'individual',
            $validated['enabled']
        );

        $status = $validated['enabled'] ? 'enabled' : 'disabled';
        return back()->with('success', "Notification {$status} successfully.");
    }

    /**
     * Reset preferences to default (all enabled)
     */
    public function reset()
    {
        $user = Auth::user();

        UserNotificationPreference::where('user_id', $user->id)->delete();
        UserNotificationPreference::clearUserCache($user->id);

        return back()->with('success', 'Notification preferences reset to default.');
    }
}

