<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Tournament;
use App\Http\Requests\StoreTournamentRequest;
use App\Http\Requests\UpdateTournamentRequest;
use Inertia\Inertia;

class TournamentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Tournament::query();

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by type
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // Filter by date range
        if ($request->has('start_date')) {
            $query->where('start_date', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->where('end_date', '<=', $request->end_date);
        }

        // Search by name
        if ($request->has('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Include relationships
        if ($request->has('with')) {
            $relations = explode(',', $request->with);
            $query->with($relations);
        }

        $tournaments = $query->paginate($request->get('per_page', 20));
        return response()->json($tournaments);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('pages.admin.tournaments.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreTournamentRequest $request)
    {
        $tournament = Tournament::create($request->validated());
        return response()->json($tournament->load(['players', 'teams']), 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Tournament $tournament, Request $request)
    {
        $relations = ['players', 'teams', 'brackets', 'gameMatches'];
        
        if ($request->has('with')) {
            $relations = array_merge($relations, explode(',', $request->with));
        }
        
        return response()->json($tournament->load($relations));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateTournamentRequest $request, Tournament $tournament)
    {
        $tournament->update($request->validated());
        return response()->json($tournament->load(['players', 'teams']));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Tournament $tournament)
    {
        $tournament->delete();
        return response()->json(['message' => 'Tournament deleted']);
    }

    // Additional endpoints
    public function participants(Tournament $tournament)
    {
        $participants = [
            'players' => $tournament->players()->with('user')->get(),
            'teams' => $tournament->teams()->with(['captain', 'members'])->get(),
        ];        
        return response()->json($participants);
    }

    public function matches(Tournament $tournament, Request $request)
    {
        $query = $tournament->gameMatches()->with(['player1', 'player2', 'team1', 'team2', 'winner', 'winningTeam']);

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by round
        if ($request->has('round')) {
            $query->where('round', $request->round);
        }

        $matches = $query->paginate($request->get('per_page', 20));
        return response()->json($matches);
    }

    public function brackets(Tournament $tournament)
    {
        $brackets = $tournament->brackets()->with('gameMatches')->get();
        return response()->json($brackets);
    }

    public function statistics(Tournament $tournament)
    {
        $tournament->load(['players', 'teams', 'gameMatches', 'brackets']);
        $stats = [
            'total_players' => $tournament->players()->count(),
            'total_teams' => $tournament->teams()->count(),
            'total_matches' => $tournament->gameMatches()->count(),
            'completed_matches' => $tournament->gameMatches()->where('status', 'completed')->count(),
            'pending_matches' => $tournament->gameMatches()->where('status', 'scheduled')->count(),
            'active_matches' => $tournament->gameMatches()->where('status', 'in_progress')->count(),
            'total_brackets' => $tournament->brackets()->count(),
            'active_brackets' => $tournament->brackets()->where('is_active', true)->count(),
        ];
        
        return response()->json($stats);
    }
}
