<?php

namespace App\Http\Controllers\TeamManager;

use App\Http\Controllers\Controller;
use App\Models\Team;
use App\Models\Player;
use App\Models\Tournament;
use App\Models\GameMatch;
use App\Models\Payment;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;

class TeamManagerController extends Controller
{
    public function __construct()
    {
        // $this->middleware('auth');
        // $this->middleware('role:team_manager');
    }

    /**
     * Upload file to public folder and return URL
     */
    private function uploadFileToPublic($file, $folder = 'uploads')
    {
        // Ensure directory exists
        $publicPath = public_path("images/{$folder}");
        if (!file_exists($publicPath)) {
            mkdir($publicPath, 0755, true);
        }
        
        // Generate unique filename
        $fileName = time() . '_' . $file->getClientOriginalName();
        $filePath = "images/{$folder}/" . $fileName;
        
        // Move file to public directory
        $file->move($publicPath, $fileName);
        
        // Return full URL
        return url($filePath);
    }

    /**
     * Delete old file from public directory
     */
    private function deleteOldFile($fileUrl)
    {
        if (!$fileUrl) {
            return;
        }

        try {
            // Extract the file path from the URL
            $filePath = str_replace(url('/'), '', $fileUrl);
            $fullPath = public_path($filePath);
            
            // Check if file exists and delete it
            if (file_exists($fullPath)) {
                unlink($fullPath);
                \Log::info('Deleted old file: ' . $fullPath);
            }
        } catch (\Exception $e) {
            \Log::error('Failed to delete old file: ' . $e->getMessage());
            // Don't throw exception here as file deletion failure shouldn't stop the update
        }
    }

    /**
     * Get all teams managed by the current user
     */
    protected function getManagedTeams()
    {
        $user = Auth::user();
        return Team::where('captain_id', $user->id)->get();
    }

    /**
     * Get a specific team managed by the current user
     */
    protected function getManagedTeam($teamId = null)
    {
        $user = Auth::user();
        $query = Team::where('captain_id', $user->id);
        
        if ($teamId) {
            return $query->where('id', $teamId)->first();
        }
        
        return $query->first();
    }

    /**
     * Team Profile Management
     */
    public function profile(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.profile')->with('error', 'Selected team not found.');
        }

        $team->load(['players.user', 'tournaments']);

        // Group tournament registrations by category
        $tournamentsByCategory = $team->tournaments->map(function ($tournament) use ($team) {
            $payment = $team->payments()->where('tournament_id', $tournament->id)->latest()->first();
            
            return [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'status' => $tournament->status,
                'start_date' => $tournament->start_date,
                'end_date' => $tournament->end_date,
                'entry_fee' => $tournament->entry_fee,
                'currency' => $tournament->currency,
                'category' => $team->category,
                'category_name' => $team->getCategoryName(),
                'category_fee' => $tournament->getFeeForCategory($team->category),
                'payment_status' => $payment ? $payment->status : 'pending',
                'approval_status' => $tournament->pivot->approval_status ?? 'pending',
                'registered_at' => $tournament->pivot->registered_at ?? $tournament->created_at,
            ];
        })->groupBy('category');

        // Get category configuration
        $categoryConfig = config('tournament.categories');
        $teamCategory = $categoryConfig[$team->category] ?? null;

        return Inertia::render('TeamManager/Profile', [
            'team' => $team,
            'teams' => $teams,
            'user' => Auth::user(),
            'tournamentsByCategory' => $tournamentsByCategory,
            'categoryConfig' => $teamCategory,
        ]);
    }

    /**
     * Update team profile
     */
    public function updateProfile(Request $request)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        // Validate request
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'website' => 'nullable|url|max:255',
            'social_media' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput()
                ->with('error', 'Validation failed. Please check the form.');
        }

        $data = $request->only(['name', 'description', 'website', 'social_media']);
        
        // Handle logo upload
        if ($request->hasFile('logo')) {
            $data['logo'] = $this->uploadFileToPublic($request->file('logo'), 'team-logos');
        }

        // Update team
        $team->update($data);

        return redirect()->back()->with('success', 'Team profile updated successfully.');
    }

    /**
     * Team Players Management
     */
    public function players(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.players')->with('error', 'Selected team not found.');
        }

        $players = $team->players()->with('user')->get();

        return Inertia::render('TeamManager/Players', [
            'team' => $team,
            'teams' => $teams,
            'players' => $players
        ]);
    }

    /**
     * Add player to team
     */
    public function addPlayer(Request $request)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $request->validate([
            'user_id' => 'required|exists:users,id',
            'position' => 'required|string|max:255',
            'jersey_number' => 'nullable|integer|min:1|max:99',
        ]);

        // Check if player is already on the team
        if ($team->players()->where('user_id', $request->user_id)->exists()) {
            return redirect()->back()->with('error', 'Player is already on this team.');
        }

        $team->players()->create([
            'user_id' => $request->user_id,
            'football_position' => $request->position,
            'jersey_number' => $request->jersey_number,
            'status' => 'active'
        ]);

        return redirect()->back()->with('success', 'Player added to team successfully.');
    }

    /**
     * Show create player form
     */
    public function createPlayer(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.players')->with('error', 'Selected team not found.');
        }

        // Get available users who are not already on any team
        $availableUsers = User::whereDoesntHave('players')
            ->select('id', 'name', 'email', 'phone')
            ->get();

        return Inertia::render('TeamManager/CreatePlayer', [
            'team' => $team,
            'teams' => $teams,
            'availableUsers' => $availableUsers
        ]);
    }

    /**
     * Store new player with complete profile
     */
    public function storePlayer(Request $request)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        

        $validator = Validator::make($request->all(), [
            'name' => 'required_if:create_new_user,true|string|max:255',
            'email' => 'required_if:create_new_user,true|email|unique:users,email',
            'password' => 'required_if:create_new_user,true|string|min:8|confirmed',
            'user_id' => 'required_if:create_new_user,false|exists:users,id',
            'display_name' => 'nullable|string|max:255',
            'bio' => 'nullable|string|max:1000',
            'country' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'date_of_birth' => 'nullable|date|before:today',
            'gender' => 'nullable|string|in:male,female,other,prefer_not_to_say',
            'phone' => 'nullable|string|max:20',
            'discord_username' => 'nullable|string|max:255',
            'steam_id' => 'nullable|string|max:255',
            'twitch_username' => 'nullable|string|max:255',
            'youtube_channel' => 'nullable|string|max:255',
            'twitter_handle' => 'nullable|string|max:255',
            'instagram_handle' => 'nullable|string|max:255',
            'website' => 'nullable|url|max:255',
            'experience_level' => 'nullable|string|max:255',
            'primary_game' => 'nullable|string|max:255',
            'football_position' => 'required|string|max:255',
            'jersey_number' => 'nullable|integer|min:1|max:99',
            
            // Avatar field
            'avatar' => 'nullable|file|ima  ge|mimes:jpeg,png,jpg,gif|max:10240',
            
            // Identification fields
            'id_number' => 'nullable|string|max:255',
            'birth_certificate_number' => 'nullable|string|max:255',
            'passport_photo' => 'required|file|image|mimes:jpeg,png,jpg,gif|max:10240', // 10MB max
            'birth_cert_photo' => 'required|file|mimes:jpeg,png,jpg,gif,pdf|max:10240', // 10MB max
            'approval_status' => 'nullable|string|in:pending,under_review,approved,rejected',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $is_team_captain = $request->is_team_captain ? true : false;
        $is_public_profile = $request->is_public_profile ? true : false;
        $allow_messages = $request->allow_messages ? true : false;
        $show_contact_info = $request->show_contact_info ? true : false;

        try {
            DB::beginTransaction();

            // Create user if needed
            if ($request->create_new_user) {
                $user = User::create([
                    'name' => $request->name,
                    'email' => $request->email,
                    'password' => Hash::make($request->password),
                    'phone' => $request->phone,
                ]);
            } else {
                $user = User::findOrFail($request->user_id);
            }

            // Check if player is already on the team
            if ($team->players()->where('user_id', $user->id)->exists()) {
                return redirect()->back()->with('error', 'Player is already on this team.');
            }

            // Create player with complete profile
            $player = $team->players()->create([
                'user_id' => $user->id,
                'display_name' => $request->display_name,
                'bio' => $request->bio,
                'country' => $request->country,
                'city' => $request->city,
                'date_of_birth' => $request->date_of_birth,
                'gender' => $request->gender,
                'phone' => $request->phone,
                'discord_username' => $request->discord_username,
                'steam_id' => $request->steam_id,
                'twitch_username' => $request->twitch_username,
                'youtube_channel' => $request->youtube_channel,
                'twitter_handle' => $request->twitter_handle,
                'instagram_handle' => $request->instagram_handle,
                'website' => $request->website,
                'player_type' => $request->player_type ?? 'amateur',
                'experience_level' => $request->experience_level ?? 'beginner',
                'primary_game' => $request->primary_game,
                'football_position' => $request->football_position,
                'jersey_number' => $request->jersey_number,
                'is_team_captain' => $is_team_captain,
                'is_public_profile' => $is_public_profile,
                'allow_messages' => $allow_messages,
                'show_contact_info' => $show_contact_info,
                'avatar' => $request->file('avatar') ? $this->uploadFileToPublic($request->file('avatar'), 'player-avatars') : null,
                'id_number' => $request->id_number,
                'birth_certificate_number' => $request->birth_certificate_number,
                'passport_photo' => $request->file('passport_photo') ? $this->uploadFileToPublic($request->file('passport_photo'), 'player-documents') : null,
                'birth_cert_photo' => $request->file('birth_cert_photo') ? $this->uploadFileToPublic($request->file('birth_cert_photo'), 'player-documents') : null,
                'approval_status' => $request->approval_status ?? "pending",
                'status' => 'active',
                'verification_status' => 'pending',
            ]);

            DB::commit();

            return response()->json(['message' => 'Player created successfully with complete profile.'], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'Failed to create player: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Show player profile
     */
    public function showPlayerProfile(Request $request, $playerId)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.players')->with('error', 'Selected team not found.');
        }

        $player = $team->players()->with('user')->findOrFail($playerId);

        return Inertia::render('TeamManager/PlayerProfile', [
            'team' => $team,
            'teams' => $teams,
            'player' => $player
        ]);
    }

    /**
     * Update player profile
     */
    public function updatePlayerProfile(Request $request, $playerId)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $player = $team->players()->findOrFail($playerId);

        // Validate all fields including file uploads
        $validator = Validator::make($request->all(), [
            // Required fields
            'display_name' => 'required|string|max:255',
            'football_position' => 'required|string|max:255',
            'jersey_number' => 'nullable|integer|min:1|max:99',
            // At least one identification number is required
            'id_number' => 'required_without:birth_certificate_number|nullable|string|max:255',
            'birth_certificate_number' => 'required_without:id_number|nullable|string|max:255',
            
            // Optional fields
            'bio' => 'nullable|string|max:1000',
            'country' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'date_of_birth' => 'nullable|date|before:today',
            'gender' => 'nullable|string|in:male,female,other,prefer_not_to_say',
            'phone' => 'nullable|string|max:20',
            'discord_username' => 'nullable|string|max:255',
            'steam_id' => 'nullable|string|max:255',
            'twitch_username' => 'nullable|string|max:255',
            'youtube_channel' => 'nullable|string|max:255',
            'twitter_handle' => 'nullable|string|max:255',
            'instagram_handle' => 'nullable|string|max:255',
            'website' => 'nullable|url|max:255',
            'player_type' => 'nullable|string|max:255',
            'experience_level' => 'nullable|string|max:255',
            'primary_game' => 'nullable|string|max:255',
            
            // File uploads - require if not already uploaded
            'avatar' => $player->avatar ? 'nullable|file|image|mimes:jpeg,png,jpg,gif|max:10240' : 'nullable|file|image|mimes:jpeg,png,jpg,gif|max:10240',
            'passport_photo' => $player->passport_photo ? 'nullable|file|image|mimes:jpeg,png,jpg,gif|max:10240' : 'required|file|image|mimes:jpeg,png,jpg,gif|max:10240',
            'birth_cert_photo' => $player->birth_cert_photo ? 'nullable|file|mimes:jpeg,png,jpg,gif,pdf|max:10240' : 'required|file|mimes:jpeg,png,jpg,gif,pdf|max:10240',
            
            // Approval status
            'approval_status' => 'nullable|string|in:pending,under_review,approved,rejected',
        ], [
            // Custom error messages
            'display_name.required' => 'Display name is required.',
            'football_position.required' => 'Football position is required.',
            'jersey_number.min' => 'Jersey number must be at least 1.',
            'jersey_number.max' => 'Jersey number cannot exceed 99.',
            'id_number.required_without' => 'Either ID number or Birth certificate number is required.',
            'birth_certificate_number.required_without' => 'Either ID number or Birth certificate number is required.',
            'passport_photo.required' => 'Passport photo is required.',
            'birth_cert_photo.required' => 'Birth certificate photo is required.',
        ]);

        $is_team_captain = $request->is_team_captain ? true : false;
        $is_public_profile = $request->is_public_profile ? true : false;
        $allow_messages = $request->allow_messages ? true : false;
        $show_contact_info = $request->show_contact_info ? true : false;

        // Prepare update data with fallbacks to existing values
        $updateData = [
            'display_name' => $request->display_name ?: $player->display_name,
            'bio' => $request->bio ?: $player->bio,
            'country' => $request->country ?: $player->country,
            'city' => $request->city ?: $player->city,
            'date_of_birth' => $request->date_of_birth ?: $player->date_of_birth,
            'gender' => $request->gender ?: $player->gender,
            'phone' => $request->phone ?: $player->phone,
            'discord_username' => $request->discord_username,
            'steam_id' => $request->steam_id,
            'twitch_username' => $request->twitch_username,
            'youtube_channel' => $request->youtube_channel,
            'twitter_handle' => $request->twitter_handle,
            'instagram_handle' => $request->instagram_handle,
            'website' => $request->website,
            'player_type' => $request->player_type ?? 'amateur',
            'experience_level' => $request->experience_level ?? 'beginner',
            'primary_game' => $request->primary_game,
            'football_position' => $request->football_position,
            'jersey_number' => $request->jersey_number,
            'is_team_captain' => $is_team_captain,
            'is_public_profile' => $is_public_profile,
            'allow_messages' => $allow_messages,
            'show_contact_info' => $show_contact_info,
            
            // Identification fields
            'id_number' => $request->id_number ?: $player->id_number,
            'birth_certificate_number' => $request->birth_certificate_number ?: $player->birth_certificate_number,
            'approval_status' => $request->approval_status ?: $player->approval_status ?: 'pending',
        ];


        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Handle file uploads - save to public folder and store full URL
        if ($request->hasFile('avatar')) {
            try {
                // Delete old avatar if exists
                if ($player->avatar) {
                    $this->deleteOldFile($player->avatar);
                }
                $updateData['avatar'] = $this->uploadFileToPublic($request->file('avatar'), 'player-avatars');
            } catch (\Exception $e) {
                \Log::error('Avatar upload failed: ' . $e->getMessage());
                return response()->json(['error' => 'Failed to upload avatar: ' . $e->getMessage()], 500);
            }
        }
        
        if ($request->hasFile('passport_photo')) {
            try {
                // Delete old passport photo if exists
                if ($player->passport_photo) {
                    $this->deleteOldFile($player->passport_photo);
                }
                $updateData['passport_photo'] = $this->uploadFileToPublic($request->file('passport_photo'), 'player-documents');
            } catch (\Exception $e) {
                \Log::error('Passport photo upload failed: ' . $e->getMessage());
                return response()->json(['error' => 'Failed to upload passport photo: ' . $e->getMessage()], 500);
            }
        }
        
        if ($request->hasFile('birth_cert_photo')) {
            try {
                // Delete old birth certificate photo if exists
                if ($player->birth_cert_photo) {
                    $this->deleteOldFile($player->birth_cert_photo);
                }
                $updateData['birth_cert_photo'] = $this->uploadFileToPublic($request->file('birth_cert_photo'), 'player-documents');
            } catch (\Exception $e) {
                \Log::error('Birth certificate photo upload failed: ' . $e->getMessage());
                return response()->json(['error' => 'Failed to upload birth certificate photo: ' . $e->getMessage()], 500);
            }
        }

        foreach ($updateData as $key => $value) {
            if ($value === null) {
                $updateData[$key] = $player->$key;
            } 
        }

        try {
        $player->update($updateData);
            return response()->json(['message' => 'Player profile updated successfully.'], 200);
        } catch (\Exception $e) {
            \Log::error('Failed to update player profile: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update player profile: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Download player certificate of participation
     */
    public function downloadCertificate(Request $request, $playerId)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $player = $team->players()->with('user')->findOrFail($playerId);

        // Generate certificate data
        $certificateData = [
            'player_name' => $player->display_name ?? $player->user->name,
            'team_name' => $team->name,
            'digital_id' => $player->digital_id,
            'position' => $player->football_position ?? $player->position,
            'jersey_number' => $player->jersey_number,
            'tournament_matches_played' => $player->tournament_matches_played ?? 0,
            'tournament_win_rate' => $player->tournament_win_rate ?? 0,
            'global_rating' => $player->global_rating ?? 0,
            'achievements' => $player->achievements ?? [],
            'issued_date' => now()->format('F j, Y'),
            'verification_status' => $player->verification_status ?? 'pending',
        ];

        // For now, return a JSON response with certificate data
        // In a real implementation, you would generate a PDF certificate
        return response()->json([
            'message' => 'Certificate generated successfully',
            'certificate_data' => $certificateData,
            'download_url' => route('team.players.certificate.pdf', $playerId)
        ]);
    }

    /**
     * Generate PDF certificate
     */
    public function generateCertificatePdf(Request $request, $playerId)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $player = $team->players()->with('user')->findOrFail($playerId);
        
        $player->load(['user', 'tournament']);
        
        // Generate unique Certificate ID
        $certificateId = 'CERT-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id . time()), 0, 8));
        
        $data = [
            'player_name' => $player->display_name ?? $player->user->name,
            'team_name' => $player->team_name ?? '',
            'position' => $player->position ?? 'Participant',
            'jersey_number' => $player->jersey_number ?? null,
            'tournament_matches_played' => $player->tournament_matches_played ?? 0,
            'tournament_win_rate' => $player->tournament_win_rate ?? 0,
            'global_rating' => $player->global_rating ?? 0,
            'digital_id' => $certificateId,
            'issued_date' => now()->format('F j, Y'),
            'tournament_name' => $player->tournament->name ?? 'The Milimus Tournament Cup',
            'description' => 'has successfully participated in the ' . $player->tournament->name . ' edition tournament held from ' . Carbon::parse($player->tournament->start_date)->format('F j, Y') . ' to ' . Carbon::parse($player->tournament->end_date)->format('F j, Y') . ' and has demonstrated exceptional sportsmanship, skill, and dedication throughout the competition.',
            'qr_code_url' => 'https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=https://example.com/verify/' . $certificateId,
            'verification_status' => $player->verification_status ?? 'verified',
        ];

        try {
            $pdf = Pdf::loadHTML(view('players.certificate', $data)->render());
            // Ensure remote assets (QR Server API) load correctly
            $pdf->setOptions([
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'isPhpEnabled' => true,
                'isFontSubsettingEnabled' => false,
            ]);
            
            // Relax SSL for remote image fetching if needed
            $pdf->getDomPDF()->setHttpContext(stream_context_create([
                'ssl' => [
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                ],
                'http' => [
                    'timeout' => 30,
                ],
            ]));

            $pdf->setPaper('a4', 'landscape');
            
            $filename = 'Certificate_' . ($player->display_name ?? 'Player') . '_' . now()->format('Y-m-d') . '.pdf';
            $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
            
            return $pdf->download($filename);
        } catch (\Exception $e) {
            \Log::error('PDF Generation Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate PDF: ' . $e->getMessage()], 500);
        }
        
    }

    /**
     * Show player certificate view
     */
    public function showPlayerCertificate(Request $request, $playerId)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.players')->with('error', 'Selected team not found.');
        }

        $player = $team->players()->with('user')->findOrFail($playerId);

        return Inertia::render('TeamManager/PlayerCertificate', [
            'team' => $team,
            'teams' => $teams,
            'player' => $player
        ]);
    }

    /**
     * Show player ID cards view
     */
    public function showPlayerIdCards(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.players')->with('error', 'Selected team not found.');
        }

        // Only show players with verification_status as verified or approved
        $players = $team->players()
            ->with('user')
            ->whereIn('verification_status', ['verified', 'approved'])
            ->get();

        return Inertia::render('TeamManager/PlayerIdCards', [
            'team' => $team,
            'teams' => $teams,
            'players' => $players
        ]);
    }

    /**
     * Generate ID cards view
     */
    public function generateIdCards(Request $request)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $request->validate([
            'player_ids' => 'required|array',
            'player_ids.*' => 'exists:players,id',
        ]);

        // Only generate ID cards for verified or approved players
        $players = $team->players()
            ->whereIn('id', $request->player_ids)
            ->whereIn('verification_status', ['verified', 'approved'])
            ->with('user')
            ->get();

        if ($players->isEmpty()) {
            return redirect()->back()->with('error', 'No verified or approved players selected. Only verified/approved players can have ID cards generated.');
        }

        // Prepare player data for the blade view
        $playersData = $players->map(function ($player) use ($team) {
            return [
                'name' => $player->user->name,
                'email' => $player->user->email,
                'phone' => $player->phone ?? $player->user->phone ?? 'Not provided',
                'type' => ucfirst($player->player_type ?? 'Amateur'),
                'tournament' => '1st Edition', // You can make this dynamic based on tournament
                'verification_status' => ucfirst($player->verification_status ?? 'Pending'),
                'registration_status' => 'Registered',
                'id' => str_pad($player->id, 3, '0', STR_PAD_LEFT),
                'tournament_year' => now()->year,
                'generated' => now()->format('F j, Y'),
                'valid_until' => now()->addYears(2)->format('F j, Y'),
                'barcode' => 'ID-' . str_pad($player->id, 3, '0', STR_PAD_LEFT) . '-' . now()->year,
                'digital_id' => $player->digital_id ?? 'PID-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id), 0, 8)),
                'validity_period' => '2 Years',
                'registration' => $player->created_at->format('M j, Y'),
                'photo' => $player->avatar ?? asset('player.jpg'),
            ];  
        });

        return response(view('players.id', [
            'players' => $playersData,
            'team_name' => $team->name,
        ]))
            ->header('Content-Type', 'text/html')
            ->header('X-Inertia', 'false');
    }

    /**
     * Download ID cards
     */
    public function downloadIdCards(Request $request)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $request->validate([
            'player_ids' => 'required|array',
            'player_ids.*' => 'exists:players,id',
        ]);

        // Only download ID cards for verified or approved players
        $players = $team->players()
            ->whereIn('id', $request->player_ids)
            ->whereIn('verification_status', ['verified', 'approved'])
            ->with('user')
            ->get();

        if ($players->isEmpty()) {
            return redirect()->back()->with('error', 'No verified or approved players selected. Only verified/approved players can have ID cards generated.');
        }

        // Prepare player data for the blade view
        $playersData = $players->map(function ($player) use ($team) {
            return [
                'name' => $player->user->name,
                'email' => $player->user->email,
                'phone' => $player->phone ?? $player->user->phone ?? 'Not provided',
                'type' => ucfirst($player->player_type ?? 'Amateur'),
                'tournament' => '1st Edition',
                'verification_status' => ucfirst($player->verification_status ?? 'Pending'),
                'registration_status' => 'Registered',
                'id' => str_pad($player->id, 3, '0', STR_PAD_LEFT),
                'tournament_year' => now()->year,
                'generated' => now()->format('F j, Y'),
                'valid_until' => now()->addYears(2)->format('F j, Y'),
                'barcode' => 'ID-' . str_pad($player->id, 3, '0', STR_PAD_LEFT) . '-' . now()->year,
                'digital_id' => $player->digital_id ?? 'PID-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id), 0, 8)),
                'validity_period' => '2 Years',
                'registration' => $player->created_at->format('M j, Y'),
                'photo' => $player->avatar ?? asset('player.jpg'),
            ];
        });

        $html = view('players.id', [
            'players' => $playersData,
            'team_name' => $team->name,
        ])->render();

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="id-cards-' . str_replace(' ', '-', strtolower($team->name)) . '.html"')
            ->header('X-Inertia', 'false');
    }

    /**
     * Show team certificate view
     */
    public function showTeamCertificate(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.profile')->with('error', 'Selected team not found.');
        }

        $team->load(['players.user', 'tournaments']);
        
        // Get team statistics
        $teamStats = [
            'total_players' => $team->players->count(),
            'verified_players' => $team->players->where('verification_status', 'verified')->count(),
            'active_tournaments' => $team->tournaments->where('status', 'active')->count(),
            'completed_tournaments' => $team->tournaments->where('status', 'completed')->count(),
            'total_matches' => $team->players->sum('tournament_matches_played'),
            'team_win_rate' => $team->players->avg('tournament_win_rate') ?? 0,
        ];

        return Inertia::render('TeamManager/TeamCertificate', [
            'team' => $team,
            'teams' => $teams,
            'user' => Auth::user(),
            'teamStats' => $teamStats,
        ]);
    }

    /**
     * Generate team certificate PDF
     */
    public function generateTeamCertificatePdf(Request $request)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $team->load(['players.user', 'tournaments']);

        // Calculate team statistics
        $totalMatches = $team->players->sum('tournament_matches_played');
        $teamWinRate = $team->players->avg('tournament_win_rate') ?? 0;
        $verifiedPlayers = $team->players->where('verification_status', 'verified')->count();
        $activeTournaments = $team->tournaments->where('status', 'active')->count();
        $completedTournaments = $team->tournaments->where('status', 'completed')->count();

        // Generate team certificate data
        $certificateData = [
            'team_name' => $team->name,
            'team_description' => $team->description ?? 'A competitive football team',
            'total_players' => $team->players->count(),
            'verified_players' => $verifiedPlayers,
            'active_tournaments' => $activeTournaments,
            'completed_tournaments' => $completedTournaments,
            'total_matches' => $totalMatches,
            'team_win_rate' => round($teamWinRate, 2),
            'team_created' => $team->created_at->format('F j, Y'),
            'digital_id' => 'TEAM-' . str_pad($team->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($team->id), 0, 8)),
            'issued_date' => now()->format('F j, Y'),
            'description' => "This certificate is awarded to {$team->name} for participation in the tournament league. The team has demonstrated excellent sportsmanship and competitive spirit.",
            'certificate_type' => 'team_participation',
            'team_stats' => [
                'total_players' => $team->players->count(),
                'verified_players' => $verifiedPlayers,
                'total_matches' => $totalMatches,
                'win_rate' => round($teamWinRate, 2),
                'tournaments_participated' => $team->tournaments->count(),
            ]
        ];

        // Generate PDF using DomPDF
        $pdf = Pdf::loadView('certificates.team', $certificateData);
        $pdf->setPaper('A4', 'landscape');
        
        $filename = 'team-certificate-' . str_replace(' ', '-', strtolower($team->name)) . '.pdf';
        
        return $pdf->download($filename);
    }

    /**
     * Remove player from team
     */
    public function removePlayer(Request $request, $playerId)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $player = $team->players()->findOrFail($playerId);
        $player->delete();

        return redirect()->back()->with('success', 'Player removed from team successfully.');
    }

    /**
     * Team Tournaments
     */
    public function tournaments(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.tournaments')->with('error', 'Selected team not found.');
        }

        $tournaments = $team->tournaments()->with(['matches'])->get()->map(function ($tournament) use ($team) {
            // Get payment status for this tournament
            $payment = $team->payments()->where('tournament_id', $tournament->id)->latest()->first();
            
            $paymentStatus = $payment ? $payment->status !== 'completed' ? 'not_paid' : 'paid' : 'not_paid';
            // Get verification status from pivot table
            $verificationStatus = $tournament->pivot->approval_status ?? 'pending';
            
            return [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'status' => $tournament->pivot->status ?? 'registered',
                'start_date' => $tournament->start_date,
                'end_date' => $tournament->end_date,
                'entry_fee' => $tournament->entry_fee,
                'currency' => $tournament->currency,
                'registered_at' => $tournament->pivot->registered_at ?? $tournament->created_at,
                'payment_status' => $paymentStatus,
                'verification_status' => $verificationStatus,
                'approval_status' => $tournament->pivot->approval_status ?? 'pending',
            ];
        });

        $availableTournaments = Tournament::where('status', 'active')
            ->where('registration_end', '>', now())
            ->whereDoesntHave('teams', function($query) use ($team) {
                $query->where('team_id', $team->id);
            })
            ->get();

        return Inertia::render('TeamManager/Tournaments', [
            'team' => $team,
            'teams' => $teams,
            'tournaments' => $tournaments,
            'availableTournaments' => $availableTournaments
        ]);
    }

    /**
     * Register team for tournament
     */
    public function registerForTournament(Request $request)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $request->validate([
            'tournament_id' => 'required|exists:tournaments,id',
        ]);

        $tournament = Tournament::findOrFail($request->tournament_id);

        // Check if team is already registered
        if ($team->tournaments()->where('tournament_id', $tournament->id)->exists()) {
            return redirect()->back()->with('error', 'Team is already registered for this tournament.');
        }

        // Check registration deadline
        if ($tournament->registration_end < now()) {
            return redirect()->back()->with('error', 'Registration deadline has passed for this tournament.');
        }

        $team->tournaments()->attach($tournament->id, [
            'registered_at' => now(),
            'status' => 'registered'
        ]);

        $payment = $team->payments()->create([
            'tournament_id' => $tournament->id,
            'team_id' => $team->id,
            'user_id' => $team->captain_id,
            'amount' => $tournament->entry_fee,
            'status' => 'pending',
            'reference_number' => uniqid('PAY_'),
            'payment_method' => 'Mpesa',
            'payment_amount' => $tournament->entry_fee,
            'currency' => 'KES',
            'description' => "Tournament Registration Fee - {$tournament->name}",
        ]);

        $payment->save();

        return redirect()->back()->with('success', 'Team registered for tournament successfully.');
    }

    /**
     * Team Matches
     */
    public function matches(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.matches')->with('error', 'Selected team not found.');
        }

        $matches = GameMatch::where(function($query) use ($team) {
            $query->where('team1_id', $team->id)
                  ->orWhere('team2_id', $team->id);
        })->with(['tournament', 'team1', 'team2', 'referee'])
        ->orderBy('scheduled_time', 'desc')
        ->paginate(10);

        return Inertia::render('TeamManager/Matches', [
            'team' => $team,
            'teams' => $teams,
            'matches' => $matches
        ]);
    }

    /**
     * Show single match details
     */
    public function showMatch(Request $request, $matchId)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $match = GameMatch::with(['tournament', 'team1', 'team2', 'referee', 'bracket'])
            ->findOrFail($matchId);

        // Check if this match involves one of the manager's teams
        $managerTeamIds = $teams->pluck('id')->toArray();
        if (!in_array($match->team1_id, $managerTeamIds) && !in_array($match->team2_id, $managerTeamIds)) {
            return redirect()->route('team.matches')->with('error', 'You do not have access to this match.');
        }

        // Determine which team belongs to the manager
        $team = $teams->firstWhere('id', $match->team1_id) ?? $teams->firstWhere('id', $match->team2_id);

        return Inertia::render('TeamManager/MatchDetails', [
            'team' => $team,
            'teams' => $teams,
            'match' => $match
        ]);
    }

    /**
     * Team Payments
     */
    public function payments(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.payments')->with('error', 'Selected team not found.');
        }

        $payments = $team->payments()
            ->with(['tournament', 'team'])
            ->orderBy('created_at', 'desc')
            ->whereIn('status', ['pending', 'completed'])
            ->paginate(10);

        return Inertia::render('TeamManager/Payments', [
            'team' => $team,
            'teams' => $teams,
            'payments' => $payments
        ]);
    }

    /**
     * Process payment using M-Pesa
     */
    public function processPayment(Request $request, $paymentId)
    {
        $team = $this->getManagedTeam();
        
        if (!$team) {
            return response()->json(['error' => 'Team not found'], 404);
        }

        $payment = $team->payments()->findOrFail($paymentId);
        
        if ($payment->status !== 'pending') {
            return response()->json(['error' => 'Payment is not pending'], 400);
        }

        $request->validate([
            'phone_number' => 'required|string|regex:/^07[0-9]{9}$/',
        ], [
            'phone_number.required' => 'Phone number is required for M-Pesa payment',
            'phone_number.regex' => 'Phone number must be in format 07XXXXXXXX',
        ]);

        try {
            $mpesaService = new \App\Services\MpesaService();
            
            // Get the team manager's user for payment processing
            $user = auth()->user();
            
            // Initiate M-Pesa payment
            $result = $mpesaService->createPayment(
                $user,
                $payment->amount,
                $team,
                "Payment for {$payment->tournament->name}",
                "254" . substr($request->phone_number, -9)
            );

            if ($result['success']) {
                return response()->json([
                    'success' => true,
                    'message' => $result['message'],
                    'payment' => $result['payment'],
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'error' => $result['message'],
                ], 400);
            }

        } catch (\Exception $e) {
            \Log::error('Payment processing failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Payment processing failed. Please try again.',
            ], 500);
        }
    }

    /**
     * Team Analytics
     */
    public function analytics(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.analytics')->with('error', 'Selected team not found.');
        }

        // Get team statistics
        $totalMatches = GameMatch::where(function($query) use ($team) {
            $query->where('team1_id', $team->id)
                  ->orWhere('team2_id', $team->id);
        })->count();

        $wins = GameMatch::where(function($query) use ($team) {
            $query->where(function($q) use ($team) {
                $q->where('team1_id', $team->id)->whereColumn('game_scores', '>', 'game_scores');
            })->orWhere(function($q) use ($team) {
                $q->where('team2_id', $team->id)->whereColumn('game_scores', '>', 'game_scores');
            });
        })->count();

        $losses = $totalMatches - $wins;
        $winRate = $totalMatches > 0 ? round(($wins / $totalMatches) * 100, 2) : 0;

        // Get recent performance
        $recentMatches = GameMatch::where(function($query) use ($team) {
            $query->where('team1_id', $team->id)
                  ->orWhere('team2_id', $team->id);
        })->with(['tournament', 'team1', 'team2'])
        ->orderBy('scheduled_time', 'desc')
        ->limit(10)
        ->get();

        // Get tournament performance
        $tournamentStats = $team->tournaments()->withCount(['matches'])->get();

        return Inertia::render('TeamManager/Analytics', [
            'team' => $team,
            'teams' => $teams,
            'stats' => [
                'total_matches' => $totalMatches,
                'wins' => $wins,
                'losses' => $losses,
                'win_rate' => $winRate,
            ],
            'recentMatches' => $recentMatches,
            'tournamentStats' => $tournamentStats
        ]);
    }
}
