<?php

namespace App\Http\Controllers\TeamManager;

use App\Http\Controllers\Controller;
use App\Models\Team;
use App\Models\Formation;
use App\Models\TeamFormation;
use App\Models\Player;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class FormationController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:team_manager');
    }

    /**
     * Get all teams managed by the current user
     */
    protected function getManagedTeams()
    {
        $user = Auth::user();
        return Team::where('captain_id', $user->id)->get();
    }

    /**
     * Get a specific team managed by the current user
     */
    protected function getManagedTeam($teamId = null)
    {
        $user = Auth::user();
        $query = Team::where('captain_id', $user->id);
        
        if ($teamId) {
            return $query->where('id', $teamId)->first();
        }
        
        return $query->first();
    }

    /**
     * Display team formations
     */
    public function index(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.formations')->with('error', 'Selected team not found.');
        }

        $formations = $team->formations()->with('formation')->get();
        $players = $team->players()->with('user')->get();

        // Load formations into the team object
        $team->setRelation('formations', $formations);

        return Inertia::render('TeamManager/Formations/Index', [
            'team' => $team,
            'teams' => $teams,
            'players' => $players
        ]);
    }

    /**
     * Show formation creation form
     */
    public function create(Request $request)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.formations')->with('error', 'Selected team not found.');
        }

        $availableFormations = Formation::active()->popular()->get();
        $players = $team->players()->with('user')->get();

        return Inertia::render('TeamManager/Formations/Create', [
            'team' => $team,
            'teams' => $teams,
            'availableFormations' => $availableFormations,
            'players' => $players
        ]);
    }

    /**
     * Store a new team formation
     */
    public function store(Request $request)
    {
        $team = $this->getManagedTeam($request->team_id);
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $request->validate([
            'team_id' => 'required|exists:teams,id',
            'formation_id' => 'required|exists:formations,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'player_positions' => 'required|array',
            'is_default' => 'boolean',
        ]);

        // If this is being set as default, unset other defaults
        if ($request->is_default) {
            $team->formations()->update(['is_default' => false]);
        }

        $formation = Formation::findOrFail($request->formation_id);

        $teamFormation = TeamFormation::create([
            'team_id' => $team->id,
            'formation_id' => $formation->id,
            'name' => $request->name,
            'description' => $request->description,
            'player_positions' => $request->player_positions,
            'formation_data' => $formation->positions,
            'is_default' => $request->is_default ?? false,
        ]);

        return redirect()->route('team.formations', ['team_id' => $team->id])
            ->with('success', 'Formation created successfully.');
    }

    /**
     * Show formation details
     */
    public function show(Request $request, $id)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.formations')->with('error', 'Selected team not found.');
        }

        $formation = $team->formations()->with('formation')->findOrFail($id);
        $players = $team->players()->with('user')->get();

        return Inertia::render('TeamManager/Formations/Show', [
            'team' => $team,
            'teams' => $teams,
            'formation' => $formation,
            'players' => $players
        ]);
    }

    /**
     * Show formation edit form
     */
    public function edit(Request $request, $id)
    {
        $teams = $this->getManagedTeams();
        
        if ($teams->isEmpty()) {
            return redirect()->route('dashboard')->with('error', 'No teams found for this manager.');
        }

        $selectedTeamId = $request->get('team_id', $teams->first()->id);
        $team = $this->getManagedTeam($selectedTeamId);
        
        if (!$team) {
            return redirect()->route('team.formations')->with('error', 'Selected team not found.');
        }

        $formation = $team->formations()->with('formation')->findOrFail($id);
        $availableFormations = Formation::active()->get();
        $players = $team->players()->with('user')->get();

        return Inertia::render('TeamManager/Formations/Edit', [
            'team' => $team,
            'teams' => $teams,
            'formation' => $formation,
            'availableFormations' => $availableFormations,
            'players' => $players
        ]);
    }

    /**
     * Update team formation
     */
    public function update(Request $request, $id)
    {
        $team = $this->getManagedTeam($request->team_id);
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $formation = $team->formations()->findOrFail($id);

        $request->validate([
            'team_id' => 'required|exists:teams,id',
            'formation_id' => 'required|exists:formations,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'player_positions' => 'required|array',
            'is_default' => 'boolean',
        ]);

        // If this is being set as default, unset other defaults
        if ($request->is_default) {
            $team->formations()->where('id', '!=', $id)->update(['is_default' => false]);
        }

        $baseFormation = Formation::findOrFail($request->formation_id);

        $formation->update([
            'formation_id' => $baseFormation->id,
            'name' => $request->name,
            'description' => $request->description,
            'player_positions' => $request->player_positions,
            'formation_data' => $baseFormation->positions,
            'is_default' => $request->is_default ?? false,
        ]);

        return redirect()->route('team.formations', ['team_id' => $team->id])
            ->with('success', 'Formation updated successfully.');
    }

    /**
     * Delete team formation
     */
    public function destroy(Request $request, $id)
    {
        $team = $this->getManagedTeam($request->team_id);
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        $formation = $team->formations()->findOrFail($id);
        $formation->delete();

        return redirect()->route('team.formations', ['team_id' => $team->id])
            ->with('success', 'Formation deleted successfully.');
    }

    /**
     * Set formation as default
     */
    public function setDefault(Request $request, $id)
    {
        $team = $this->getManagedTeam($request->team_id);
        
        if (!$team) {
            return redirect()->back()->with('error', 'No team found for this manager.');
        }

        // Unset all other defaults
        $team->formations()->update(['is_default' => false]);

        // Set this one as default
        $formation = $team->formations()->findOrFail($id);
        $formation->update(['is_default' => true]);

        return redirect()->back()->with('success', 'Formation set as default successfully.');
    }
}
