<?php

namespace App\Http\Controllers;

use App\Models\Team;
use App\Models\Tournament;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Inertia\Response;

class TeamInvitationController extends Controller
{
    /**
     * Show the team invitation response page.
     */
    public function show(string $invitationId): Response
    {
        // Get the notification from the database
        $notification = Auth::user()->notifications()
            ->where('id', $invitationId)
            ->where('type', 'App\Notifications\User\TeamCaptainInvitationNotification')
            ->firstOrFail();

        $data = $notification->data;
        
        return Inertia::render('TeamInvitation', [
            'invitation' => [
                'id' => $invitationId,
                'team_name' => $data['team_name'],
                'tournament_name' => $data['tournament_name'],
                'registration_details' => $data['registration_details'],
                'invitation_date' => $data['invitation_date'],
                'status' => $data['status'],
            ]
        ]);
    }

    /**
     * Accept the team captain invitation.
     */
    public function accept(Request $request, string $invitationId)
    {
        $request->validate([
            'invitation_id' => 'required|string',
        ]);

        $user = Auth::user();
        
        // Get the notification
        $notification = $user->notifications()
            ->where('id', $invitationId)
            ->where('type', 'App\Notifications\User\TeamCaptainInvitationNotification')
            ->firstOrFail();

        $data = $notification->data;
        
        DB::beginTransaction();
        
        try {
            // Find the team
            $team = Team::where('name', $data['team_name'])->firstOrFail();
            
            // Update team captain
            $team->update(['captain_id' => $user->id]);
            
            // Update notification status
            $notification->update([
                'data' => array_merge($data, ['status' => 'accepted'])
            ]);
            
            // Mark notification as read
            $notification->markAsRead();
            
            DB::commit();
            
            return redirect()->route('dashboard')->with('success', [
                'message' => 'You have successfully accepted the team captain role for ' . $data['team_name']
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Failed to accept invitation. Please try again.']);
        }
    }

    /**
     * Decline the team captain invitation.
     */
    public function decline(Request $request, string $invitationId)
    {
        $request->validate([
            'invitation_id' => 'required|string',
        ]);

        $user = Auth::user();
        
        // Get the notification
        $notification = $user->notifications()
            ->where('id', $invitationId)
            ->where('type', 'App\Notifications\User\TeamCaptainInvitationNotification')
            ->firstOrFail();

        $data = $notification->data;
        
        try {
            // Update notification status
            $notification->update([
                'data' => array_merge($data, ['status' => 'declined'])
            ]);
            
            // Mark notification as read
            $notification->markAsRead();
            
            return redirect()->route('dashboard')->with('success', [
                'message' => 'You have declined the team captain role for ' . $data['team_name']
            ]);
            
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to decline invitation. Please try again.']);
        }
    }
}
