<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Team;
use App\Http\Requests\StoreTeamRequest;
use App\Http\Requests\UpdateTeamRequest;

class TeamController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Team::query();

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Search by name
        if ($request->has('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Filter by captain
        if ($request->has('captain_id')) {
            $query->where('captain_id', $request->captain_id);
        }

        // Include relationships
        if ($request->has('with')) {
            $relations = explode(',', $request->with);
            $query->with($relations);
        }

        $teams = $query->with(['tournament', 'captain', 'members'])->paginate($request->get('per_page', 20));
        return response()->json($teams);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreTeamRequest $request)
    {
        $team = Team::create($request->validated());
        return response()->json($team->load(['tournament', 'captain', 'members']), 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Team $team, Request $request)
    {
        $relations = ['tournament', 'captain', 'members', 'matchesAsTeam1', 'matchesAsTeam2', 'matchesAsWinningTeam'];
        
        if ($request->has('with')) {
            $relations = array_merge($relations, explode(',', $request->with));
        }
        
        return response()->json($team->load($relations));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateTeamRequest $request, Team $team)
    {
        $team->update($request->validated());
        return response()->json($team->load(['tournament', 'captain', 'members']));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Team $team)
    {
        $team->delete();
        return response()->json(['message' => 'Team deleted']);
    }

    // Additional endpoints
    public function members(Team $team)
    {
        $members = $team->members()->with('user')->get();
        return response()->json($members);
    }

    public function addMember(Team $team, Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'role' => 'nullable|string|max:255',
        ]);

        $team->members()->attach($request->user_id, ['role' => $request->role ?? 'member']);
        
        return response()->json(['message' => 'Member added to team']);
    }

    public function removeMember(Team $team, Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
        ]);

        $team->members()->detach($request->user_id);
        
        return response()->json(['message' => 'Member removed from team']);
    }

    public function matches(Team $team, Request $request)
    {
        $query = $team->matchesAsTeam1()
            ->union($team->matchesAsTeam2())
            ->with(['tournament', 'bracket', 'team1', 'team2', 'winningTeam']);

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        $matches = $query->paginate($request->get('per_page', 20));
        return response()->json($matches);
    }

    public function statistics(Team $team)
    {
        $totalMatches = $team->matchesAsTeam1()->count() + $team->matchesAsTeam2()->count();
        $wins = $team->matchesAsWinningTeam()->count();
        $losses = $totalMatches - $wins;
        $winRate = $totalMatches > 0 ? round(($wins / $totalMatches) * 100, 2) : 0;

        $stats = [
            'total_matches' => $totalMatches,
            'wins' => $wins,
            'losses' => $losses,
            'win_rate' => $winRate,
            'member_count' => $team->members()->count(),
            'current_status' => $team->status,
        ];
        
        return response()->json($stats);
    }

    public function roster(Team $team)
    {
        $roster = [
            'captain' => $team->captain,
            'members' => $team->members()->with('user')->get(),
            'total_members' => $team->members()->count() + 1, // +1 for captain
        ];
        
        return response()->json($roster);
    }
}
