<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;

class SystemMaintenanceController extends Controller
{
    /**
     * Show the maintenance dashboard.
     */
    public function index()
    {
        return view('system.maintenance.index', [
            'status' => $this->getSystemStatus(),
        ]);
    }

    /**
     * Test cron job execution.
     */
    public function testCron()
    {
        $startTime = microtime(true);
        $output = [];
        $errors = [];

        try {
            // Test 1: PHP Version
            $output[] = [
                'test' => 'PHP Version',
                'status' => version_compare(PHP_VERSION, '8.0.0', '>=') ? 'pass' : 'fail',
                'details' => PHP_VERSION,
            ];

            // Test 2: Artisan exists
            $artisanPath = base_path('artisan');
            $output[] = [
                'test' => 'Artisan File',
                'status' => file_exists($artisanPath) ? 'pass' : 'fail',
                'details' => $artisanPath,
            ];

            // Test 3: Storage writable
            $storageDir = storage_path('logs');
            $output[] = [
                'test' => 'Storage Writable',
                'status' => is_writable($storageDir) ? 'pass' : 'fail',
                'details' => $storageDir,
            ];

            // Test 4: Queue connection
            $queueConnection = config('queue.default');
            $output[] = [
                'test' => 'Queue Connection',
                'status' => $queueConnection !== 'sync' ? 'pass' : 'warning',
                'details' => $queueConnection,
            ];

            // Test 5: Database connection
            try {
                DB::connection()->getPdo();
                $output[] = [
                    'test' => 'Database Connection',
                    'status' => 'pass',
                    'details' => 'Connected to ' . config('database.default'),
                ];
            } catch (\Exception $e) {
                $output[] = [
                    'test' => 'Database Connection',
                    'status' => 'fail',
                    'details' => $e->getMessage(),
                ];
            }

            // Test 6: Run scheduler
            Artisan::call('schedule:run');
            $schedulerOutput = Artisan::output();
            $output[] = [
                'test' => 'Scheduler Execution',
                'status' => 'pass',
                'details' => trim($schedulerOutput) ?: 'No scheduled commands',
            ];

            // Test 7: Queue verification
            Artisan::call('queue:verify');
            $queueOutput = Artisan::output();
            $output[] = [
                'test' => 'Queue Verification',
                'status' => 'pass',
                'details' => trim($queueOutput),
            ];

            // Test 8: Log file
            $logFile = storage_path('logs/cron-test-' . date('Y-m-d') . '.log');
            $logMessage = '[' . date('Y-m-d H:i:s') . '] Cron test executed via web interface';
            File::append($logFile, $logMessage . PHP_EOL);
            $output[] = [
                'test' => 'Log File Creation',
                'status' => file_exists($logFile) ? 'pass' : 'fail',
                'details' => $logFile,
            ];

        } catch (\Exception $e) {
            $errors[] = $e->getMessage();
        }

        $executionTime = round(microtime(true) - $startTime, 2);

        return view('system.maintenance.test-cron', [
            'output' => $output,
            'errors' => $errors,
            'executionTime' => $executionTime,
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Run system optimization.
     */
    public function optimize()
    {
        $startTime = microtime(true);
        $steps = [];

        try {
            // Step 1: Clear caches
            Artisan::call('cache:clear');
            $steps[] = [
                'step' => 'Clear Cache',
                'status' => 'success',
                'output' => 'Cache cleared successfully',
            ];

            Artisan::call('config:clear');
            $steps[] = [
                'step' => 'Clear Config',
                'status' => 'success',
                'output' => 'Config cleared successfully',
            ];

            Artisan::call('route:clear');
            $steps[] = [
                'step' => 'Clear Routes',
                'status' => 'success',
                'output' => 'Routes cleared successfully',
            ];

            Artisan::call('view:clear');
            $steps[] = [
                'step' => 'Clear Views',
                'status' => 'success',
                'output' => 'Views cleared successfully',
            ];

            // Step 2: Cache optimizations
            Artisan::call('config:cache');
            $steps[] = [
                'step' => 'Cache Config',
                'status' => 'success',
                'output' => 'Configuration cached successfully',
            ];

            Artisan::call('route:cache');
            $steps[] = [
                'step' => 'Cache Routes',
                'status' => 'success',
                'output' => 'Routes cached successfully',
            ];

            Artisan::call('view:cache');
            $steps[] = [
                'step' => 'Cache Views',
                'status' => 'success',
                'output' => 'Views compiled successfully',
            ];

            // Step 3: Optimize autoloader (if composer available)
            if (file_exists(base_path('composer.json')) && function_exists('exec')) {
                exec('composer dump-autoload --optimize 2>&1', $output, $returnCode);
                $steps[] = [
                    'step' => 'Optimize Autoloader',
                    'status' => $returnCode === 0 ? 'success' : 'warning',
                    'output' => $returnCode === 0 ? 'Autoloader optimized' : 'Skipped (composer not available)',
                ];
            } else {
                $steps[] = [
                    'step' => 'Optimize Autoloader',
                    'status' => 'warning',
                    'output' => 'Skipped (exec function disabled on server)',
                ];
            }

            // Step 4: Queue restart
            Artisan::call('queue:restart');
            $steps[] = [
                'step' => 'Restart Queue',
                'status' => 'success',
                'output' => 'Queue workers will restart',
            ];

        } catch (\Exception $e) {
            $steps[] = [
                'step' => 'Error',
                'status' => 'error',
                'output' => $e->getMessage(),
            ];
        }

        $executionTime = round(microtime(true) - $startTime, 2);

        return view('system.maintenance.optimize', [
            'steps' => $steps,
            'executionTime' => $executionTime,
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Get system status.
     */
    protected function getSystemStatus()
    {
        $status = [];

        // PHP version
        $status['php_version'] = PHP_VERSION;
        $status['php_ok'] = version_compare(PHP_VERSION, '8.0.0', '>=');

        // Laravel version
        $status['laravel_version'] = app()->version();

        // Environment
        $status['environment'] = config('app.env');
        $status['debug'] = config('app.debug');

        // Cache
        $status['cache_driver'] = config('cache.default');
        $status['session_driver'] = config('session.driver');
        $status['queue_connection'] = config('queue.default');

        // Config cached
        $status['config_cached'] = file_exists(base_path('bootstrap/cache/config.php'));
        $status['routes_cached'] = file_exists(base_path('bootstrap/cache/routes-v7.php'));
        
        // Views cached
        $viewPath = storage_path('framework/views');
        $status['views_cached'] = is_dir($viewPath) && count(scandir($viewPath)) > 2;

        // OPcache
        $status['opcache_enabled'] = function_exists('opcache_get_status') && opcache_get_status() !== false;

        // Database
        try {
            DB::connection()->getPdo();
            $status['database_connected'] = true;
            $status['database_name'] = config('database.connections.' . config('database.default') . '.database');
        } catch (\Exception $e) {
            $status['database_connected'] = false;
            $status['database_error'] = $e->getMessage();
        }

        // Redis (if configured)
        if (config('cache.default') === 'redis' || config('session.driver') === 'redis') {
            try {
                Cache::driver('redis')->get('test');
                $status['redis_connected'] = true;
            } catch (\Exception $e) {
                $status['redis_connected'] = false;
                $status['redis_error'] = $e->getMessage();
            }
        }

        // Disk space
        $status['disk_free'] = disk_free_space(base_path());
        $status['disk_total'] = disk_total_space(base_path());
        $status['disk_used_percent'] = round((1 - ($status['disk_free'] / $status['disk_total'])) * 100, 2);

        // Memory
        $status['memory_limit'] = ini_get('memory_limit');
        $status['memory_usage'] = memory_get_usage(true);
        $status['memory_peak'] = memory_get_peak_usage(true);

        return $status;
    }

    /**
     * Run full system optimization command.
     */
    public function fullOptimize()
    {
        $startTime = microtime(true);

        try {
            Artisan::call('system:optimize');
            $output = Artisan::output();
            $status = 'success';
        } catch (\Exception $e) {
            $output = $e->getMessage();
            $status = 'error';
        }

        $executionTime = round(microtime(true) - $startTime, 2);

        return view('system.maintenance.full-optimize', [
            'output' => $output,
            'status' => $status,
            'executionTime' => $executionTime,
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ]);
    }

    /**
     * View logs.
     */
    public function viewLogs(Request $request)
    {
        $logType = $request->get('type', 'cron-queue');
        $logFile = storage_path('logs/' . $logType . '.log');

        if (!file_exists($logFile)) {
            $content = "Log file not found: {$logFile}";
            $lines = 0;
        } else {
            $content = File::get($logFile);
            $lines = substr_count($content, "\n");
        }

        return view('system.maintenance.logs', [
            'logType' => $logType,
            'content' => $content,
            'lines' => $lines,
            'file' => $logFile,
            'size' => file_exists($logFile) ? filesize($logFile) : 0,
        ]);
    }

    /**
     * Clear specific cache.
     */
    public function clearCache(Request $request)
    {
        $type = $request->get('type', 'all');
        $result = [];

        switch ($type) {
            case 'config':
                Artisan::call('config:clear');
                $result = ['message' => 'Configuration cache cleared'];
                break;
            case 'route':
                Artisan::call('route:clear');
                $result = ['message' => 'Route cache cleared'];
                break;
            case 'view':
                Artisan::call('view:clear');
                $result = ['message' => 'View cache cleared'];
                break;
            case 'cache':
                Artisan::call('cache:clear');
                $result = ['message' => 'Application cache cleared'];
                break;
            case 'all':
            default:
                Artisan::call('optimize:clear');
                $result = ['message' => 'All caches cleared'];
                break;
        }

        if ($request->wantsJson()) {
            return response()->json($result);
        }

        return redirect()->back()->with('success', $result['message']);
    }

    /**
     * Send test email.
     */
    public function sendTestEmail(Request $request)
    {
        $email = $request->get('email', auth()->user()->email ?? 'admin@example.com');
        $startTime = microtime(true);
        $result = [];

        try {
            // Send test email
            \Illuminate\Support\Facades\Mail::raw(
                'This is a test email from The Milimus Tournament system. If you received this, your email configuration is working correctly!',
                function ($message) use ($email) {
                    $message->to($email)
                        ->subject('Test Email - The Milimus Tournament')
                        ->from(config('mail.from.address'), config('mail.from.name'));
                }
            );

            $executionTime = round(microtime(true) - $startTime, 2);
            
            $result = [
                'status' => 'success',
                'message' => "Test email sent successfully to {$email}",
                'execution_time' => $executionTime,
                'mail_driver' => config('mail.default'),
                'mail_host' => config('mail.mailers.' . config('mail.default') . '.host', 'N/A'),
            ];

        } catch (\Exception $e) {
            $executionTime = round(microtime(true) - $startTime, 2);
            
            $result = [
                'status' => 'error',
                'message' => 'Failed to send test email',
                'error' => $e->getMessage(),
                'execution_time' => $executionTime,
            ];
        }

        if ($request->wantsJson()) {
            return response()->json($result);
        }

        return view('system.maintenance.test-email', $result);
    }
}

