<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class SvgCertificateController extends Controller
{
    /**
     * Generate certificate using SVG-based template
     */
    public function generateCertificate(Request $request)
    {
        // Default values with full customization options
        $data = [
            // Recipient Information
            'recipient' => [
                'name' => $request->get('name', 'Victor Kipkoech'),
                'displayName' => $request->get('displayName', 'Victor Kipkoech'),
                'team' => $request->get('team', 'Lwanda K FC'),
                'position' => $request->get('position', 'Striker'),
                'jerseyNumber' => $request->get('jerseyNumber', '10'),
            ],
            
            // Tournament Information
            'tournament' => [
                'name' => $request->get('tournament_name', 'Milimus Tournament Championship 2024'),
                'startDate' => $request->get('start_date', 'August 3, 2025'),
                'endDate' => $request->get('end_date', 'August 31, 2025'),
                'location' => $request->get('location', 'Nairobi, Kenya'),
                'edition' => $request->get('edition', '1st Edition'),
            ],
            
            // Certificate Details
            'certificateId' => $request->get('certificate_id', 'CERT-' . str_pad(rand(1, 999), 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5(time()), 0, 8))),
            'generatedAt' => $request->get('generated_at', Carbon::now()->format('F j, Y')),
            'validUntil' => $request->get('valid_until', Carbon::now()->addYear()->format('F j, Y')),
            
            // QR Code
            'qrCode' => $request->get('qr_code', 'https://api.qrserver.com/v1/create-qr-code/?size=60x60&data=' . urlencode($request->get('verification_url', 'https://milimus.com/verify/' . $request->get('certificate_id', 'CERT-000075-770DE884')))),
            
            // Styling Options
            'styling' => [
                'header_color' => $request->get('header_color', '#c3292b'),
                'accent_color' => $request->get('accent_color', '#206633'),
                'text_color' => $request->get('text_color', '#4f585a'),
                'recipient_color' => $request->get('recipient_color', '#2b672f'),
                'logo_text' => $request->get('logo_text', 'THE MILIMUS'),
                'logo_subtitle' => $request->get('logo_subtitle', 'TOURNAMENT CUP'),
                'logo_motto' => $request->get('logo_motto', 'IN GOD WE TRUST'),
            ]
        ];
        
        // Generate PDF
        $pdf = Pdf::loadView('certificates.svg-certificate-template', $data);
        $pdf->setOptions([
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true,
            'defaultFont' => 'Montserrat',
        ]);
        $pdf->getDomPDF()->setHttpContext(stream_context_create([
            'ssl' => [
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true,
            ],
        ]));
        $pdf->setPaper('a4', 'landscape');
        
        // Generate filename
        $filename = 'Certificate_' . $data['recipient']['name'] . '_' . Carbon::now()->format('Y-m-d') . '.pdf';
        $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
        
        return $pdf->download($filename);
    }
    
    /**
     * Generate certificate for a specific player (matches the SVG example)
     */
    public function generatePlayerCertificate(Request $request, $playerId = null)
    {
        // Mock player data matching the SVG example
        $player = [
            'name' => 'Victor Kipkoech',
            'team' => 'Lwanda K FC',
            'position' => 'Striker',
            'jerseyNumber' => '10',
        ];
        
        $data = [
            'recipient' => [
                'name' => $player['name'],
                'displayName' => $player['name'],
                'team' => $player['team'],
                'position' => $player['position'],
                'jerseyNumber' => $player['jerseyNumber'],
            ],
            'tournament' => [
                'name' => 'Milimus Tournament Championship 2024',
                'startDate' => 'August 3, 2025',
                'endDate' => 'August 31, 2025',
                'location' => 'Nairobi, Kenya',
                'edition' => '1st Edition',
            ],
            'certificateId' => 'CERT-000075-770DE884',
            'generatedAt' => 'September 20, 2025',
            'validUntil' => 'September 20, 2026',
            'qrCode' => 'https://api.qrserver.com/v1/create-qr-code/?size=60x60&data=https://milimus.com/verify/CERT-000075-770DE884',
            'styling' => [
                'header_color' => '#c3292b',
                'accent_color' => '#206633',
                'text_color' => '#4f585a',
                'recipient_color' => '#2b672f',
                'logo_text' => 'THE MILIMUS',
                'logo_subtitle' => 'TOURNAMENT CUP',
                'logo_motto' => 'IN GOD WE TRUST',
            ]
        ];
        
        $pdf = Pdf::loadView('certificates.svg-certificate-template', $data);
        $pdf->setOptions([
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true,
            'defaultFont' => 'Montserrat',
        ]);
        $pdf->getDomPDF()->setHttpContext(stream_context_create([
            'ssl' => [
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true,
            ],
        ]));
        $pdf->setPaper('a4', 'landscape');
        
        $filename = 'Certificate_' . $player['name'] . '_' . Carbon::now()->format('Y-m-d') . '.pdf';
        $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
        
        return $pdf->download($filename);
    }
    
    /**
     * Preview certificate in browser
     */
    public function previewCertificate(Request $request)
    {
        $data = [
            'recipient' => [
                'name' => $request->get('name', 'Victor Kipkoech'),
                'displayName' => $request->get('displayName', 'Victor Kipkoech'),
                'team' => $request->get('team', 'Lwanda K FC'),
                'position' => $request->get('position', 'Striker'),
                'jerseyNumber' => $request->get('jerseyNumber', '10'),
            ],
            'tournament' => [
                'name' => $request->get('tournament_name', 'Milimus Tournament Championship 2024'),
                'startDate' => $request->get('start_date', 'August 3, 2025'),
                'endDate' => $request->get('end_date', 'August 31, 2025'),
                'location' => $request->get('location', 'Nairobi, Kenya'),
                'edition' => $request->get('edition', '1st Edition'),
            ],
            'certificateId' => $request->get('certificate_id', 'CERT-000075-770DE884'),
            'generatedAt' => $request->get('generated_at', Carbon::now()->format('F j, Y')),
            'validUntil' => $request->get('valid_until', Carbon::now()->addYear()->format('F j, Y')),
            'qrCode' => 'https://api.qrserver.com/v1/create-qr-code/?size=60x60&data=https://milimus.com/verify/CERT-000075-770DE884',
        ];
        
        return view('certificates.svg-certificate-template', $data);
    }
}
