<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;

class QueueTriggerController extends Controller
{
    /**
     * Trigger queue processing via web request
     * This is useful for shared hosting where you can't set up cron jobs
     */
    public function trigger(Request $request)
    {
        // Validate the request has a valid token
        $validToken = config('app.queue_trigger_token', env('QUEUE_TRIGGER_TOKEN', ''));
        
        if (empty($validToken) || $request->input('token') !== $validToken) {
            Log::warning('Queue trigger attempted with invalid token', [
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        try {
            // Process queue with stop-when-empty (safe for shared hosting)
            $output = '';
            Artisan::call('queue:work', [
                '--stop-when-empty' => true,
                '--queue' => 'default',
                '--timeout' => 60,
                '--tries' => 3
            ], $output);

            $message = Artisan::output();
            
            Log::info('Queue triggered via web endpoint', [
                'output' => $message,
                'ip' => $request->ip()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Queue processed successfully',
                'output' => $message,
                'timestamp' => now()->toISOString()
            ]);

        } catch (\Exception $e) {
            Log::error('Queue trigger failed', [
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Queue processing failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get queue status without authentication (for monitoring)
     */
    public function status()
    {
        try {
            $queueConnection = config('queue.default');
            $queueTable = config('queue.connections.database.table', 'jobs');
            $failedTable = config('queue.failed.table', 'failed_jobs');
            
            $pendingJobs = \DB::table($queueTable)->count();
            $failedJobs = \DB::table($failedTable)->count();
            
            return response()->json([
                'success' => true,
                'status' => [
                    'connection' => $queueConnection,
                    'pending_jobs' => $pendingJobs,
                    'failed_jobs' => $failedJobs,
                    'timestamp' => now()->toISOString()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to get queue status: ' . $e->getMessage()
            ], 500);
        }
    }
}
