<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Inertia\Response;

class ProfileController extends Controller
{
    /**
     * Show the profile completion form
     */
    public function showComplete(): Response
    {
        $user = Auth::user();
        
        // Load user with team and player data if applicable
        $user->load(['teams', 'players.team', 'teamMemberships']);
        
        $userTeam = null;
        $userPlayer = null;
        
        if ($user->user_type === 'team') {
            $userTeam = $user->teams->first();
            if ($userTeam) {
                $userPlayer = $user->players()->where('team_id', $userTeam->id)->first();
            }
        }
        
        return Inertia::render('frontend/profile/Complete', [
            'user' => $user,
            'team' => $userTeam,
            'player' => $userPlayer,
            'countries' => $this->getCountries(),
            'positions' => ['Goalkeeper', 'Defender', 'Midfielder', 'Forward'],
            'experienceLevels' => [
                'beginner' => 'Beginner (0-2 years)',
                'intermediate' => 'Intermediate (2-5 years)', 
                'advanced' => 'Advanced (5-10 years)',
                'expert' => 'Expert (10+ years)'
            ]
        ]);
    }
    
    /**
     * Update user profile
     */
    public function updateComplete(Request $request): RedirectResponse
    {
        $user = Auth::user();
        
        $request->validate([
            // Personal Information
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'date_of_birth' => 'required|date|before:today',
            'gender' => 'required|in:male,female,other,prefer_not_to_say',
            'country' => 'required|string|max:255',
            'city' => 'required|string|max:255',
            'bio' => 'nullable|string|max:500',
            
            // Profile Picture (required)
            'avatar' => 'required|file|mimes:jpeg,png,jpg,webp|max:4096',
            
            // National ID Front (required)
            'national_id_front' => 'required|file|mimes:jpeg,png,jpg,webp,pdf|max:4096',
            'national_id_back' => 'required|file|mimes:jpeg,png,jpg,webp,pdf|max:4096',
            
            // Social Links (optional)
            'discord_username' => 'nullable|string|max:255',
            'steam_id' => 'nullable|string|max:255',
            'twitch_username' => 'nullable|string|max:255',
            'youtube_channel' => 'nullable|url|max:255',
            'twitter_handle' => 'nullable|string|max:255',
            'instagram_handle' => 'nullable|string|max:255',
            'website' => 'nullable|url|max:255',
            
            // Player specific fields (if user is team captain/player)
            'experience_level' => 'nullable|in:beginner,intermediate,advanced,expert',
            'primary_game' => 'nullable|string|max:255',
            'game_preferences' => 'nullable|array',
        ]);
        
        try {
            // Handle avatar upload
            $avatarPath = null;
            if ($request->hasFile('avatar')) {
                $avatarPath = $request->file('avatar')->store('avatars', 'public');
                // Delete old avatar if exists
                if ($user->avatar) {
                    Storage::disk('public')->delete($user->avatar);
                }
            }

            // Handle national ID (front) upload
            $nationalIdFrontPath = null;
            if ($request->hasFile('national_id_front')) {
                $nationalIdFrontPath = $request->file('national_id_front')->store('ids', 'public');
                if (method_exists($user, 'getAttribute') && $user->getAttribute('national_id_front')) {
                    Storage::disk('public')->delete($user->national_id_front);
                }
            }

            // Handle national ID (back) upload
            $nationalIdBackPath = null;
            if ($request->hasFile('national_id_back')) {
                $nationalIdBackPath = $request->file('national_id_back')->store('ids', 'public');
                if (method_exists($user, 'getAttribute') && $user->getAttribute('national_id_back')) {
                    Storage::disk('public')->delete($user->national_id_back);
                }
            }
            // Update user data
            $userData = [
                'name' => $request->name,
                'phone' => $request->phone,
                'profile_complete' => true,
            ];
            
            if ($avatarPath) {
                $userData['avatar'] = $avatarPath;
            }
            if ($nationalIdFrontPath) {
                $userData['national_id_front'] = $nationalIdFrontPath;
            }
            if ($nationalIdBackPath) {
                $userData['national_id_back'] = $nationalIdBackPath;
            }
            $user->update($userData);
            
            // If user is team captain/player, update player record
            if ($user->user_type === 'team') {
                $player = $user->players()->first();
                if ($player) {
                    $player->update([
                        'display_name' => $request->name,
                        'bio' => $request->bio,
                        'avatar' => $avatarPath,
                        'national_id_front' => $nationalIdFrontPath,
                        'national_id_back' => $nationalIdBackPath,
                        'country' => $request->country,
                        'city' => $request->city,
                        'date_of_birth' => $request->date_of_birth,
                        'gender' => $request->gender,
                        'phone' => $request->phone,
                        'discord_username' => $request->discord_username,
                        'steam_id' => $request->steam_id,
                        'twitch_username' => $request->twitch_username,
                        'youtube_channel' => $request->youtube_channel,
                        'twitter_handle' => $request->twitter_handle,
                        'instagram_handle' => $request->instagram_handle,
                        'website' => $request->website,
                        'experience_level' => $request->experience_level ?? 'intermediate',
                        'primary_game' => $request->primary_game ?? 'Football',
                        'game_preferences' => $request->game_preferences ?? ['Football'],
                    ]);
                }
            }
            
            return redirect()->route('payment.required')->with('success', 
                'Profile completed successfully! Please proceed with payment to complete registration.'
            );
            
        } catch (\Exception $e) {
            \Log::error('Profile completion failed: ' . $e->getMessage());
            return back()->withErrors(['profile' => 'Profile update failed. Please try again.']);
        }
    }
    
    /**
     * Show payment required page
     */
    public function showPaymentRequired(): Response
    {
        $user = Auth::user();
        
        if ($user->payment_status === 'completed') {
            return redirect()->route('dashboard');
        }
        
        $team = null;
        $category = null;
        $tournament = null;
        
        // Load team for team managers or team users
        if (in_array($user->user_type, ['team_manager', 'team'])) {
            // First try to get team where user is a member
            $team = $user->teams()->with('tournaments')->first();
            
            // If not found, try to get team where user is the captain
            if (!$team) {
                $team = \App\Models\Team::where('captain_id', $user->id)
                    ->with('tournaments')
                    ->first();
            }
            
            \Log::info('Loading team for payment', [
                'user_id' => $user->id,
                'user_type' => $user->user_type,
                'team_found' => $team ? 'yes' : 'no',
                'team_id' => $team?->id,
                'team_name' => $team?->name,
                'team_category' => $team?->category,
                'teams' => $user->teams
            ]);
            
            if ($team) {
                $category = $team->category;
                // Get the tournament the team is registered for
                $tournament = $team->tournaments()->latest()->first();
            }
        }
        
        // If no team-specific tournament, get the active tournament
        if (!$tournament) {
            $tournament = \App\Models\Tournament::where('status', 'active')->first();
        }
        
        $existingPayments = $user->payments()
            ->where('status', '!=', 'cancelled')
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Calculate registration fee based on category
        $registrationFee = 0;
        
        if ($tournament) {
            // If team has a category and tournament has category fees, use category-specific fee
            if ($category && $tournament->category_fees && isset($tournament->category_fees[$category])) {
                $registrationFee = $tournament->category_fees[$category];
                \Log::info('Using category-specific fee', [
                    'user_id' => $user->id,
                    'team_id' => $team?->id,
                    'team_name' => $team?->name,
                    'category' => $category,
                    'fee' => $registrationFee,
                    'tournament_id' => $tournament->id,
                    'tournament_name' => $tournament->name,
                    'category_fees' => $tournament->category_fees,
                ]);
            } else {
                // Otherwise use tournament's general entry fee
                $registrationFee = $tournament->entry_fee ?? 0;
                \Log::info('Using tournament entry fee', [
                    'user_id' => $user->id,
                    'team_id' => $team?->id,
                    'team_name' => $team?->name,
                    'category' => $category,
                    'fee' => $registrationFee,
                    'tournament_id' => $tournament->id,
                    'tournament_name' => $tournament->name,
                    'tournament_entry_fee' => $tournament->entry_fee,
                    'tournament_has_category_fees' => !empty($tournament->category_fees),
                ]);
            }
        } else {
            \Log::warning('No tournament found for payment', [
                'user_id' => $user->id,
                'team_id' => $team?->id,
            ]);
        }
        
        return Inertia::render('frontend/payment/Required', [
            'user' => $user,
            'team' => $team ? [
                'id' => $team->id,
                'name' => $team->name,
                'category' => $category,
            ] : null,
            'registration_fee' => $registrationFee,
            'existing_payments' => $existingPayments,
            'payment_methods' => ['mpesa'],
            'tournament' => $tournament ? [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'entry_fee' => $tournament->entry_fee,
                'currency' => $tournament->currency ?? 'KES',
                'category_fees' => $tournament->category_fees ?? null,
            ] : null
        ]);
    }
    
    /**
     * Get countries list
     */
    private function getCountries(): array
    {
        return [
            'Kenya' => 'Kenya',
            'Uganda' => 'Uganda', 
            'Tanzania' => 'Tanzania',
            'Rwanda' => 'Rwanda',
            'Burundi' => 'Burundi',
            'South Sudan' => 'South Sudan',
            'Ethiopia' => 'Ethiopia',
            'Somalia' => 'Somalia',
            // Add more countries as needed
        ];
    }
}
