<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Player;
use App\Http\Requests\StorePlayerRequest;
use App\Http\Requests\UpdatePlayerRequest;

class PlayerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Player::query();

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Search by display name
        if ($request->has('search')) {
            $query->where('display_name', 'like', '%' . $request->search . '%');
        }

        // Filter by user
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Include relationships
        if ($request->has('with')) {
            $relations = explode(',', $request->with);
            $query->with($relations);
        }

        $players = $query->with(['user', 'tournament'])->paginate($request->get('per_page', 20));
        return response()->json($players);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StorePlayerRequest $request)
    {
        $player = Player::create($request->validated());
        return response()->json($player->load(['user', 'tournament']), 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Player $player, Request $request)
    {
        $relations = ['user', 'tournament', 'matchesAsPlayer1', 'matchesAsPlayer2', 'matchesAsWinner'];
        
        if ($request->has('with')) {
            $relations = array_merge($relations, explode(',', $request->with));
        }
        
        return response()->json($player->load($relations));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdatePlayerRequest $request, Player $player)
    {
        $player->update($request->validated());
        return response()->json($player->load(['user', 'tournament']));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Player $player)
    {
        $player->delete();
        return response()->json(['message' => 'Player deleted']);
    }

    // Additional endpoints
    public function matches(Player $player, Request $request)
    {
        $query = $player->matchesAsPlayer1()
            ->union($player->matchesAsPlayer2())
            ->with(['tournament', 'bracket', 'player1', 'player2', 'winner']);

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        $matches = $query->paginate($request->get('per_page', 20));
        return response()->json($matches);
    }

    public function statistics(Player $player)
    {
        $totalMatches = $player->matchesAsPlayer1()->count() + $player->matchesAsPlayer2()->count();
        $wins = $player->matchesAsWinner()->count();
        $losses = $totalMatches - $wins;
        $winRate = $totalMatches > 0 ? round(($wins / $totalMatches) * 100, 2) : 0;

        $stats = [
            'total_matches' => $totalMatches,
            'wins' => $wins,
            'losses' => $losses,
            'win_rate' => $winRate,
            'tournaments_played' => $player->tournament()->count(),
            'current_status' => $player->status,
        ];
        
        return response()->json($stats);
    }

    public function tournamentHistory(Player $player)
    {
        $history = $player->tournament()
            ->with(['gameMatches' => function($query) use ($player) {
                $query->where(function($q) use ($player) {
                    $q->where('player1_id', $player->id)
                      ->orWhere('player2_id', $player->id);
                });
            }])
            ->get();
        
        return response()->json($history);
    }
}
