<?php

namespace App\Http\Controllers\Player;

use App\Http\Controllers\Controller;
use App\Models\Player;
use App\Models\Tournament;
use App\Models\GameMatch;
use App\Models\Team;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class PlayerPortalController extends Controller
{
    /**
     * Show player's tournaments
     */
    public function tournaments(Request $request)
    {
        $user = Auth::user();
        $playerProfiles = Player::where('user_id', $user->id)
            ->with(['tournament', 'team'])
            ->get();

        $tournaments = $playerProfiles->map(function ($player) {
            return [
                'id' => $player->tournament->id,
                'name' => $player->tournament->name,
                'description' => $player->tournament->description,
                'status' => $player->tournament->status,
                'start_date' => $player->tournament->start_date,
                'end_date' => $player->tournament->end_date,
                'player_status' => $player->status,
                'team' => $player->team ? [
                    'id' => $player->team->id,
                    'name' => $player->team->name,
                    'logo' => $player->team->logo,
                ] : null,
                'registration_date' => $player->registered_at,
                'is_team_captain' => $player->is_team_captain,
            ];
        });

        return Inertia::render('Player/Tournaments', [
            'tournaments' => $tournaments,
            'user' => $user
        ]);
    }

    /**
     * Show player's team information
     */
    public function team(Request $request)
    {
        $user = Auth::user();
        $playerProfile = Player::where('user_id', $user->id)
            ->with(['team.teamPlayers.user', 'tournament'])
            ->first();

        if (!$playerProfile || !$playerProfile->team) {
            return Inertia::render('Player/Team', [
                'team' => null,
                'message' => 'You are not currently assigned to a team.',
                'user' => $user
            ]);
        }

        $team = [
            'id' => $playerProfile->team->id,
            'name' => $playerProfile->team->name,
            'description' => $playerProfile->team->description,
            'logo' => $playerProfile->team->logo,
            'status' => $playerProfile->team->status,
            'tournament' => [
                'id' => $playerProfile->tournament->id,
                'name' => $playerProfile->tournament->name,
            ],
            'members' => $playerProfile->team->teamPlayers->map(function ($player) {
                return [
                    'id' => $player->id,
                    'display_name' => $player->display_name,
                    'user_name' => $player->user->name,
                    'avatar' => $player->avatar,
                    'jersey_number' => $player->jersey_number,
                    'football_position' => $player->football_position,
                    'is_team_captain' => $player->is_team_captain,
                    'status' => $player->status,
                ];
            }),
            'is_captain' => $playerProfile->is_team_captain,
        ];

        return Inertia::render('Player/Team', [
            'team' => $team,
            'user' => $user
        ]);
    }

    /**
     * Show player's matches
     */
    public function matches(Request $request)
    {
        $user = Auth::user();
        $playerProfiles = Player::where('user_id', $user->id)->get();
        $playerIds = $playerProfiles->pluck('id');

        $matches = GameMatch::where(function ($query) use ($playerIds) {
                $query->whereIn('player1_id', $playerIds)
                      ->orWhereIn('player2_id', $playerIds);
            })
            ->orWhereHas('team1.teamPlayers', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->orWhereHas('team2.teamPlayers', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->with([
                'tournament',
                'bracket',
                'player1.user',
                'player2.user',
                'team1',
                'team2',
                'winner.user',
                'winningTeam'
            ])
            ->orderBy('scheduled_time', 'desc')
            ->get();

        $matchesData = $matches->map(function ($match) use ($user) {
            return [
                'id' => $match->id,
                'tournament' => [
                    'id' => $match->tournament->id,
                    'name' => $match->tournament->name,
                ],
                'bracket' => $match->bracket ? [
                    'id' => $match->bracket->id,
                    'name' => $match->bracket->name,
                ] : null,
                'round' => $match->round,
                'match_number' => $match->match_number,
                'player1' => $match->player1 ? [
                    'id' => $match->player1->id,
                    'display_name' => $match->player1->display_name,
                    'user_name' => $match->player1->user->name,
                ] : null,
                'player2' => $match->player2 ? [
                    'id' => $match->player2->id,
                    'display_name' => $match->player2->display_name,
                    'user_name' => $match->player2->user->name,
                ] : null,
                'team1' => $match->team1 ? [
                    'id' => $match->team1->id,
                    'name' => $match->team1->name,
                    'logo' => $match->team1->logo,
                ] : null,
                'team2' => $match->team2 ? [
                    'id' => $match->team2->id,
                    'name' => $match->team2->name,
                    'logo' => $match->team2->logo,
                ] : null,
                'player1_score' => $match->player1_score,
                'player2_score' => $match->player2_score,
                'winner' => $match->winner ? [
                    'id' => $match->winner->id,
                    'display_name' => $match->winner->display_name,
                    'user_name' => $match->winner->user->name,
                ] : null,
                'winning_team' => $match->winningTeam ? [
                    'id' => $match->winningTeam->id,
                    'name' => $match->winningTeam->name,
                ] : null,
                'status' => $match->status,
                'scheduled_time' => $match->scheduled_time,
                'started_at' => $match->started_at,
                'completed_at' => $match->completed_at,
            ];
        });

        return Inertia::render('Player/Matches', [
            'matches' => $matchesData,
            'user' => $user
        ]);
    }

    /**
     * Show player's profile
     */
    public function profile(Request $request)
    {
        $user = Auth::user();
        $playerProfiles = Player::where('user_id', $user->id)
            ->with(['tournament', 'team'])
            ->get();

        $profileData = $playerProfiles->map(function ($player) {
            return [
                'id' => $player->id,
                'tournament' => [
                    'id' => $player->tournament->id,
                    'name' => $player->tournament->name,
                ],
                'team' => $player->team ? [
                    'id' => $player->team->id,
                    'name' => $player->team->name,
                ] : null,
                'display_name' => $player->display_name,
                'bio' => $player->bio,
                'avatar' => $player->avatar,
                'country' => $player->country,
                'city' => $player->city,
                'date_of_birth' => $player->date_of_birth,
                'gender' => $player->gender,
                'phone' => $player->phone,
                'discord_username' => $player->discord_username,
                'steam_id' => $player->steam_id,
                'twitch_username' => $player->twitch_username,
                'youtube_channel' => $player->youtube_channel,
                'twitter_handle' => $player->twitter_handle,
                'instagram_handle' => $player->instagram_handle,
                'website' => $player->website,
                'player_type' => $player->player_type,
                'experience_level' => $player->experience_level,
                'primary_game' => $player->primary_game,
                'football_position' => $player->football_position,
                'jersey_number' => $player->jersey_number,
                'is_team_captain' => $player->is_team_captain,
                'global_rating' => $player->global_rating,
                'global_rank' => $player->global_rank,
                'tournament_rating' => $player->tournament_rating,
                'tournament_rank' => $player->tournament_rank,
                'status' => $player->status,
                'verification_status' => $player->verification_status,
                'registered_at' => $player->registered_at,
                'tournament_wins' => $player->tournament_wins,
                'tournament_losses' => $player->tournament_losses,
                'tournament_draws' => $player->tournament_draws,
                'tournament_win_rate' => $player->tournament_win_rate,
                'tournament_matches_played' => $player->tournament_matches_played,
                'tournament_matches_won' => $player->tournament_matches_won,
                'total_earnings' => $player->total_earnings,
                'tournaments_won' => $player->tournaments_won,
                'achievements' => $player->achievements,
                'performance_metrics' => $player->performance_metrics,
            ];
        });

        return Inertia::render('Player/Profile', [
            'profiles' => $profileData,
            'user' => $user
        ]);
    }

    /**
     * Update player profile
     */
    public function updateProfile(Request $request)
    {
        $request->validate([
            'player_id' => 'required|exists:players,id',
            'display_name' => 'sometimes|string|max:255',
            'bio' => 'sometimes|string|max:1000',
            'avatar' => 'sometimes|string|max:255',
            'country' => 'sometimes|string|max:100',
            'city' => 'sometimes|string|max:100',
            'date_of_birth' => 'sometimes|date',
            'gender' => 'sometimes|in:male,female,other,prefer_not_to_say',
            'phone' => 'sometimes|string|max:20',
            'discord_username' => 'sometimes|string|max:100',
            'steam_id' => 'sometimes|string|max:100',
            'twitch_username' => 'sometimes|string|max:100',
            'youtube_channel' => 'sometimes|string|max:255',
            'twitter_handle' => 'sometimes|string|max:100',
            'instagram_handle' => 'sometimes|string|max:100',
            'website' => 'sometimes|url|max:255',
        ]);

        $user = Auth::user();
        $player = Player::where('id', $request->player_id)
            ->where('user_id', $user->id)
            ->firstOrFail();

        $player->update($request->only([
            'display_name', 'bio', 'avatar', 'country', 'city', 'date_of_birth',
            'gender', 'phone', 'discord_username', 'steam_id', 'twitch_username',
            'youtube_channel', 'twitter_handle', 'instagram_handle', 'website'
        ]));

        return back()->with('success', 'Profile updated successfully.');
    }

    /**
     * Show player's performance analytics
     */
    public function performance(Request $request)
    {
        $user = Auth::user();
        $playerProfiles = Player::where('user_id', $user->id)
            ->with(['tournament', 'team'])
            ->get();

        $playerIds = $playerProfiles->pluck('id');

        // Get all matches for the player
        $matches = GameMatch::where(function ($query) use ($playerIds) {
                $query->whereIn('player1_id', $playerIds)
                      ->orWhereIn('player2_id', $playerIds);
            })
            ->orWhereHas('team1.teamPlayers', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->orWhereHas('team2.teamPlayers', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->with([
                'tournament',
                'player1.user',
                'player2.user',
                'team1',
                'team2',
                'winner.user',
                'winningTeam'
            ])
            ->orderBy('completed_at', 'desc')
            ->get();

        // Calculate overall performance statistics
        $totalMatches = $matches->where('status', 'completed')->count();
        $wins = 0;
        $losses = 0;
        $draws = 0;

        foreach ($matches->where('status', 'completed') as $match) {
            $isPlayerWinner = false;
            $isPlayerParticipant = false;

            // Check if player won (individual matches)
            if ($match->winner && $playerIds->contains($match->winner->id)) {
                $isPlayerWinner = true;
                $isPlayerParticipant = true;
            }
            // Check if player's team won
            elseif ($match->winningTeam) {
                $playerTeams = $playerProfiles->pluck('team_id')->filter();
                if ($playerTeams->contains($match->winningTeam->id)) {
                    $isPlayerWinner = true;
                    $isPlayerParticipant = true;
                }
            }
            // Check if player participated but didn't win
            elseif ($playerIds->contains($match->player1_id) || $playerIds->contains($match->player2_id)) {
                $isPlayerParticipant = true;
            }
            // Check team participation
            elseif ($match->team1 || $match->team2) {
                $playerTeams = $playerProfiles->pluck('team_id')->filter();
                if ($playerTeams->contains($match->team1_id) || $playerTeams->contains($match->team2_id)) {
                    $isPlayerParticipant = true;
                }
            }

            if ($isPlayerParticipant) {
                if ($isPlayerWinner) {
                    $wins++;
                } elseif ($match->player1_score === $match->player2_score) {
                    $draws++;
                } else {
                    $losses++;
                }
            }
        }

        $winRate = $totalMatches > 0 ? round(($wins / $totalMatches) * 100, 1) : 0;

        // Recent form (last 5 matches)
        $recentMatches = $matches->where('status', 'completed')->take(5);
        $recentForm = [];
        foreach ($recentMatches as $match) {
            $result = 'L'; // Loss by default
            
            // Check if player/team won
            if ($match->winner && $playerIds->contains($match->winner->id)) {
                $result = 'W';
            } elseif ($match->winningTeam) {
                $playerTeams = $playerProfiles->pluck('team_id')->filter();
                if ($playerTeams->contains($match->winningTeam->id)) {
                    $result = 'W';
                }
            } elseif ($match->player1_score === $match->player2_score) {
                $result = 'D';
            }
            
            $recentForm[] = [
                'result' => $result,
                'opponent' => $this->getOpponentName($match, $playerProfiles),
                'score' => $match->player1_score . '-' . $match->player2_score,
                'date' => $match->completed_at,
                'tournament' => $match->tournament->name,
            ];
        }

        // Performance by tournament
        $tournamentPerformance = [];
        foreach ($playerProfiles as $profile) {
            $tournamentMatches = $matches->where('tournament_id', $profile->tournament_id)
                                        ->where('status', 'completed');
            
            $tWins = 0;
            $tLosses = 0;
            $tDraws = 0;

            foreach ($tournamentMatches as $match) {
                if (($match->winner && $match->winner->id === $profile->id) ||
                    ($match->winningTeam && $match->winningTeam->id === $profile->team_id)) {
                    $tWins++;
                } elseif ($match->player1_score === $match->player2_score) {
                    $tDraws++;
                } else {
                    $tLosses++;
                }
            }

            $tTotal = $tWins + $tLosses + $tDraws;
            $tWinRate = $tTotal > 0 ? round(($tWins / $tTotal) * 100, 1) : 0;

            $tournamentPerformance[] = [
                'tournament' => $profile->tournament->name,
                'wins' => $tWins,
                'losses' => $tLosses,
                'draws' => $tDraws,
                'total' => $tTotal,
                'win_rate' => $tWinRate,
                'rating' => $profile->tournament_rating,
                'rank' => $profile->tournament_rank,
                'status' => $profile->status,
            ];
        }

        // Rating progression (mock data for now - in real app, you'd store historical ratings)
        $ratingProgression = [];
        $currentRating = $playerProfiles->first()?->global_rating ?? 1000;
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subWeeks($i);
            $rating = $currentRating + (rand(-50, 50) * ($i / 6)); // Mock progression
            $ratingProgression[] = [
                'date' => $date->format('M d'),
                'rating' => max(800, min(2000, (int)$rating)),
            ];
        }

        // Performance streaks
        $currentStreak = 0;
        $longestWinStreak = 0;
        $longestLossStreak = 0;
        $tempWinStreak = 0;
        $tempLossStreak = 0;

        foreach ($matches->where('status', 'completed')->reverse() as $match) {
            $isWin = false;
            if ($match->winner && $playerIds->contains($match->winner->id)) {
                $isWin = true;
            } elseif ($match->winningTeam) {
                $playerTeams = $playerProfiles->pluck('team_id')->filter();
                if ($playerTeams->contains($match->winningTeam->id)) {
                    $isWin = true;
                }
            }

            if ($isWin) {
                $tempWinStreak++;
                $tempLossStreak = 0;
                $longestWinStreak = max($longestWinStreak, $tempWinStreak);
            } else {
                $tempLossStreak++;
                $tempWinStreak = 0;
                $longestLossStreak = max($longestLossStreak, $tempLossStreak);
            }
        }

        // Current streak is the last streak
        $lastMatch = $matches->where('status', 'completed')->first();
        if ($lastMatch) {
            $isLastWin = false;
            if ($lastMatch->winner && $playerIds->contains($lastMatch->winner->id)) {
                $isLastWin = true;
            } elseif ($lastMatch->winningTeam) {
                $playerTeams = $playerProfiles->pluck('team_id')->filter();
                if ($playerTeams->contains($lastMatch->winningTeam->id)) {
                    $isLastWin = true;
                }
            }
            $currentStreak = $isLastWin ? $tempWinStreak : -$tempLossStreak;
        }

        return Inertia::render('Player/Performance', [
            'overallStats' => [
                'total_matches' => $totalMatches,
                'wins' => $wins,
                'losses' => $losses,
                'draws' => $draws,
                'win_rate' => $winRate,
                'current_rating' => $playerProfiles->first()?->global_rating ?? 1000,
                'current_rank' => $playerProfiles->first()?->global_rank,
                'current_streak' => $currentStreak,
                'longest_win_streak' => $longestWinStreak,
                'longest_loss_streak' => $longestLossStreak,
            ],
            'recentForm' => $recentForm,
            'tournamentPerformance' => $tournamentPerformance,
            'ratingProgression' => $ratingProgression,
            'recentMatches' => $recentMatches->take(10)->map(function ($match) use ($playerProfiles) {
                return [
                    'id' => $match->id,
                    'opponent' => $this->getOpponentName($match, $playerProfiles),
                    'result' => $this->getMatchResult($match, $playerProfiles),
                    'score' => $match->player1_score . '-' . $match->player2_score,
                    'tournament' => $match->tournament->name,
                    'date' => $match->completed_at,
                ];
            }),
            'user' => $user
        ]);
    }

    /**
     * Helper method to get opponent name from match
     */
    private function getOpponentName($match, $playerProfiles)
    {
        $playerIds = $playerProfiles->pluck('id');
        $playerTeamIds = $playerProfiles->pluck('team_id')->filter();

        // Individual match
        if ($match->player1 && $match->player2) {
            if ($playerIds->contains($match->player1_id)) {
                return $match->player2->display_name ?? 'Unknown';
            } elseif ($playerIds->contains($match->player2_id)) {
                return $match->player1->display_name ?? 'Unknown';
            }
        }

        // Team match
        if ($match->team1 && $match->team2) {
            if ($playerTeamIds->contains($match->team1_id)) {
                return $match->team2->name ?? 'Unknown Team';
            } elseif ($playerTeamIds->contains($match->team2_id)) {
                return $match->team1->name ?? 'Unknown Team';
            }
        }

        return 'Unknown';
    }

    /**
     * Helper method to get match result
     */
    private function getMatchResult($match, $playerProfiles)
    {
        $playerIds = $playerProfiles->pluck('id');
        $playerTeamIds = $playerProfiles->pluck('team_id')->filter();

        // Check if player won
        if ($match->winner && $playerIds->contains($match->winner->id)) {
            return 'W';
        }

        // Check if player's team won
        if ($match->winningTeam && $playerTeamIds->contains($match->winningTeam->id)) {
            return 'W';
        }

        // Check for draw
        if ($match->player1_score === $match->player2_score) {
            return 'D';
        }

        return 'L';
    }
}