<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Team;
use App\Services\MpesaService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;
use Inertia\Response;

class PaymentController extends Controller
{
    protected $mpesaService;
    
    public function __construct(MpesaService $mpesaService)
    {
        $this->mpesaService = $mpesaService;
    }
    
    /**
     * Initiate payment for team registration
     */
    public function initiatePayment(Request $request): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'phone_number' => 'required|string|regex:/^254[0-9]{9}$/',
            'team_id' => 'nullable|exists:teams,id',
            'description' => 'nullable|string|max:255'
        ]);
        
        $user = Auth::user();
        
        $team = $request->team_id ? Team::with('tournaments')->find($request->team_id) : null;
        $amount = $request->amount;
        $phoneNumber = $request->phone_number;
        $description = $request->description ?? 'Tournament Registration Fee';
        
        // Log payment initiation details
        \Log::info('Initiating payment via checkout', [
            'user_id' => $user->id,
            'user_type' => $user->user_type,
            'team_id' => $team?->id,
            'team_name' => $team?->name,
            'team_category' => $team?->category,
            'amount_requested' => $amount,
            'phone_number' => $phoneNumber,
            'description' => $description,
        ]);
        
        $result = $this->mpesaService->createPayment($user, $amount, $team, $description, $phoneNumber);
        
        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => $result['message'],
                'payment_id' => $result['payment']->id,
                'checkout_request_id' => $result['payment']->checkout_request_id
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => $result['message']
        ], 400);
    }
    
    /**
     * Check payment status
     */
    public function checkPaymentStatus(Payment $payment): JsonResponse
    {
        if (!$payment->checkout_request_id) {
            return response()->json([
                'success' => false,
                'message' => 'Payment not initiated'
            ], 400);
        }
        
        // Query M-Pesa for status if payment is still processing
        if ($payment->status === 'processing') {
            $statusResult = $this->mpesaService->queryStkPush($payment->checkout_request_id);
            
            if (isset($statusResult['ResultCode'])) {
                if ($statusResult['ResultCode'] == '0') {
                    $payment->markAsCompleted();
                    $payment->user->update(['payment_status' => 'completed']);
                } elseif ($statusResult['ResultCode'] != '1032') { // 1032 means still processing
                    $payment->markAsFailed($statusResult['ResultDesc'] ?? 'Payment failed');
                }
            }
        }
        
        return response()->json([
            'success' => true,
            'status' => $payment->status,
            'payment' => $payment->fresh()
        ]);
    }

    /**
     * Poll payment status for real-time updates
     * Enhanced to handle timer-triggered API checks
     */
    public function pollPaymentStatus(Request $request): JsonResponse
    {
        $request->validate([
            'checkout_request_id' => 'required|string'
        ]);

        $payment = Payment::where('checkout_request_id', $request->checkout_request_id)->first();

        if (!$payment) {
            return response()->json([
                'success' => false,
                'message' => 'Payment not found'
            ], 404);
        }

        // Check database status first (callback might have updated it)
        if ($payment->status === 'completed') {
            return response()->json([
                'success' => true,
                'status' => 'completed',
                'message' => 'Payment completed successfully!',
                'payment' => $payment->fresh(),
                'should_redirect' => true,
                'redirect_url' => route('dashboard')
            ]);
        }

        if ($payment->status === 'failed') {
            return response()->json([
                'success' => true,
                'status' => 'failed',
                'message' => $payment->failure_reason ?? 'Payment failed',
                'payment' => $payment->fresh(),
                'should_redirect' => false
            ]);
        }

        // Only query M-Pesa API if still processing
        if ($payment->status === 'processing') {
            $timeSinceCreated = now()->diffInSeconds($payment->created_at);
            
            // Query M-Pesa API after 60 seconds to avoid rate limiting
            // This is typically triggered when frontend timer hits 0
            if ($timeSinceCreated > 60) {
                Log::info("Querying M-Pesa API for payment {$payment->id} after {$timeSinceCreated} seconds");
                
                try {
                    $statusResult = $this->mpesaService->queryStkPush($payment->checkout_request_id);
                    
                    if (isset($statusResult['ResultCode'])) {
                        if ($statusResult['ResultCode'] == '0') {
                            // Payment successful
                            $payment->markAsCompleted();
                            $payment->user->update(['payment_status' => 'completed']);
                            
                            Log::info("Payment {$payment->id} completed successfully via API query");
                            
                            return response()->json([
                                'success' => true,
                                'status' => 'completed',
                                'message' => 'Payment completed successfully!',
                                'payment' => $payment->fresh(),
                                'should_redirect' => true,
                                'redirect_url' => route('dashboard')
                            ]);
                        } elseif ($statusResult['ResultCode'] == '4999') {
                            // Still waiting for user to complete payment
                            Log::info("Payment {$payment->id} still waiting for user action (4999)");
                            
                            return response()->json([
                                'success' => true,
                                'status' => 'waiting',
                                'message' => 'Waiting for you to complete payment on your phone',
                                'payment' => $payment,
                                'should_redirect' => false
                            ]);
                        } elseif ($statusResult['ResultCode'] == '1032') {
                            // Still processing
                            Log::info("Payment {$payment->id} still processing (1032)");
                            
                            return response()->json([
                                'success' => true,
                                'status' => 'processing',
                                'message' => 'Payment is being processed. Please wait...',
                                'payment' => $payment,
                                'should_redirect' => false
                            ]);
                        } else {
                            // Payment failed
                            $errorMessage = $statusResult['ResultDesc'] ?? 'Payment failed';
                            $payment->markAsFailed($errorMessage);
                            
                            Log::info("Payment {$payment->id} failed: {$errorMessage}");
                            
                            return response()->json([
                                'success' => true,
                                'status' => 'failed',
                                'message' => $errorMessage,
                                'payment' => $payment->fresh(),
                                'should_redirect' => false
                            ]);
                        }
                    } else {
                        // API response doesn't contain ResultCode, assume still processing
                        Log::warning("Payment {$payment->id} API response missing ResultCode", $statusResult);
                        
                        return response()->json([
                            'success' => true,
                            'status' => 'processing',
                            'message' => 'Payment is being processed. Please wait...',
                            'payment' => $payment,
                            'should_redirect' => false
                        ]);
                    }
                } catch (\Exception $e) {
                    Log::error("Error querying M-Pesa API for payment {$payment->id}: " . $e->getMessage());
                    
                    return response()->json([
                        'success' => false,
                        'status' => 'processing',
                        'message' => 'Unable to check payment status. Please try again.',
                        'error' => 'API query failed'
                    ], 500);
                }
            } else {
                // Not enough time has passed, return current status without API query
                Log::info("Payment {$payment->id} still processing, time since created: {$timeSinceCreated}s");
                
                return response()->json([
                    'success' => true,
                    'status' => 'processing',
                    'message' => 'Payment is being processed. Please wait...',
                    'payment' => $payment,
                    'should_redirect' => false
                ]);
            }
        }

        // Payment is not in processing state
        return response()->json([
            'success' => true,
            'status' => $payment->status,
            'message' => 'Payment status checked',
            'payment' => $payment,
            'should_redirect' => false
        ]);
    }
    
    /**
     * M-Pesa callback endpoint
     */
    public function mpesaCallback(Request $request): JsonResponse
    {
        try {
            $callbackData = $request->all();
            Log::info('M-Pesa Callback Received: ', $callbackData);
            
            $result = $this->mpesaService->processCallback($callbackData);
            
            return response()->json([
                'ResultCode' => $result ? 0 : 1,
                'ResultDesc' => $result ? 'Success' : 'Failed'
            ]);
            
        } catch (\Exception $e) {
            Log::error('M-Pesa Callback Exception: ' . $e->getMessage());
            return response()->json([
                'ResultCode' => 1,
                'ResultDesc' => 'Internal Server Error'
            ]);
        }
    }
    
    /**
     * Show payment history
     */
    public function paymentHistory(): Response
    {
        $user = Auth::user();
        $payments = $user->payments()
            ->with('team')
            ->orderBy('created_at', 'desc')
            ->paginate(10);
            
        return Inertia::render('frontend/payment/History', [
            'payments' => $payments
        ]);
    }
    
    /**
     * Show payment form
     */
    public function showPaymentForm(?Team $team = null): Response
    {
        $user = Auth::user();
        
        // Get the active tournament and its entry fee
        $tournament = \App\Models\Tournament::where('status', 'active')->first();
        $registrationFee = $tournament ? $tournament->entry_fee : config('tournament.registration_fee', 1000);
        
        return Inertia::render('frontend/payment/PaymentForm', [
            'team' => $team,
            'user' => $user,
            'registration_fee' => $registrationFee,
            'tournament' => $tournament ? [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'entry_fee' => $tournament->entry_fee,
                'currency' => $tournament->currency ?? 'KES'
            ] : null
        ]);
    }
    
    /**
     * Retry failed payment
     */
    public function retryPayment(Payment $payment): JsonResponse
    {
        if ($payment->status !== 'failed') {
            return response()->json([
                'success' => false,
                'message' => 'Payment cannot be retried'
            ], 400);
        }
        
        $result = $this->mpesaService->stkPush(
            $payment->user->phone,
            $payment->amount,
            $payment->reference_number,
            $payment->description
        );
        
        if ($result['success']) {
            $payment->update([
                'checkout_request_id' => $result['checkout_request_id'],
                'merchant_request_id' => $result['merchant_request_id'],
                'status' => 'processing',
                'failure_reason' => null
            ]);
            
            return response()->json([
                'success' => true,
                'message' => $result['customer_message']
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => $result['message']
        ], 400);
    }
}
