<?php

namespace App\Http\Controllers;

use App\Models\Player;
use App\Models\TeamOfficial;
use App\Services\OtpService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Inertia\Inertia;
use Inertia\Response;

class OtpController extends Controller
{
    protected $otpService;
    
    public function __construct(OtpService $otpService)
    {
        $this->otpService = $otpService;
    }
    
    /**
     * Show OTP verification form for players
     */
    public function showPlayerVerification(Request $request): Response
    {
        $email = $request->query('email');
        $token = $request->query('token'); // Additional security token if needed
        
        if (!$email) {
            abort(404);
        }
        
        $player = Player::where('email', $email)->first();
        
        if (!$player) {
            abort(404);
        }
        
        return Inertia::render('frontend/otp/PlayerVerification', [
            'player' => $player,
            'team' => $player->team,
            'email' => $email
        ]);
    }
    
    /**
     * Verify player OTP
     */
    public function verifyPlayer(Request $request): JsonResponse
    {
        $request->validate([
            'email' => 'required|email',
            'otp' => 'required|string|size:6'
        ]);
        
        $player = Player::where('email', $request->email)->first();
        
        if (!$player) {
            return response()->json([
                'success' => false,
                'message' => 'Player not found'
            ], 404);
        }
        
        if ($player->verifyOtp($request->otp)) {
            return response()->json([
                'success' => true,
                'message' => 'Email verified successfully! You will receive login credentials soon.',
                'redirect' => route('otp.success', ['type' => 'player'])
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => 'Invalid or expired OTP code'
        ], 400);
    }
    
    /**
     * Show OTP verification form for officials
     */
    public function showOfficialVerification(Request $request): Response
    {
        $email = $request->query('email');
        $token = $request->query('token');
        
        if (!$email) {
            abort(404);
        }
        
        $official = TeamOfficial::where('email', $email)->first();
        
        if (!$official) {
            abort(404);
        }
        
        return Inertia::render('frontend/otp/OfficialVerification', [
            'official' => $official,
            'team' => $official->team,
            'email' => $email
        ]);
    }
    
    /**
     * Verify official OTP
     */
    public function verifyOfficial(Request $request): JsonResponse
    {
        $request->validate([
            'email' => 'required|email',
            'otp' => 'required|string|size:6'
        ]);
        
        $official = TeamOfficial::where('email', $request->email)->first();
        
        if (!$official) {
            return response()->json([
                'success' => false,
                'message' => 'Official not found'
            ], 404);
        }
        
        if ($official->verifyOtp($request->otp)) {
            return response()->json([
                'success' => true,
                'message' => 'Email verified successfully! You will receive login credentials soon.',
                'redirect' => route('otp.success', ['type' => 'official'])
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => 'Invalid or expired OTP code'
        ], 400);
    }
    
    /**
     * Resend OTP to player
     */
    public function resendPlayerOtp(Request $request): JsonResponse
    {
        $request->validate([
            'email' => 'required|email'
        ]);
        
        $player = Player::where('email', $request->email)->first();
        
        if (!$player) {
            return response()->json([
                'success' => false,
                'message' => 'Player not found'
            ], 404);
        }
        
        if ($player->otp_verified) {
            return response()->json([
                'success' => false,
                'message' => 'Email already verified'
            ], 400);
        }
        
        if ($this->otpService->sendPlayerOtp($player)) {
            return response()->json([
                'success' => true,
                'message' => 'OTP resent successfully'
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to send OTP. Please try again.'
        ], 500);
    }
    
    /**
     * Resend OTP to official
     */
    public function resendOfficialOtp(Request $request): JsonResponse
    {
        $request->validate([
            'email' => 'required|email'
        ]);
        
        $official = TeamOfficial::where('email', $request->email)->first();
        
        if (!$official) {
            return response()->json([
                'success' => false,
                'message' => 'Official not found'
            ], 404);
        }
        
        if ($official->otp_verified) {
            return response()->json([
                'success' => false,
                'message' => 'Email already verified'
            ], 400);
        }
        
        if ($this->otpService->sendOfficialOtp($official)) {
            return response()->json([
                'success' => true,
                'message' => 'OTP resent successfully'
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to send OTP. Please try again.'
        ], 500);
    }
    
    /**
     * Show success page after OTP verification
     */
    public function showSuccess(Request $request): Response
    {
        $type = $request->query('type', 'player');
        
        return Inertia::render('frontend/otp/Success', [
            'type' => $type
        ]);
    }
    
    /**
     * Get team verification status (for team captains)
     */
    public function getTeamVerificationStatus(Request $request): JsonResponse
    {
        $user = auth()->user();
        
        if ($user->user_type !== 'team') {
            return response()->json([
                'success' => false,
                'message' => 'Access denied'
            ], 403);
        }
        
        $team = $user->teams->first();
        
        if (!$team) {
            return response()->json([
                'success' => false,
                'message' => 'Team not found'
            ], 404);
        }
        
        $status = $this->otpService->getTeamVerificationStatus($team);
        
        return response()->json([
            'success' => true,
            'verification_status' => $status
        ]);
    }
}
