<?php

namespace App\Http\Controllers;

use App\Models\Media;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\File;
use Intervention\Image\Facades\Image;

class MediaController extends Controller
{
    /**
     * Upload a file
     */
    public function upload(Request $request): JsonResponse
    {
        // Handle FilePond file upload
        $files = [];
        
        // FilePond sends files with different possible keys
        if ($request->hasFile('filepond')) {
            $filepondFiles = $request->allFiles()['filepond'];
            if (is_array($filepondFiles)) {
                $files = $filepondFiles;
            } else {
                $files = [$filepondFiles];
            }
        } elseif ($request->hasFile('file')) {
            $files = [$request->file('file')];
        } elseif ($request->hasFile('files')) {
            $files = $request->allFiles()['files'];
        } else {
            // Check for any uploaded files
            $allFiles = $request->allFiles();
            if (!empty($allFiles)) {
                foreach ($allFiles as $key => $file) {
                    if (is_array($file)) {
                        $files = array_merge($files, $file);
                    } else {
                        $files[] = $file;
                    }
                }
            }
        }
        
        if (empty($files)) {
            return response()->json([
                'success' => false,
                'message' => 'No files provided'
            ], 400);
        }

        $validator = Validator::make($request->all(), [
            'filepond' => [
                'sometimes',
                'file',
                File::types(['jpeg', 'jpg', 'png', 'gif', 'webp', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv', 'mp4', 'avi', 'mov', 'wmv', 'mp3', 'wav', 'ogg'])
                    ->max(50 * 1024), // 50MB max
            ],
            'filepond.*' => [
                'sometimes',
                'file',
                File::types(['jpeg', 'jpg', 'png', 'gif', 'webp', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv', 'mp4', 'avi', 'mov', 'wmv', 'mp3', 'wav', 'ogg'])
                    ->max(50 * 1024), // 50MB max
            ],
            'file' => [
                'sometimes',
                File::types(['jpeg', 'jpg', 'png', 'gif', 'webp', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv', 'mp4', 'avi', 'mov', 'wmv', 'mp3', 'wav', 'ogg'])
                    ->max(50 * 1024), // 50MB max
            ],
            'files.*' => [
                'sometimes',
                File::types(['jpeg', 'jpg', 'png', 'gif', 'webp', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv', 'mp4', 'avi', 'mov', 'wmv', 'mp3', 'wav', 'ogg'])
                    ->max(50 * 1024), // 50MB max
            ],
            'collection' => 'nullable|string|max:255',
            'alt_text' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:1000',
            'crop_data' => 'nullable|json',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $collection = $request->input('collection', 'default');
            $uploadedMedia = [];

            foreach ($files as $file) {
                $options = [
                    'collection' => $collection,
                    'alt_text' => $request->input('alt_text'),
                    'description' => $request->input('description'),
                ];

                // Handle image cropping if crop data is provided
                if ($request->has('crop_data') && $this->isImage($file)) {
                    $cropData = json_decode($request->input('crop_data'), true);
                    $file = $this->cropImage($file, $cropData);
                }

                $media = Media::createFromUpload($file, $options);
                $uploadedMedia[] = $media->load('user');
            }

            return response()->json([
                'success' => true,
                'data' => count($uploadedMedia) === 1 ? $uploadedMedia[0] : $uploadedMedia,
                'message' => count($uploadedMedia) === 1 ? 'File uploaded successfully' : count($uploadedMedia) . ' files uploaded successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get media files with filtering and pagination
     */
    public function index(Request $request): JsonResponse
    {
        $query = Media::with('user');

        // Filter by collection
        if ($request->has('collection')) {
            $query->inCollection($request->input('collection'));
        }

        // Filter by type
        if ($request->has('type')) {
            switch ($request->input('type')) {
                case 'images':
                    $query->images();
                    break;
                case 'videos':
                    $query->videos();
                    break;
                case 'audios':
                    $query->audios();
                    break;
                case 'documents':
                    $query->documents();
                    break;
            }
        }

        // Search by name
        if ($request->has('search')) {
            $search = $request->input('search');
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('alt_text', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Sort
        $sortBy = $request->input('sort_by', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Paginate
        $perPage = $request->input('per_page', 20);
        $media = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $media->items(),
            'pagination' => [
                'current_page' => $media->currentPage(),
                'last_page' => $media->lastPage(),
                'per_page' => $media->perPage(),
                'total' => $media->total(),
            ]
        ]);
    }

    /**
     * Get a specific media file
     */
    public function show(Media $media): JsonResponse
    {
        return response()->json([
            'success' => true,
            'data' => $media->load('user')
        ]);
    }

    /**
     * Update media metadata
     */
    public function update(Request $request, Media $media): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'alt_text' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:1000',
            'collection_name' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $media->update($request->only(['name', 'alt_text', 'description', 'collection_name']));

        return response()->json([
            'success' => true,
            'data' => $media->fresh(),
            'message' => 'Media updated successfully'
        ]);
    }

    /**
     * Delete a media file
     */
    public function destroy(Media $media): JsonResponse
    {
        try {
            $media->delete();

            return response()->json([
                'success' => true,
                'message' => 'Media deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk delete media files
     */
    public function bulkDelete(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'ids' => 'required|array',
            'ids.*' => 'integer|exists:media,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $media = Media::whereIn('id', $request->input('ids'))->get();
            
            foreach ($media as $item) {
                $item->delete();
            }

            return response()->json([
                'success' => true,
                'message' => count($media) . ' files deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Bulk delete failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get media collections
     */
    public function collections(): JsonResponse
    {
        $collections = Media::select('collection_name')
            ->distinct()
            ->whereNotNull('collection_name')
            ->pluck('collection_name')
            ->map(function ($collection) {
                return [
                    'name' => $collection,
                    'count' => Media::inCollection($collection)->count()
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $collections
        ]);
    }

    /**
     * Serve a media file (with optional transformations)
     */
    public function serve(Media $media, Request $request)
    {
        $path = Storage::disk($media->disk)->path($media->path);

        if (!file_exists($path)) {
            abort(404, 'File not found');
        }

        // Handle image transformations
        if ($media->is_image && ($request->has('w') || $request->has('h') || $request->has('q'))) {
            return $this->serveTransformedImage($media, $request);
        }

        return response()->file($path);
    }

    /**
     * Check if file is an image
     */
    private function isImage($file): bool
    {
        return str_starts_with($file->getMimeType(), 'image/');
    }

    /**
     * Crop an image based on crop data
     */
    private function cropImage($file, array $cropData)
    {
        try {
            $image = Image::make($file);
            
            // Extract crop data
            $x = $cropData['x'] ?? 0;
            $y = $cropData['y'] ?? 0;
            $width = $cropData['width'] ?? $image->width();
            $height = $cropData['height'] ?? $image->height();
            
            // Crop the image
            $image->crop($width, $height, $x, $y);
            
            // Create a new temporary file
            $tempPath = tempnam(sys_get_temp_dir(), 'cropped_');
            $image->save($tempPath);
            
            // Create a new UploadedFile instance
            return new \Illuminate\Http\UploadedFile(
                $tempPath,
                $file->getClientOriginalName(),
                $file->getMimeType(),
                null,
                true
            );
        } catch (\Exception $e) {
            // If cropping fails, return original file
            return $file;
        }
    }

    /**
     * Serve transformed image
     */
    private function serveTransformedImage(Media $media, Request $request)
    {
        $path = Storage::disk($media->disk)->path($media->path);
        
        try {
            $image = Image::make($path);
            
            // Resize if width/height specified
            if ($request->has('w') || $request->has('h')) {
                $width = $request->get('w');
                $height = $request->get('h');
                
                if ($width && $height) {
                    $image->resize($width, $height);
                } elseif ($width) {
                    $image->resize($width, null, function ($constraint) {
                        $constraint->aspectRatio();
                    });
                } elseif ($height) {
                    $image->resize(null, $height, function ($constraint) {
                        $constraint->aspectRatio();
                    });
                }
            }
            
            // Set quality if specified
            if ($request->has('q')) {
                $quality = $request->get('q');
                $image->encode('jpg', $quality);
            }
            
            return response($image->encode(), 200, [
                'Content-Type' => $image->mime(),
                'Cache-Control' => 'public, max-age=31536000'
            ]);
        } catch (\Exception $e) {
            // If transformation fails, return original image
            return response()->file($path);
        }
    }
}
