<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Barryvdh\DomPDF\Facade\Pdf;
use Inertia\Inertia;

class HtmlCertificateController extends Controller
{
    /**
     * Preview the HTML certificate with background image
     */
    public function previewCertificate(Request $request)
    {
        $data = $this->prepareCertificateData($request);
        
        return view('certificates.participation-certificate', $data);
    }

    /**
     * Generate PDF certificate with background image
     */
    public function generateCertificate(Request $request)
    {
        $data = $this->prepareCertificateData($request);
        
        $pdf = Pdf::loadView('certificates.participation-certificate', $data);
        $pdf->setPaper('A4', 'landscape');
        
        return $pdf->stream('certificate-' . ($data['certificateId'] ?? 'unknown') . '.pdf');
    }

    /**
     * Generate player participation certificate
     */
    public function generatePlayerCertificate(Request $request)
    {
        $data = $this->prepareCertificateData($request);
        $data['type'] = 'player';
        
        return view('certificates.participation-certificate', $data);
    }

    /**
     * Generate team participation certificate
     */
    public function generateTeamCertificate(Request $request)
    {
        $data = $this->prepareCertificateData($request);
        $data['type'] = 'team';
        
        return view('certificates.participation-certificate', $data);
    }

    /**
     * Generate PDF for player participation certificate
     */
    public function generatePlayerCertificatePdf(Request $request)
    {
        $data = $this->prepareCertificateData($request);
        $data['type'] = 'player';
        
        $pdf = Pdf::loadView('certificates.participation-certificate', $data);
        $pdf->setPaper('A4', 'landscape');
        
        return $pdf->stream('player-certificate-' . ($data['certificateId'] ?? 'unknown') . '.pdf');
    }

    /**
     * Generate PDF for team participation certificate
     */
    public function generateTeamCertificatePdf(Request $request)
    {
        $data = $this->prepareCertificateData($request);
        $data['type'] = 'team';
        
        $pdf = Pdf::loadView('certificates.participation-certificate', $data);
        $pdf->setPaper('A4', 'landscape');
        
        return $pdf->stream('team-certificate-' . ($data['certificateId'] ?? 'unknown') . '.pdf');
    }

    /**
     * Generate SVG replica certificate
     */
    public function generateSvgReplica(Request $request)
    {
        $data = $this->prepareCertificateData($request);
        
        // For SVG replica, we might want to return the SVG content directly
        return view('certificates.html-certificate-template', $data);
    }

    /**
     * Bulk generate certificates
     */
    public function bulkGenerate(Request $request)
    {
        $recipients = $request->input('recipients', []);
        $certificates = [];

        foreach ($recipients as $recipient) {
            $data = $this->prepareCertificateData($request, $recipient);
            $certificates[] = $data;
        }

        return response()->json([
            'success' => true,
            'certificates' => $certificates
        ]);
    }

    /**
     * Prepare certificate data with default values
     */
    private function prepareCertificateData(Request $request, array $overrideRecipient = null)
    {
        $type = $request->input('type', 'player'); // 'player' or 'team'
        $certificateId = $request->input('certificate_id', 'CERT-' . str_pad(rand(1, 999), 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5(time()), 0, 8)));
        
        $recipient = $overrideRecipient ?? [
            'name' => $request->input('recipient_name', $type === 'team' ? 'Team Alpha' : 'Victor Kipkoech'),
            'tournament' => $request->input('tournament_name', 'The Milimus Tournament Cup'),
            'position' => $request->input('position', 'Participant'),
            'date' => $request->input('date', 'August 31, 2025'),
            'team_name' => $request->input('team_name', ''),
            'player_count' => $request->input('player_count', 0),
        ];

        // Adjust description based on type
        $description = $type === 'team' 
            ? "has successfully participated in the 1st Edition tournament held from August 3, 2025 to August 31, 2025 and has demonstrated exceptional teamwork, sportsmanship, and dedication throughout the competition."
            : "has successfully participated in the 1st Edition tournament held from August 3, 2025 to August 31, 2025 and has demonstrated exceptional sportsmanship, skill, and dedication throughout the competition.";

        return [
            'recipient' => $recipient,
            'certificateId' => $certificateId,
            'generatedAt' => $request->input('generated_at', now()->format('F j, Y')),
            'tournamentName' => $request->input('tournament_name', 'The Milimus Tournament Cup'),
            'description' => $request->input('description', $description),
            'qrCodeUrl' => $request->input('qr_code_url', 'https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=https://example.com/verify/' . $certificateId),
            'verificationUrl' => $request->input('verification_url', 'https://example.com/verify/' . $certificateId),
            'type' => $type,
        ];
    }
}
