<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\GameMatch;
use App\Http\Requests\StoreGameMatchRequest;
use App\Http\Requests\UpdateGameMatchRequest;

class GameMatchController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = GameMatch::query();

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        // Filter by bracket
        if ($request->has('bracket_id')) {
            $query->where('bracket_id', $request->bracket_id);
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by round
        if ($request->has('round')) {
            $query->where('round', $request->round);
        }

        // Filter by match number
        if ($request->has('match_number')) {
            $query->where('match_number', $request->match_number);
        }

        // Filter by player
        if ($request->has('player_id')) {
            $query->where(function($q) use ($request) {
                $q->where('player1_id', $request->player_id)
                  ->orWhere('player2_id', $request->player_id);
            });
        }

        // Filter by team
        if ($request->has('team_id')) {
            $query->where(function($q) use ($request) {
                $q->where('team1_id', $request->team_id)
                  ->orWhere('team2_id', $request->team_id);
            });
        }

        // Include relationships
        if ($request->has('with')) {
            $relations = explode(',', $request->with);
            $query->with($relations);
        }

        $matches = $query->with(['tournament', 'bracket', 'player1', 'player2', 'team1', 'team2', 'winner', 'winningTeam'])
            ->paginate($request->get('per_page', 20));
        return response()->json($matches);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreGameMatchRequest $request)
    {
        $match = GameMatch::create($request->validated());
        return response()->json($match->load(['tournament', 'bracket', 'player1', 'player2', 'team1', 'team2']), 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(GameMatch $gameMatch, Request $request)
    {
        $relations = ['tournament', 'bracket', 'player1', 'player2', 'team1', 'team2', 'winner', 'winningTeam'];
        
        if ($request->has('with')) {
            $relations = array_merge($relations, explode(',', $request->with));
        }
        
        return response()->json($gameMatch->load($relations));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateGameMatchRequest $request, GameMatch $gameMatch)
    {
        $gameMatch->update($request->validated());
        return response()->json($gameMatch->load(['tournament', 'bracket', 'player1', 'player2', 'team1', 'team2']));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(GameMatch $gameMatch)
    {
        $gameMatch->delete();
        return response()->json(['message' => 'Game match deleted']);
    }

    // Additional endpoints
    public function startMatch(GameMatch $gameMatch)
    {
        if ($gameMatch->status !== 'scheduled') {
            return response()->json(['error' => 'Match is not scheduled'], 400);
        }

        $gameMatch->update([
            'status' => 'in_progress',
            'started_at' => now(),
        ]);

        return response()->json(['message' => 'Match started', 'match' => $gameMatch->load(['player1', 'player2', 'team1', 'team2'])]);
    }

    public function updateScore(GameMatch $gameMatch, Request $request)
    {
        $request->validate([
            'player1_score' => 'nullable|integer|min:0',
            'player2_score' => 'nullable|integer|min:0',
        ]);

        $gameMatch->update($request->only(['player1_score', 'player2_score']));

        return response()->json(['message' => 'Score updated', 'match' => $gameMatch->load(['player1', 'player2', 'team1', 'team2'])]);
    }

    public function completeMatch(GameMatch $gameMatch, Request $request)
    {
        $request->validate([
            'winner_id' => 'nullable|exists:players,id',
            'winning_team_id' => 'nullable|exists:teams,id',
        ]);

        if ($gameMatch->status !== 'in_progress') {
            return response()->json(['error' => 'Match is not in progress'], 400);
        }

        $gameMatch->update([
            'status' => 'completed',
            'winner_id' => $request->winner_id,
            'winning_team_id' => $request->winning_team_id,
            'completed_at' => now(),
        ]);

        return response()->json(['message' => 'Match completed', 'match' => $gameMatch->load(['winner', 'winningTeam'])]);
    }

    public function upcomingMatches(Request $request)
    {
        $query = GameMatch::where('status', 'scheduled')
            ->where('scheduled_time', '>=', now())
            ->with(['tournament', 'player1', 'player2', 'team1', 'team2']);

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        $matches = $query->orderBy('scheduled_time')->paginate($request->get('per_page', 20));
        return response()->json($matches);
    }

    public function liveMatches(Request $request)
    {
        $query = GameMatch::where('status', 'in_progress')
            ->with(['tournament', 'player1', 'player2', 'team1', 'team2']);

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        $matches = $query->orderBy('started_at')->paginate($request->get('per_page', 20));
        return response()->json($matches);
    }

    public function recentMatches(Request $request)
    {
        $query = GameMatch::where('status', 'completed')
            ->with(['tournament', 'player1', 'player2', 'team1', 'team2', 'winner', 'winningTeam']);

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        $matches = $query->orderBy('completed_at', 'desc')->paginate($request->get('per_page', 20));
        return response()->json($matches);
    }
}
