<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Team;
use App\Models\Tournament;
use App\Models\TeamFormation;
use App\Models\Player;
use App\Models\TeamOfficial;

class TeamController extends Controller
{
    public function index(Request $request)
    {
        $query = Team::with(['tournaments', 'captain', 'manager', 'players'])
            ->whereHas('tournaments', function ($q) {
                $q->where('tournaments.status', 'active');
            });

        // Filter by tournament if specified
        if ($request->has('tournament') && $request->tournament) {
            $query->whereHas('tournaments', function ($q) use ($request) {
                $q->where('tournament_id', $request->tournament);
            });
        }

        // Filter by status
        if ($request->has('status') && $request->status) {
            $query->whereHas('tournaments', function ($q) use ($request) {
                $q->where('tournaments.status', $request->status);
            });
        }

        // Search by team name
        if ($request->has('search') && $request->search) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        $teams = $query->orderBy('name', 'asc')->paginate(12);

        // Get all active tournaments for filter dropdown
        $tournaments = Tournament::where('status', 'active')
            ->where('visibility', 'public')
            ->orderBy('name', 'asc')
            ->get()
            ->map(function ($tournament) {
                return [
                    'id' => $tournament->id,
                    'name' => $tournament->name,
                    'slug' => $tournament->slug,
                ];
            });

        // Transform teams data
        $teamsData = $teams->through(function ($team) {
            $activeTournaments = $team->tournaments->where('pivot.status', 'active');
            $defaultFormation = $team->defaultFormation()->first();
            
            return [
                'id' => $team->id,
                'name' => $team->name,
                'description' => $team->description,
                'logo' => $team->logo,
                'country' => $team->country,
                'county' => $team->county,
                'location' => $team->location,
                'captain' => $team->captain ? [
                    'id' => $team->captain->id,
                    'name' => $team->captain->name,
                ] : null,
                'manager' => $team->manager ? [
                    'id' => $team->manager->id,
                    'name' => $team->manager->name,
                ] : null,
                'player_count' => $team->players->count(),
                'active_tournaments' => $activeTournaments->map(function ($tournament) {
                    return [
                        'id' => $tournament->id,
                        'name' => $tournament->name,
                        'slug' => $tournament->slug,
                    ];
                }),
                'has_formation' => $defaultFormation ? true : false,
                'formation_name' => $defaultFormation ? $defaultFormation->name : null,
                'stats' => $team->stats,
            ];
        });

        return Inertia::render('frontend/Teams', [
            'teams' => $teamsData,
            'tournaments' => $tournaments,
            'filters' => [
                'tournament' => $request->tournament,
                'status' => $request->status,
                'search' => $request->search,
            ],
        ]);
    }

    public function show(Team $team)
    {
        // Load team with all necessary relationships
        $team->load([
            'tournaments' => function ($query) {
                $query->where('tournaments.status', 'active');
            },
            'captain',
            'manager',
            'players.user',
            'officials.user',
            'formations' => function ($query) {
                $query->where('is_default', true);
            },
            'formations.formation'
        ]);

        // Get active tournaments for this team
        $activeTournaments = $team->tournaments->where('pivot.status', 'active');
        
        // Get default formation
        $defaultFormation = $team->formations->first();
        
        // Get team statistics
        $teamStats = [
            'total_players' => $team->players->count(),
            'active_players' => $team->players->where('status', 'active')->count(),
            'total_officials' => $team->officials->count(),
            'tournaments_participated' => $activeTournaments->count(),
            'matches_played' => $team->matchesAsTeam1()->count() + $team->matchesAsTeam2()->count(),
            'matches_won' => $team->matchesAsWinningTeam()->count(),
        ];

        // Transform team data
        $teamData = [
            'id' => $team->id,
            'name' => $team->name,
            'description' => $team->description,
            'logo' => $team->logo,
            'country' => $team->country,
            'county' => $team->county,
            'location' => $team->location,
            'captain' => $team->captain ? [
                'id' => $team->captain->id,
                'name' => $team->captain->name,
                'email' => $team->captain->email,
            ] : null,
            'manager' => $team->manager ? [
                'id' => $team->manager->id,
                'name' => $team->manager->name,
                'email' => $team->manager->email,
            ] : null,
            'players' => $team->players->map(function ($player) {
                return [
                    'id' => $player->id,
                    'display_name' => $player->display_name,
                    'user' => [
                        'name' => $player->user->name,
                    ],
                    'jersey_number' => $player->jersey_number,
                    'football_position' => $player->football_position,
                    'status' => $player->status,
                    'avatar' => $player->avatar,
                ];
            }),
            'officials' => $team->officials->map(function ($official) {
                return [
                    'id' => $official->id,
                    'user' => [
                        'name' => $official->user->name,
                    ],
                    'role' => $official->role,
                    'status' => $official->status,
                ];
            }),
            'active_tournaments' => $activeTournaments->map(function ($tournament) {
                return [
                    'id' => $tournament->id,
                    'name' => $tournament->name,
                    'slug' => $tournament->slug,
                    'start_date' => $tournament->start_date?->format('Y-m-d'),
                    'end_date' => $tournament->end_date?->format('Y-m-d'),
                ];
            }),
            'default_formation' => $defaultFormation ? [
                'id' => $defaultFormation->id,
                'name' => $defaultFormation->name,
                'formation' => [
                    'id' => $defaultFormation->formation->id,
                    'name' => $defaultFormation->formation->name,
                    'display_name' => $defaultFormation->formation->display_name,
                ],
                'player_positions' => $defaultFormation->player_positions,
                'formation_data' => $defaultFormation->formation_data,
            ] : null,
            'stats' => $team->stats,
            'team_stats' => $teamStats,
        ];

        return Inertia::render('frontend/Team', [
            'team' => $teamData,
        ]);
    }
}
