<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Frontend\HeroSlide;
use App\Models\Frontend\NewsArticle;
use App\Models\Frontend\GalleryImage;
use App\Models\Frontend\Sponsor;
use App\Models\Frontend\Testimonial;
use App\Models\Frontend\TournamentHighlight;
use App\Models\Frontend\TournamentAchievement;
use App\Models\Frontend\Fixture;
use App\Models\Frontend\VlogVideo;
use App\Models\Tournament;
use App\Models\Player;
use App\Models\GameMatch;
use App\Models\Team;
use App\Models\Faq;
use App\Models\PrizeAward;

class HomeController extends Controller
{
    /**
     * Sanitize icon to prevent UTF-8 encoding issues
     */
    private function sanitizeIcon($icon)
    {
        if (empty($icon)) {
            return 'Circle';
        }
        
        // Remove any non-ASCII characters (emojis, special characters)
        $sanitized = preg_replace('/[^\x00-\x7F]/', '', $icon);
        
        // If empty after sanitization, return default
        if (empty($sanitized)) {
            return 'Circle';
        }
        
        return $sanitized;
    }

    /**
     * Sanitize text to prevent UTF-8 encoding issues
     */
    private function sanitizeText($text)
    {
        if (empty($text)) {
            return '';
        }
        
        // Convert to UTF-8 and remove any malformed characters
        $sanitized = mb_convert_encoding($text, 'UTF-8', 'UTF-8');
        
        // Remove any remaining non-printable characters
        $sanitized = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $sanitized);
        
        return $sanitized;
    }
    public function index()
    {
        // Fetch dynamic data for homepage
        $heroSlides = HeroSlide::active()->published()->ordered()->get()->map(function ($slide) {
            return [
                'id' => $slide->id,
                'title' => $this->sanitizeText($slide->title),
                'subtitle' => $this->sanitizeText($slide->subtitle),
                'bgImage' => $slide->bg_image,
                'ctaPrimary' => $this->sanitizeText($slide->cta_primary),
                'ctaSecondary' => $this->sanitizeText($slide->cta_secondary),
                'ctaPrimaryUrl' => $slide->cta_primary_url,
                'ctaSecondaryUrl' => $slide->cta_secondary_url,
            ];
        });

        $newsArticles = NewsArticle::published()->featured()->latest()->limit(3)->get()->map(function ($article) {
            return [
                'id' => $article->id,
                'title' => $this->sanitizeText($article->title),
                'excerpt' => $this->sanitizeText($article->excerpt),
                'category' => $this->sanitizeText($article->category),
                'author' => $this->sanitizeText($article->author),
                'date' => $article->date->format('Y-m-d'),
                'readTime' => $article->read_time,
                'image' => $article->image,
                'slug' => $article->slug,
            ];
        });

        $tournamentHighlights = TournamentHighlight::active()->ordered()->get()->map(function ($highlight) {
            return [
                'icon' => $this->sanitizeIcon($highlight->icon),
                'title' => $this->sanitizeText($highlight->title),
                'description' => $this->sanitizeText($highlight->description),
                'value' => $this->sanitizeText($highlight->value),
                'color' => $highlight->color,
            ];
        });

        $tournamentAchievements = TournamentAchievement::active()->ordered()->get()->map(function ($achievement) {
            return [
                'year' => $achievement->year,
                'title' => $this->sanitizeText($achievement->title),
                'description' => $this->sanitizeText($achievement->description),
                'participants' => $achievement->participants,
            ];
        });

        $sponsors = Sponsor::active()->ordered()->get()->map(function ($sponsor) {
            return [
                'name' => $sponsor->name,
                'category' => $sponsor->category,
                'logo' => $sponsor->logo,
                'description' => $sponsor->description,
                'tier' => $sponsor->tier,
            ];
        });

        $testimonials = Testimonial::active()->featured()->ordered()->limit(3)->get()->map(function ($testimonial) {
            return [
                'name' => $testimonial->name,
                'role' => $testimonial->role,
                'team' => $testimonial->team,
                'image' => $testimonial->image,
                'rating' => $testimonial->rating,
                'quote' => $testimonial->quote,
                'achievement' => $testimonial->achievement,
            ];
        });

        // Get tournament statistics from actual data
        $tournamentStats = [
            ['label' => 'Active Tournaments', 'value' => Tournament::where('status', 'active')->count(), 'icon' => ''],
            ['label' => 'Registered Players', 'value' => Player::count() . '+', 'icon' => ''],
            ['label' => 'Matches Played', 'value' => GameMatch::where('status', 'completed')->count() . '+', 'icon' => ''],
            ['label' => 'Total Prize Pool', 'value' => 'Ksh. 70K', 'icon' => ''],
        ];

        // Get featured vlog video
        $featuredVlog = VlogVideo::published()->featured()->ordered()->where('platform', 'youtube')->first();
        $featuredVlogData = null;
        
        if ($featuredVlog) {
            $featuredVlogData = [
                'id' => $featuredVlog->id,
                'title' => $featuredVlog->title,
                'description' => $featuredVlog->description,
                'platform' => $featuredVlog->platform,
                'platform_icon' => $featuredVlog->platform_icon,
                'video_url' => $featuredVlog->video_url,
                'embed_url' => $featuredVlog->embed_code,
                'thumbnail_url' => $featuredVlog->thumbnail_url,
                'duration' => $featuredVlog->formatted_duration,
                'views' => $featuredVlog->formatted_views,
                'likes' => $featuredVlog->likes,
                'category' => $featuredVlog->category,
                'tags' => $featuredVlog->tags,
            ];
        }

        return Inertia::render('frontend/Index', [
            'heroSlides' => $heroSlides,
            'newsArticles' => $newsArticles,
            'tournamentHighlights' => $tournamentHighlights,
            'tournamentAchievements' => $tournamentAchievements,
            'sponsors' => $sponsors,
            'testimonials' => $testimonials,
            'tournamentStats' => $tournamentStats,
            'featuredVlog' => $featuredVlogData,
        ]);
    }

    public function about()
    {
        $tournamentAchievements = TournamentAchievement::active()->ordered()->get()->map(function ($achievement) {
            return [
                'year' => $achievement->year,
                'title' => $achievement->title,
                'description' => $achievement->description,
                'participants' => $achievement->participants,
            ];
        });

        $testimonials = Testimonial::active()->byRating(4)->ordered()->get()->map(function ($testimonial) {
            return [
                'name' => $testimonial->name,
                'role' => $testimonial->role,
                'team' => $testimonial->team,
                'image' => $testimonial->image,
                'rating' => $testimonial->rating,
                'quote' => $testimonial->quote,
                'achievement' => $testimonial->achievement,
            ];
        });

        // Get tournament statistics from actual data
        $tournamentStats = [
            ['label' => 'Active Tournaments', 'value' => Tournament::where('status', 'active')->count(), 'icon' => 'Trophy'],
            ['label' => 'Registered Players', 'value' => Player::count() . '+', 'icon' => 'Users'],
            ['label' => 'Matches Played', 'value' => GameMatch::where('status', 'completed')->count() . '+', 'icon' => 'Gamepad2'],
            ['label' => 'Total Prize Pool', 'value' => 'Ksh. 70K', 'icon' => 'DollarSign'],
        ];

        $tournamentHighlights = TournamentHighlight::active()->ordered()->get()->map(function ($highlight) {
            return [
                'icon' => $this->sanitizeIcon($highlight->icon),
                'title' => $highlight->title,
                'description' => $highlight->description,
                'value' => $highlight->value,
                'color' => $highlight->color,
            ];
        });

        $sponsors = Sponsor::active()->ordered()->get()->map(function ($sponsor) {
            return [
                'name' => $sponsor->name,
                'category' => $sponsor->category,
                'logo' => $sponsor->logo,
                'description' => $sponsor->description,
                'tier' => $sponsor->tier,
            ];
        });

        // Create timeline data from achievements
        $timeline = $tournamentAchievements->map(function ($achievement) {
            return [
                'year' => $achievement['year'],
                'event' => $achievement['title'],
                'icon' => 'Trophy',
            ];
        })->toArray();

        // Create why join data
        $whyJoin = [
            [
                'title' => 'Professional Experience',
                'desc' => 'Compete in a professionally organized tournament with proper facilities and officiating.',
                'icon' => 'ShieldCheck'
            ],
            [
                'title' => 'Attractive Prizes',
                'desc' => 'Win cash prizes, trophies, and recognition for outstanding performance.',
                'icon' => 'Award'
            ],
            [
                'title' => 'Community Building',
                'desc' => 'Connect with fellow players and build lasting friendships in the gaming community.',
                'icon' => 'Users'
            ],
            [
                'title' => 'Skill Development',
                'desc' => 'Improve your gaming skills through competitive matches and strategic gameplay.',
                'icon' => 'TrendingUp'
            ]
        ];

        // Create gallery images from achievements (placeholder images)
        $galleryImages = GalleryImage::take(6)->get()->map(function ($galleryImage) {
            return [
                'src' => $galleryImage->src,
                'alt' => $galleryImage->alt
            ];
        })->toArray();

        // Create organizers data
        $organizers = [
            // [
            //     'name' => 'Tournament Committee',
            //     'role' => 'Event Organizers',
            //     'img' => '/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png'
            // ],
            // [
            //     'name' => 'Technical Team',
            //     'role' => 'System Administrators',
            //     'img' => '/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png'
            // ],
            // [
            //     'name' => 'Media Team',
            //     'role' => 'Content & Broadcasting',
            //     'img' => '/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png'
            // ]
        ];

        return Inertia::render('frontend/About', [
            'tournamentAchievements' => $tournamentAchievements,
            'testimonials' => $testimonials,
            'tournamentStats' => $tournamentStats,
            'tournamentHighlights' => $tournamentHighlights,
            'sponsors' => $sponsors,
            'timeline' => $timeline,
            'whyJoin' => $whyJoin,
            'galleryImages' => $galleryImages,
            'organizers' => $organizers,
        ]);
    }

    public function founder()
    {
        return Inertia::render('frontend/Founder');
    }

    public function fixtures()
    {
        $fixtures = GameMatch::with(['team1', 'team2', 'tournament'])
            ->orderBy('scheduled_time', 'asc')
            ->get()
            ->map(function ($match) {
                $status = 'upcoming';
                $result = '-';
                
                // Determine status based on match progress
                if ($match->status === 'completed') {
                    $status = 'completed';
                    // Format result based on scores
                    if ($match->player1_score !== null && $match->player2_score !== null) {
                        $result = $match->player1_score . ' - ' . $match->player2_score;
                    } elseif ($match->team1 && $match->team2) {
                        // For team matches, use team games won or default format
                        $team1Score = $match->team1_games_won ?? 0;
                        $team2Score = $match->team2_games_won ?? 0;
                        $result = $team1Score . ' - ' . $team2Score;
                    }
                } elseif ($match->progress_status === 'in_progress') {
                    $status = 'live';
                    // For live matches, show current score with time if available
                    if ($match->player1_score !== null && $match->player2_score !== null) {
                        $liveTime = $match->started_at ? $match->started_at->diffInMinutes(now()) : '';
                        $result = $match->player1_score . ' - ' . $match->player2_score . ($liveTime ? " ({$liveTime}')" : ' (Live)');
                    } elseif ($match->team1 && $match->team2) {
                        $team1Score = $match->team1_games_won ?? 0;
                        $team2Score = $match->team2_games_won ?? 0;
                        $liveTime = $match->started_at ? $match->started_at->diffInMinutes(now()) : '';
                        $result = $team1Score . ' - ' . $team2Score . ($liveTime ? " ({$liveTime}')" : ' (Live)');
                    } else {
                        $result = 'Live';
                    }
                }

                // Get team information
                $teamA = [
                    'name' => 'Team A',
                    'logo' => '/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png' // Default logo
                ];
                $teamB = [
                    'name' => 'Team B', 
                    'logo' => '/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png' // Default logo
                ];

                // Use team1/team2 if available, otherwise use player names
                if ($match->team1) {
                    $teamA['name'] = $match->team1->name;
                    $teamA['logo'] = $match->team1->logo ?: '/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png';
                } elseif ($match->player1) {
                    $teamA['name'] = $match->player1->display_name ?? $match->player1->name ?? 'Player 1';
                }

                if ($match->team2) {
                    $teamB['name'] = $match->team2->name;
                    $teamB['logo'] = $match->team2->logo ?: '/lovable-uploads/18075074-5322-458c-b0cf-a75db311c3c0.png';
                } elseif ($match->player2) {
                    $teamB['name'] = $match->player2->display_name ?? $match->player2->name ?? 'Player 2';
                }

                return [
                    'id' => $match->id,
                    'date' => $match->scheduled_time ? $match->scheduled_time->format('Y-m-d') : now()->format('Y-m-d'),
                    'time' => $match->scheduled_time ? $match->scheduled_time->format('H:i') : '00:00',
                    'teamA' => $teamA,
                    'teamB' => $teamB,
                    'venue' => $match->server_location ?: 'Main Stadium',
                    'status' => $status,
                    'result' => $result,
                ];
            });

        return Inertia::render('frontend/Fixtures', [
            'fixtures' => $fixtures,
        ]);
    }

    public function contact()
    {
        $contactInfo = [
            ['icon' => 'Mail', 'label' => 'Email', 'value' => 'themilimusgerrytournamentcup@gmail.com'],
            ['icon' => 'Phone', 'label' => 'Phone', 'value' => '+254 796 7396 996'],
            ['icon' => 'MapPin', 'label' => 'Location', 'value' => 'Ingolomosio, Shinyalu Constituency, Kakamega, Kenya'],
        ];

        return Inertia::render('frontend/Contact', [
            'contactInfo' => $contactInfo,
        ]);
    }

    public function contactSubmit(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:2000',
        ]);

        // TODO: Implement contact form submission logic
        // For now, just return success response
        return redirect()->back()->with('success', 'Your message has been sent successfully!');
    }

    public function rules()
    {
        return Inertia::render('frontend/Rules');
    }

    public function prizes()
    {
        // Fetch active prize awards grouped by category
        $prizeAwards = PrizeAward::active()->ordered()->with('tournament')->get()->groupBy('category');
        
        // Calculate tournament statistics
        $tournamentStats = [
            [
                'icon' => 'Trophy',
                'value' => PrizeAward::active()->where('category', 'team')->count(),
                'label' => 'Team Prizes'
            ],
            [
                'icon' => 'Medal',
                'value' => PrizeAward::active()->where('category', 'individual')->count(),
                'label' => 'Individual Awards'
            ],
            [
                'icon' => 'Award',
                'value' => PrizeAward::active()->where('category', 'special')->count(),
                'label' => 'Special Awards'
            ],
            [
                'icon' => 'Star',
                'value' => PrizeAward::active()->where('category', 'performance')->count(),
                'label' => 'Performance Awards'
            ]
        ];

        // Calculate total prize pool
        $totalPrizePool = PrizeAward::active()->sum('prize_value');
        
        // Calculate prize distribution by category
        $prizeDistribution = [
            'team' => [
                'total' => PrizeAward::active()->where('category', 'team')->sum('prize_value'),
                'percentage' => $totalPrizePool > 0 ? round((PrizeAward::active()->where('category', 'team')->sum('prize_value') / $totalPrizePool) * 100) : 0
            ],
            'individual' => [
                'total' => PrizeAward::active()->where('category', 'individual')->sum('prize_value'),
                'percentage' => $totalPrizePool > 0 ? round((PrizeAward::active()->where('category', 'individual')->sum('prize_value') / $totalPrizePool) * 100) : 0
            ],
            'special' => [
                'total' => PrizeAward::active()->where('category', 'special')->sum('prize_value'),
                'percentage' => $totalPrizePool > 0 ? round((PrizeAward::active()->where('category', 'special')->sum('prize_value') / $totalPrizePool) * 100) : 0
            ]
        ];

        return Inertia::render('frontend/Prizes', [
            'prizeAwards' => $prizeAwards,
            'tournamentStats' => $tournamentStats,
            'totalPrizePool' => $totalPrizePool,
            'prizeDistribution' => $prizeDistribution,
            'categories' => PrizeAward::getCategories(),
            'prizeTypes' => PrizeAward::getPrizeTypes(),
            'positions' => PrizeAward::getPositions(),
        ]);
    }

    public function certificates()
    {
        return Inertia::render('frontend/Certificates');
    }

    public function help()
    {
        return Inertia::render('frontend/Help');
    }

    public function faq()
    {
        $faqs = Faq::active()->ordered()->get()->groupBy('category');
        
        return Inertia::render('frontend/FAQ', [
            'faqs' => $faqs,
            'categories' => Faq::getCategories(),
        ]);
    }

    public function techSupport()
    {
        return Inertia::render('frontend/TechSupport');
    }

    public function privacy()
    {
        return Inertia::render('frontend/Privacy');
    }

    public function terms()
    {
        return Inertia::render('frontend/Terms');
    }

    public function cookies()
    {
        return Inertia::render('frontend/Cookies');
    }

    public function howToRegister()
    {
        return Inertia::render('frontend/HowToRegister');
    }

    public function gallery()
    {
        $galleryImages = GalleryImage::published()->ordered()->get()->map(function ($image) {
            return [
                'id' => $image->id,
                'src' => $image->src,
                'alt' => $image->alt,
                'category' => $image->category,
                'edition' => $image->edition,
                'height' => $image->height,
            ];
        });

        $galleryCategories = [
            ['id' => 'all', 'name' => 'All Photos'],
            ['id' => 'matches', 'name' => 'Matches'],
            ['id' => 'celebrations', 'name' => 'Celebrations'],
            ['id' => 'awards', 'name' => 'Awards'],
            ['id' => 'venues', 'name' => 'Venues'],
            ['id' => 'training', 'name' => 'Training'],
            ['id' => 'fans', 'name' => 'Fans'],
            ['id' => 'teams', 'name' => 'Teams'],
        ];

        $galleryEditions = [
            ['id' => 'all', 'name' => 'All Editions'],
            ['id' => '1st', 'name' => '1st Edition'],
            ['id' => '2nd', 'name' => '2nd Edition'],
            ['id' => '3rd', 'name' => '3rd Edition'],
            ['id' => '4th', 'name' => '4th Edition'],
        ];

        return Inertia::render('frontend/Gallery', [
            'galleryImages' => $galleryImages,
            'galleryCategories' => $galleryCategories,
            'galleryEditions' => $galleryEditions,
        ]);
    }

    public function vlog()
    {
        $vlogVideos = \App\Models\Frontend\VlogVideo::published()->ordered()->get()->map(function ($video) {
            return [
                'id' => $video->id,
                'title' => $video->title,
                'description' => $video->description,
                'platform' => $video->platform,
                'platform_icon' => $video->platform_icon,
                'video_url' => $video->video_url,
                'embed_url' => $video->embed_code,
                'thumbnail_url' => $video->thumbnail_url,
                'duration' => $video->formatted_duration,
                'views' => $video->formatted_views,
                'likes' => $video->likes,
                'category' => $video->category,
                'tags' => $video->tags,
                'featured' => $video->featured,
                'published_at' => $video->published_at?->format('Y-m-d'),
            ];
        });

        $vlogCategories = [
            ['id' => 'all', 'name' => 'All Videos'],
            ['id' => 'matches', 'name' => 'Match Highlights'],
            ['id' => 'interviews', 'name' => 'Player Interviews'],
            ['id' => 'behind-scenes', 'name' => 'Behind the Scenes'],
            ['id' => 'training', 'name' => 'Training Sessions'],
            ['id' => 'celebrations', 'name' => 'Celebrations'],
            ['id' => 'awards', 'name' => 'Award Ceremonies'],
            ['id' => 'fans', 'name' => 'Fan Content'],
            ['id' => 'announcements', 'name' => 'Announcements'],
            ['id' => 'tutorials', 'name' => 'Tutorials'],
        ];

        $vlogPlatforms = [
            ['id' => 'all', 'name' => 'All Platforms'],
            ['id' => 'youtube', 'name' => 'YouTube'],
            ['id' => 'instagram', 'name' => 'Instagram'],
            ['id' => 'facebook', 'name' => 'Facebook'],
            ['id' => 'tiktok', 'name' => 'TikTok'],
            ['id' => 'twitter', 'name' => 'Twitter/X'],
            ['id' => 'linkedin', 'name' => 'LinkedIn'],
            ['id' => 'vimeo', 'name' => 'Vimeo'],
        ];

        return Inertia::render('frontend/Vlog', [
            'vlogVideos' => $vlogVideos,
            'vlogCategories' => $vlogCategories,
            'vlogPlatforms' => $vlogPlatforms,
        ]);
    }

    public function news()
    {
        $newsArticles = NewsArticle::published()->latest()->get()->map(function ($article) {
            return [
                'id' => $article->id,
                'title' => $article->title,
                'excerpt' => $article->excerpt,
                'category' => $article->category,
                'author' => $article->author,
                'authorImage' => $article->author_image,
                'date' => $article->date->format('Y-m-d'),
                'readTime' => $article->read_time,
                'image' => $article->image,
                'tags' => $article->tags,
                'views' => $article->views,
                'likes' => $article->likes,
                'featured' => $article->featured,
                'slug' => $article->slug,
            ];
        });

        $newsCategories = [
            ['id' => 'all', 'name' => 'All News'],
            ['id' => 'tournament', 'name' => 'Tournament Updates'],
            ['id' => 'registration', 'name' => 'Registration'],
            ['id' => 'teams', 'name' => 'Team News'],
            ['id' => 'results', 'name' => 'Match Results'],
            ['id' => 'announcements', 'name' => 'Announcements'],
        ];

        return Inertia::render('frontend/News', [
            'newsArticles' => $newsArticles,
            'newsCategories' => $newsCategories,
        ]);
    }

    public function newsArticle($slug)
    {
        $article = NewsArticle::published()->where('slug', $slug)->firstOrFail();
        
        // Increment views
        $article->increment('views');

        $articleData = [
            'id' => $article->id,
            'title' => $article->title,
            'content' => $article->content,
            'category' => $article->category,
            'author' => $article->author,
            'authorImage' => $article->author_image,
            'date' => $article->date->format('Y-m-d'),
            'readTime' => $article->read_time,
            'image' => $article->image,
            'tags' => $article->tags,
            'views' => $article->views,
            'likes' => $article->likes,
            'slug' => $article->slug,
        ];

        // Get related articles
        $relatedArticles = NewsArticle::published()
            ->where('category', $article->category)
            ->where('id', '!=', $article->id)
            ->latest()
            ->limit(3)
            ->get()
            ->map(function ($relatedArticle) {
                return [
                    'id' => $relatedArticle->id,
                    'title' => $relatedArticle->title,
                    'excerpt' => $relatedArticle->excerpt,
                    'image' => $relatedArticle->image,
                    'slug' => $relatedArticle->slug,
                ];
            });

        return Inertia::render('frontend/NewsArticle', [
            'article' => $articleData,
            'relatedArticles' => $relatedArticles,
        ]);
    }

    public function foundersMessage()
    {
        return Inertia::render('frontend/FoundersMessage');
    }


    public function teams()
    {
        $query = Team::with(['tournaments', 'captain', 'manager', 'players'])
            ->whereHas('tournaments', function ($q) {
                $q->where('tournaments.status', 'active');
            });

        // Filter by tournament if specified
        if (request()->has('tournament') && request()->tournament) {
            $query->whereHas('tournaments', function ($q) {
                $q->where('tournament_id', request()->tournament);
            });
        }

        // Filter by status
        if (request()->has('status') && request()->status) {
            $query->whereHas('tournaments', function ($q) {
                $q->where('tournaments.status', request()->status);
            });
        }

        // Search by team name
        if (request()->has('search') && request()->search) {
            $query->where('name', 'like', '%' . request()->search . '%');
        }

        $teams = $query->orderBy('name', 'asc')->paginate(12);

        // Get all active tournaments for filter dropdown
        $tournaments = Tournament::where('status', 'active')
            ->where('visibility', 'public')
            ->orderBy('name', 'asc')
            ->get()
            ->map(function ($tournament) {
                return [
                    'id' => $tournament->id,
                    'name' => $tournament->name,
                    'slug' => $tournament->slug,
                ];
            });

        // Transform teams data
        $teamsData = $teams->through(function ($team) {
            $activeTournaments = $team->tournaments->where('pivot.status', 'active');
            $defaultFormation = $team->defaultFormation()->first();
            
            return [
                'id' => $team->id,
                'name' => $team->name,
                'description' => $team->description,
                'logo' => $team->logo,
                'country' => $team->country,
                'county' => $team->county,
                'location' => $team->location,
                'captain' => $team->captain ? [
                    'id' => $team->captain->id,
                    'name' => $team->captain->name,
                ] : null,
                'manager' => $team->manager ? [
                    'id' => $team->manager->id,
                    'name' => $team->manager->name,
                ] : null,
                'player_count' => $team->players->count(),
                'active_tournaments' => $activeTournaments->map(function ($tournament) {
                    return [
                        'id' => $tournament->id,
                        'name' => $tournament->name,
                        'slug' => $tournament->slug,
                    ];
                }),
                'has_formation' => $defaultFormation ? true : false,
                'formation_name' => $defaultFormation ? $defaultFormation->name : null,
                'stats' => $team->stats,
            ];
        });

        return Inertia::render('frontend/Teams', [
            'teams' => $teamsData,
            'tournaments' => $tournaments,
            'filters' => [
                'tournament' => request()->tournament,
                'status' => request()->status,
                'search' => request()->search,
            ],
        ]);
    }   

    public function team(Team $team, Request $request)
    {
         // Load team with all necessary relationships
            $team->load([
                'tournaments' => function ($query) {
                    $query->where('tournaments.status', 'active');
                },
                'captain',
                'manager',
                'players.user',
                'officials.user',
                'formations' => function ($query) {
                    $query->where('is_default', true);
                },
                'formations.formation'
            ]);

            // Get active tournaments for this team
            $activeTournaments = $team->tournaments->where('pivot.status', 'active');
            
            // Get default formation
            $defaultFormation = $team->formations->first();
            
            // Get team statistics
            $teamStats = [
                'total_players' => $team->players->count(),
                'active_players' => $team->players->where('status', 'active')->count(),
                'total_officials' => $team->officials->count(),
                'tournaments_participated' => $activeTournaments->count(),
                'matches_played' => $team->matchesAsTeam1()->count() + $team->matchesAsTeam2()->count(),
                'matches_won' => $team->matchesAsWinningTeam()->count(),
            ];

            // Transform team data
            $teamData = [
                'id' => $team->id,
                'name' => $team->name,
                'description' => $team->description,
                'logo' => $team->logo,
                'country' => $team->country,
                'county' => $team->county,
                'location' => $team->location,
                'captain' => $team->captain ? [
                    'id' => $team->captain->id,
                    'name' => $team?->captain?->name,
                    'email' => $team?->captain?->email,
                ] : null,
                'manager' => $team->manager ? [
                    'id' => $team->manager->id,
                    'name' => $team?->manager?->name,
                    'email' => $team?->manager?->email,
                ] : null,
                'players' => $team->players->map(function ($player) {
                    return [
                        'id' => $player->id,
                        'display_name' => $player->display_name,
                        'user' => [
                            'name' => $player?->user?->name,
                        ],
                        'jersey_number' => $player->jersey_number,
                        'football_position' => $player->football_position,
                        'status' => $player->status,
                        'avatar' => $player->avatar,
                    ];
                }),
                'officials' => $team->officials->map(function ($official) {
                    return [
                        'id' => $official->id,
                        'user' => [
                            'name' => $official?->user?->name,
                        ],
                        'role' => $official->role,
                        'status' => $official->status,
                    ];
                }),
                'active_tournaments' => $activeTournaments->map(function ($tournament) {
                    return [
                        'id' => $tournament->id,
                        'name' => $tournament->name,
                        'slug' => $tournament->slug,
                        'start_date' => $tournament->start_date?->format('Y-m-d'),
                        'end_date' => $tournament->end_date?->format('Y-m-d'),
                    ];
                }),
                'default_formation' => $defaultFormation ? [
                    'id' => $defaultFormation->id,
                    'name' => $defaultFormation->name,
                    'formation' => [
                        'id' => $defaultFormation->formation->id,
                        'name' => $defaultFormation->formation->name,
                        'display_name' => $defaultFormation->formation->display_name,
                    ],
                    'player_positions' => $defaultFormation->player_positions,
                    'formation_data' => $defaultFormation->formation_data,
                ] : null,
                'stats' => $team->stats,
                'team_stats' => $teamStats,
            ];

            return Inertia::render('frontend/Team', [
                'team' => $teamData,
            ]);
        }
    
} 