<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Tournament;
use App\Models\Player;
use App\Models\Team;
use App\Models\GameMatch;
use App\Models\User;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        // Get overview statistics
        $totalTournaments = Tournament::count();
        $activeTournaments = Tournament::where('status', 'active')->count();
        $totalPlayers = Player::count();
        $totalTeams = Team::count();
        $totalMatches = GameMatch::count();
        $activeMatches = GameMatch::whereIn('status', ['scheduled', 'in_progress'])->count();
        $totalUsers = User::count();

        // Get recent tournaments
        $recentTournaments = Tournament::with(['matches'])
            ->latest()
            ->limit(5)
            ->get()
            ->map(function ($tournament) {
                return [
                    'id' => $tournament->id,
                    'name' => $tournament->name,
                    'status' => $tournament->status,
                    'start_date' => $tournament->start_date,
                    'end_date' => $tournament->end_date,
                    'matches_count' => $tournament->matches->count(),
                    'game_type' => $tournament->game_type,
                ];
            });

        // Get upcoming matches
        $upcomingMatches = GameMatch::with(['tournament', 'player1', 'player2', 'team1', 'team2'])
            ->where('scheduled_time', '>', now())
            ->orderBy('scheduled_time')
            ->limit(5)
            ->get()
            ->map(function ($match) {
                return [
                    'id' => $match->id,
                    'name' => $match->name,
                    'tournament' => $match->tournament ? $match->tournament->name : null,
                    'scheduled_time' => $match->scheduled_time,
                    'match_type' => $match->match_type,
                    'participant1' => $match->player1 ? $match->player1->display_name : ($match->team1 ? $match->team1->name : null),
                    'participant2' => $match->player2 ? $match->player2->display_name : ($match->team2 ? $match->team2->name : null),
                    'status' => $match->status,
                ];
            });

        // Get recent matches
        $recentMatches = GameMatch::with(['tournament', 'player1', 'player2', 'team1', 'team2'])
            ->where('status', 'completed')
            ->latest('updated_at')
            ->limit(5)
            ->get()
            ->map(function ($match) {
                return [
                    'id' => $match->id,
                    'name' => $match->name,
                    'tournament' => $match->tournament ? $match->tournament->name : null,
                    'completed_at' => $match->updated_at,
                    'match_type' => $match->match_type,
                    'participant1' => $match->player1 ? $match->player1->display_name : ($match->team1 ? $match->team1->name : null),
                    'participant2' => $match->player2 ? $match->player2->display_name : ($match->team2 ? $match->team2->name : null),
                    'score' => $match->player1_score . ' - ' . $match->player2_score,
                ];
            });

        // Get tournament status distribution
        $tournamentStatusStats = Tournament::selectRaw('status, COUNT(*) as count')
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();

        // Get match status distribution
        $matchStatusStats = GameMatch::selectRaw('status, COUNT(*) as count')
            ->groupBy('status')
            ->pluck('count', 'status')
            ->toArray();

        // Get monthly tournament creation stats (last 6 months)
        $monthlyTournamentStats = Tournament::selectRaw('YEAR(created_at) as year, MONTH(created_at) as month, COUNT(*) as count')
            ->where('created_at', '>=', Carbon::now()->subMonths(6))
            ->groupByRaw('YEAR(created_at), MONTH(created_at)')
            ->orderByRaw('YEAR(created_at), MONTH(created_at)')
            ->get()
            ->map(function ($item) {
                return [
                    'month' => Carbon::create($item->year, $item->month)->format('M Y'),
                    'count' => $item->count,
                ];
            });

        return Inertia::render('dashboard', [
            'stats' => [
                'totalTournaments' => $totalTournaments,
                'activeTournaments' => $activeTournaments,
                'totalPlayers' => $totalPlayers,
                'totalTeams' => $totalTeams,
                'totalMatches' => $totalMatches,
                'activeMatches' => $activeMatches,
                'totalUsers' => $totalUsers,
            ],
            'recentTournaments' => $recentTournaments,
            'upcomingMatches' => $upcomingMatches,
            'recentMatches' => $recentMatches,
            'tournamentStatusStats' => $tournamentStatusStats,
            'matchStatusStats' => $matchStatusStats,
            'monthlyTournamentStats' => $monthlyTournamentStats,
        ]);
    }
} 