<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class CertificateVerificationController extends Controller
{
    /**
     * Verify a certificate by ID or QR code data
     */
    public function verify(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'certificate_id' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid certificate ID format',
                'errors' => $validator->errors()
            ], 422);
        }

        $certificateId = $request->input('certificate_id');

        try {
            // In a real implementation, you would query the database
            // For now, we'll simulate the verification process
            
            // Mock certificate data - replace with actual database query
            $certificate = $this->findCertificateById($certificateId);

            if (!$certificate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Certificate not found',
                    'data' => null
                ], 404);
            }

            // Check if certificate is expired
            $isExpired = $this->isCertificateExpired($certificate['valid_until']);
            
            $status = 'valid';
            if ($isExpired) {
                $status = 'expired';
            }

            $certificate['status'] = $status;

            return response()->json([
                'success' => true,
                'message' => 'Certificate verification completed',
                'data' => $certificate
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Verification failed. Please try again.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Find certificate by ID (mock implementation)
     */
    private function findCertificateById($certificateId)
    {
        // Mock data - replace with actual database query
        $mockCertificates = [
            'CERT-000075-770DE884' => [
                'id' => 'CERT-000075-770DE884',
                'type' => 'participation',
                'recipient' => [
                    'name' => 'Victor Kipkoech',
                    'team' => 'Lwanda K FC',
                    'position' => 'Striker'
                ],
                'tournament' => [
                    'name' => 'Milimus Tournament Championship 2024',
                    'startDate' => '2024-08-03',
                    'endDate' => '2024-08-31',
                    'location' => 'Nairobi, Kenya'
                ],
                'generatedAt' => 'September 20, 2025',
                'validUntil' => '2026-09-20',
                'qrCode' => 'https://milimus.com/verify/CERT-000075-770DE884'
            ],
            'CERT-000001-ABC12345' => [
                'id' => 'CERT-000001-ABC12345',
                'type' => 'achievement',
                'recipient' => [
                    'name' => 'John Doe',
                    'team' => 'Team Alpha',
                    'position' => 'Midfielder'
                ],
                'tournament' => [
                    'name' => 'Spring Football League 2024',
                    'startDate' => '2024-04-01',
                    'endDate' => '2024-04-30',
                    'location' => 'Mombasa, Kenya'
                ],
                'generatedAt' => 'May 1, 2024',
                'validUntil' => '2025-05-01',
                'qrCode' => 'https://milimus.com/verify/CERT-000001-ABC12345'
            ]
        ];

        return $mockCertificates[$certificateId] ?? null;
    }

    /**
     * Check if certificate is expired
     */
    private function isCertificateExpired($validUntil)
    {
        try {
            $expiryDate = \Carbon\Carbon::parse($validUntil);
            return $expiryDate->isPast();
        } catch (\Exception $e) {
            return true; // If we can't parse the date, consider it expired
        }
    }

    /**
     * Get certificate verification page
     */
    public function index()
    {
        return view('certificates.verification');
    }

    /**
     * Bulk verify multiple certificates
     */
    public function bulkVerify(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'certificate_ids' => 'required|array|min:1|max:10',
            'certificate_ids.*' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid certificate IDs format',
                'errors' => $validator->errors()
            ], 422);
        }

        $certificateIds = $request->input('certificate_ids');
        $results = [];

        foreach ($certificateIds as $certificateId) {
            $certificate = $this->findCertificateById($certificateId);
            
            if ($certificate) {
                $isExpired = $this->isCertificateExpired($certificate['valid_until']);
                $certificate['status'] = $isExpired ? 'expired' : 'valid';
                $results[] = $certificate;
            } else {
                $results[] = [
                    'id' => $certificateId,
                    'status' => 'invalid',
                    'error' => 'Certificate not found'
                ];
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Bulk verification completed',
            'data' => $results
        ]);
    }

    /**
     * Get certificate statistics
     */
    public function statistics()
    {
        // Mock statistics - replace with actual database queries
        $stats = [
            'total_certificates' => 1247,
            'valid_certificates' => 1189,
            'expired_certificates' => 45,
            'invalid_certificates' => 13,
            'verification_requests_today' => 23,
            'most_common_type' => 'participation'
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }
}
