<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Bracket;
use App\Http\Requests\StoreBracketRequest;
use App\Http\Requests\UpdateBracketRequest;

class BracketController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Bracket::query();

        // Filter by tournament
        if ($request->has('tournament_id')) {
            $query->where('tournament_id', $request->tournament_id);
        }

        // Filter by type
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // Filter by active status
        if ($request->has('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        // Search by name
        if ($request->has('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Include relationships
        if ($request->has('with')) {
            $relations = explode(',', $request->with);
            $query->with($relations);
        }

        $brackets = $query->with(['tournament', 'gameMatches'])->paginate($request->get('per_page', 20));
        return response()->json($brackets);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreBracketRequest $request)
    {
        $bracket = Bracket::create($request->validated());
        return response()->json($bracket->load(['tournament', 'gameMatches']), 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Bracket $bracket, Request $request)
    {
        $relations = ['tournament', 'gameMatches'];
        
        if ($request->has('with')) {
            $relations = array_merge($relations, explode(',', $request->with));
        }
        
        return response()->json($bracket->load($relations));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateBracketRequest $request, Bracket $bracket)
    {
        $bracket->update($request->validated());
        return response()->json($bracket->load(['tournament', 'gameMatches']));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Bracket $bracket)
    {
        $bracket->delete();
        return response()->json(['message' => 'Bracket deleted']);
    }

    // Additional endpoints
    public function matches(Bracket $bracket, Request $request)
    {
        $query = $bracket->gameMatches()->with(['player1', 'player2', 'team1', 'team2', 'winner', 'winningTeam']);

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by round
        if ($request->has('round')) {
            $query->where('round', $request->round);
        }

        $matches = $query->orderBy('round')->orderBy('match_number')->paginate($request->get('per_page', 20));
        return response()->json($matches);
    }

    public function rounds(Bracket $bracket)
    {
        $rounds = $bracket->gameMatches()
            ->select('round')
            ->distinct()
            ->orderBy('round')
            ->pluck('round');

        $roundData = [];
        foreach ($rounds as $round) {
            $matches = $bracket->gameMatches()
                ->where('round', $round)
                ->with(['player1', 'player2', 'team1', 'team2', 'winner', 'winningTeam'])
                ->orderBy('match_number')
                ->get();

            $roundData[] = [
                'round' => $round,
                'matches' => $matches,
                'total_matches' => $matches->count(),
                'completed_matches' => $matches->where('status', 'completed')->count(),
            ];
        }

        return response()->json($roundData);
    }

    public function generateBracket(Bracket $bracket, Request $request)
    {
        $request->validate([
            'participants' => 'required|array',
            'participants.*.id' => 'required|integer',
            'participants.*.type' => 'required|in:player,team',
            'bracket_type' => 'nullable|string|in:winners,losers,final',
            'seeding_method' => 'nullable|string|in:random,ranking,rating,balanced',
        ]);

        try {
            // Use the enhanced tournament bracket service
            $bracketService = new \App\Services\TournamentBracketService($bracket->tournament);
            
            $bracketType = $request->get('bracket_type', 'winners');
            $seedingMethod = $request->get('seeding_method', 'random');
            
            // Update tournament settings if seeding method is provided
            if ($seedingMethod) {
                $bracketSettings = $bracket->tournament->bracket_settings ?? [];
                $bracketSettings['seeding_method'] = $seedingMethod;
                $bracket->tournament->update(['bracket_settings' => $bracketSettings]);
            }
            
            // Generate the bracket using the enhanced service
            $generatedBracket = $bracketService->generateBracket($request->participants, $bracketType);
            
            return response()->json([
                'message' => 'Bracket generated successfully',
                'bracket' => $generatedBracket->load('gameMatches'),
                'structure' => $generatedBracket->structure,
                'seeding' => $generatedBracket->seeding,
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Bracket generation failed: ' . $e->getMessage());
            return response()->json([
                'error' => 'Bracket generation failed: ' . $e->getMessage()
            ], 500);
        }
    }

    public function advanceRound(Bracket $bracket, Request $request)
    {
        $request->validate([
            'current_round' => 'required|integer|min:1',
        ]);

        try {
            $bracketService = new \App\Services\TournamentBracketService($bracket->tournament);
            $result = $bracketService->advanceRound($bracket, $request->current_round);
            
            return response()->json([
                'message' => 'Round advanced successfully',
                'next_round' => $result['round'],
                'matches_created' => $result['matches_created'],
                'winners' => $result['winners'],
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Round advancement failed: ' . $e->getMessage());
            return response()->json([
                'error' => 'Round advancement failed: ' . $e->getMessage()
            ], 500);
        }
    }

    public function statistics(Bracket $bracket)
    {
        $stats = [
            'total_matches' => $bracket->gameMatches()->count(),
            'completed_matches' => $bracket->gameMatches()->where('status', 'completed')->count(),
            'scheduled_matches' => $bracket->gameMatches()->where('status', 'scheduled')->count(),
            'in_progress_matches' => $bracket->gameMatches()->where('status', 'in_progress')->count(),
            'total_rounds' => $bracket->rounds,
            'current_round' => $bracket->gameMatches()->max('round') ?? 0,
            'is_active' => $bracket->is_active,
        ];
        
        return response()->json($stats);
    }
}
