<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Team;
use App\Models\Player;
use App\Models\TeamOfficial;
use App\Services\OtpService;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rules;
use Inertia\Inertia;
use Inertia\Response;
use App\Models\Tournament;
use Illuminate\Support\Facades\Validator;
use App\Notifications\User\TeamCaptainInvitationNotification;
use App\Models\Role;

class RegisteredUserController extends Controller
{
    protected $otpService;
    
    public function __construct(OtpService $otpService)
    {
        $this->otpService = $otpService;
    }
    
    /**
     * Show the team registration page.
     */
    public function create(): Response
    {
        // Get all available tournaments for registration
        $tournaments = Tournament::where('status', 'active')
            ->select('id', 'name', 'description', 'entry_fee', 'currency', 'start_date', 'end_date', 'registration_end', 'min_participants', 'max_participants', 'min_players_per_team', 'max_players_per_team', 'enable_categories', 'category_fees', 'available_categories')
            ->get()
            ->map(function ($tournament) {
                // Use min/max_players_per_team for player limits, fallback to config if not set
                $tournament->min_players = $tournament->min_players_per_team ?? config('tournament.min_players_per_team', 11);
                $tournament->max_players = $tournament->max_players_per_team ?? config('tournament.max_players_per_team', 23);
                return $tournament;
            });
        
        // Get the default registration fee from config
        $defaultRegistrationFee = config('tournament.registration_fee', 1000);
        
        // Get category locations from config
        $categoryLocations = config('tournament.category_locations', []);
        
        // Get global location settings
        $countriesString = config('tournament.available_countries', 'Kenya');
        $countiesString = config('tournament.available_counties', 'Kakamega County,Nairobi County,Mombasa County');
        
        $countries = array_map('trim', explode(',', $countriesString));
        $counties = array_map('trim', explode(',', $countiesString));

        return Inertia::render('frontend/auth/RegisterTeam', [
            'registration_fee' => $defaultRegistrationFee,
            'tournaments' => $tournaments->map(function($tournament) {
                return [
                    'id' => $tournament->id,
                    'name' => $tournament->name,
                    'description' => $tournament->description,
                    'entry_fee' => $tournament->entry_fee,
                    'currency' => $tournament->currency ?? 'KES',
                    'start_date' => $tournament->start_date?->format('F j, Y'),
                    'end_date' => $tournament->end_date?->format('F j, Y'),
                    'registration_end' => $tournament->registration_end?->format('F j, Y'),
                    'is_free' => !$tournament->entry_fee,
                    'enable_categories' => $tournament->enable_categories ?? false,
                    'category_fees' => $tournament->getAllCategoryFees(),
                    'available_categories' => $tournament->getAvailableCategories(),
                    'min_players' => $tournament->min_participants,
                    'max_players' => $tournament->max_participants,
                ];
            }),
            'min_players' => config('tournament.min_players_per_team', 11),
            'max_players' => config('tournament.max_players_per_team', 23),
            'max_officials' => config('tournament.max_officials_per_team', 10),
            'default_tournament' => $tournaments->first() ? [
                'id' => $tournaments->first()->id,
                'name' => $tournaments->first()->name,
                'entry_fee' => $tournaments->first()->entry_fee,
                'currency' => $tournaments->first()->currency ?? 'KES',
                'enable_categories' => $tournaments->first()->enable_categories ?? false,
                'category_fees' => $tournaments->first()->getAllCategoryFees(),
                'available_categories' => $tournaments->first()->getAvailableCategories(),
                'min_players' => $tournaments->first()->min_players_per_team ?? config('tournament.min_players_per_team', 11),
                'max_players' => $tournaments->first()->max_players_per_team ?? config('tournament.max_players_per_team', 23),
            ] : null,
            'categories' => config('tournament.categories', []),
            'category_locations' => $categoryLocations,
            'countries' => $countries,
            'counties' => $counties,
        ]);
    }

    /**
     * Show the user registration page.
     */
    public function createUser(): Response
    {
        // Get the active tournament and its entry fee
        $tournament = \App\Models\Tournament::where('status', 'active')->first();
        $registrationFee = $tournament ? $tournament->entry_fee : config('tournament.registration_fee', 1000);
        
        return Inertia::render('frontend/auth/RegisterUser', [
            'registration_fee' => $registrationFee,
            'tournament' => $tournament ? [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'entry_fee' => $tournament->entry_fee,
                'currency' => $tournament->currency ?? 'KES'
            ] : null
        ]);
    }

    /**
     * Handle an incoming team registration request.
     */
    public function store(Request $request)
    {
        // Get tournament first to use its player limits for validation
        $tournament = Tournament::find($request->tournament_id);
        $minPlayers = $tournament ? ($tournament->min_players_per_team ?? config('tournament.min_players_per_team', 11)) : 11;
        $maxPlayers = $tournament ? ($tournament->max_players_per_team ?? config('tournament.max_players_per_team', 23)) : 23;

        $validator = Validator::make($request->all(), [
            // Tournament selection
            'tournament_id' => 'required|exists:tournaments,id',
            
            // Team details
            'team.name' => 'required|string|max:255',
            'team.email' => 'required|string|lowercase|email|max:255',
            'team.phone' => 'required|string|max:20',
            'team.country' => 'required|string|max:255',
            'team.county' => 'required|string|max:255',
            'team.location' => 'required|string|max:255',
            'team.year' => 'required|integer',
            'team.category' => 'required|in:men,women,youth_male,youth_female,children',
            
            // Team captain (user account)
            'captain.name' => 'required|string|max:255',
            'captain.email' => 'required|string|lowercase|email|max:255',
            'captain.phone' => 'required|string|max:20',
            'captain.password' => ['required', 'confirmed', Rules\Password::defaults()],
            
            // Players - Use tournament's player limits dynamically
            'players' => "required|array|min:{$minPlayers}|max:{$maxPlayers}",
            'players.*.name' => 'required|string|max:255',
            'players.*.email' => 'required|string|lowercase|email|max:255|distinct',
            'players.*.phone' => 'nullable|string|max:20',
            'players.*.position' => 'required|in:Goalkeeper,Defender,Midfielder,Forward',
            'players.*.id_number' => 'required|string|max:50',
            'players.*.country' => 'required|string|max:255',
            'players.*.county' => 'required|string|max:255',
            'players.*.location' => 'required|string|max:255',
            'players.*.date_of_birth' => 'required|date|before:today',
            'players.*.gender' => 'required|in:male,female,other',
            
            // Officials
            'officials' => 'required|array|min:1|max:10',
            'officials.*.name' => 'required|string|max:255',
            'officials.*.email' => 'required|string|lowercase|email|max:255|distinct',
            'officials.*.phone' => 'required|string|max:20',
            'officials.*.role' => 'required|in:Coach,Assistant Coach,Manager,Physio,Doctor,Other',
            'officials.*.qualifications' => 'nullable|string|max:500',
            
            // Terms and conditions
            // 'terms_accepted' => 'required|accepted',
        ],[
            'captain.password.confirmed' => 'The password confirmation does not match.',    
            'players.*.jersey_number.numeric' => 'The jersey number must be a number.',
            'officials.*.qualifications.max' => 'The qualifications must be less than 500 characters.',
            'terms_accepted.accepted' => 'You must accept the terms and conditions.',
            'tournament_id.required' => 'The tournament is required.',
            'tournament_id.exists' => 'The selected tournament is not valid.',
            'team.name.required' => 'The team name is required.',
            'team.email.required' => 'The team email is required.',
            'team.phone.required' => 'The team phone is required.',
            'team.location.required' => 'The team location is required.',
            'team.year.required' => 'The team year is required.',
            'captain.name.required' => 'The captain name is required.',
            'captain.email.required' => 'The captain email is required.',
            'captain.phone.required' => 'The captain phone is required.',
            'players.*.name.required' => 'The player name is required.',
            'players.*.email.required' => 'The player email is required.',
            'players.*.phone.required' => 'The player phone is required.',
            'players.*.position.required' => 'The player position is required.',
            'officials.*.name.required' => 'The official name is required.',
            'officials.*.email.required' => 'The official email is required.',
            'officials.*.phone.required' => 'The official phone is required.',
            'officials.*.role.required' => 'The official role is required.',    
        ]);

        if ($validator->fails()) {
            dd($validator->errors());
            return back()->withErrors($validator)->withInput();
        }

        // Get tournament and validate category
        $tournament = \App\Models\Tournament::findOrFail($request->tournament_id);
        $selectedCategory = $request->team['category'] ?? 'men';

        // Check if category is enabled for this tournament
        if (!$tournament->isCategoryEnabled($selectedCategory)) {
            return back()->withErrors(['team.category' => 'The selected category is not available for this tournament.']);
        }

        // Validate all players match the selected category
        $categoryValidator = new \App\Services\CategoryValidationService();
        $validationErrors = [];

        foreach ($request->players as $index => $playerData) {
            $playerValidation = $categoryValidator->validatePlayerData($playerData, $selectedCategory, $tournament->start_date);
            
            if (!$playerValidation['valid']) {
                foreach ($playerValidation['errors'] as $error) {
                    $validationErrors["players.{$index}"] = "{$playerData['name']}: {$error}";
                }
            }
        }

        // If any player validation failed, return errors
        if (!empty($validationErrors)) {
            return back()->withErrors($validationErrors)->withInput();
        }

        $captain = null;

        // Check if captain email is already taken and send notification
        $existingCaptain = User::where('email', $request->captain['email'])->first();
        if ($existingCaptain) {
            // Send notification to existing user about team invitation
            // $existingCaptain->notify(new TeamCaptainInvitationNotification([
                // 'name' => $request->captain['name'],
                // 'phone' => $request->captain['phone'],
                // 'team_name' => $request->team['name'],
                // 'tournament_name' => \App\Models\Tournament::find($request->tournament_id)->name,
            // ]));
            
            // Use existing user as captain
            $captain = $existingCaptain;

            // Assign team manager role to existing captain if they don't have it
            $teamManagerRole = Role::where('name', 'team_manager')->first();
            if ($teamManagerRole && !$captain->hasRole('team_manager')) {
                $captain->assignRole($teamManagerRole, true); // true for is_primary
            }
        } else {
            // Set temporary registration data for the observer
            // \App\Observers\UserObserver::setTempRegistrationData($request->captain['password'], $request->captain['name']);
            
            // Create new user as captain
            $captain = User::create([
                'name' => $request->captain['name'],
                'email' => $request->captain['email'],
                'phone' => $request->captain['phone'],
                'password' => Hash::make($request->captain['password']),
                'user_type' => 'team_manager',
                'profile_complete' => false,
                'payment_status' => 'pending',
                'must_change_password' => true,
                'terms_accepted_at' => now(),
            ]);

            // Assign team manager role to the captain
            $teamManagerRole = Role::where('name', 'team_manager')->first();
            if ($teamManagerRole) {
                $captain->assignRole($teamManagerRole, true); // true for is_primary
            }
        }
        
        DB::beginTransaction();
        
        // Get the selected tournament
        $tournament = \App\Models\Tournament::findOrFail($request->tournament_id);
        if ($tournament->status !== 'active') {
            return back()->withErrors(['tournament_id' => 'Selected tournament is not active']);
        }

        // Check if team already exists
        $existingTeam = Team::where('name', $request->team['name'])->first();
        
        if ($existingTeam) {
            // Check if team is already registered for this tournament
            if ($existingTeam->isRegisteredForTournament($tournament)) {
                return back()->withErrors(['team.name' => 'This team is already registered for this tournament']);
            }
            
            // Use existing team
            $team = $existingTeam;
        } else {
            // Create new team
            $team = Team::create([
                'name' => $request->team['name'],
                'description' => "Team {$request->team['name']} - Founded {$request->team['year']}",
                'captain_id' => $captain->id,
                'status' => 'active',
                'category' => $selectedCategory,
            ]);
        }

        // Register team for tournament
        $team->registerForTournament($tournament, [
            'notes' => "Registered via public form",
            'tournament_data' => [
                'registration_method' => 'public_form',
                'team_email' => $request->team['email'],
                'team_phone' => $request->team['phone'],
                'team_country' => $request->team['country'],
                'team_county' => $request->team['county'],
                'team_location' => $request->team['location'],
                'team_year' => $request->team['year'],
            ]
        ]);

        // Add Team Tournament Registration Fee Payment (category-specific)
        $categoryFee = $tournament->getFeeForCategory($selectedCategory);
        $categoryName = $team->getCategoryName();
        
        \Log::info('Creating payment for team registration', [
            'team_id' => $team->id,
            'team_name' => $team->name,
            'tournament_id' => $tournament->id,
            'tournament_name' => $tournament->name,
            'category' => $selectedCategory,
            'category_name' => $categoryName,
            'category_fee' => $categoryFee,
            'tournament_entry_fee' => $tournament->entry_fee,
            'tournament_category_fees' => $tournament->category_fees,
        ]);
        
        $team->payments()->create([
            'tournament_id' => $tournament->id,
            'team_id' => $team->id,
            'user_id' => $captain->id,
            'amount' => $categoryFee,
            'status' => 'pending',
            'reference_number' => uniqid('PAY_'),
            'payment_method' => 'Mpesa',
            'payment_amount' => $categoryFee,
            'currency' => $tournament->currency ?? 'KES',
            'description' => "Tournament Registration Fee - {$tournament->name} ({$categoryName} Category)",
        ]);

        // Create players for this tournament (invite existing users; avoid duplicate player per tournament)
        $captainPlayerCreated = false;
        foreach ($request->players as $index => $playerData) {
            $isCaptain = $index === 0; // First player is captain
            $password = Str::random(12);
            
            if ($isCaptain && !$captainPlayerCreated) {
                // Create captain's player record for this tournament
                $captain->players()->create([
                    'tournament_id' => $tournament->id,
                    'team_id' => $team->id,
                    'display_name' => $playerData['name'],
                    'phone' => $playerData['phone'],
                    'football_position' => $playerData['position'],
                    'is_team_captain' => true,
                    'player_type' => 'amateur',
                    'experience_level' => 'intermediate',
                    'status' => 'registered',
                    'verification_status' => 'pending',
                    'otp_verified' => false,
                ]);
                $captainPlayerCreated = true;
            } else {
                // Check if player user already exists
                $existingPlayerUser = User::where('email', $playerData['email'])->first();

                if ($existingPlayerUser) {
                    $playerUser = $existingPlayerUser;

                    // Prevent duplicate player in same tournament
                    $alreadyPlayer = $playerUser->players()
                        ->where('tournament_id', $tournament->id)
                        ->exists();
                    if ($alreadyPlayer) {
                        // Skip creating duplicate player; send invitation instead
                        // $playerUser->notify(new \App\Notifications\User\PlayerInvitationNotification($team, $tournament));
                    } else {
                        // Create player record
                        $playerUser->players()->create([
                            'tournament_id' => $tournament->id,
                            'team_id' => $team->id,
                            'display_name' => $playerData['name'],
                            'phone' => $playerData['phone'],
                            'football_position' => $playerData['position'],
                            'jersey_number' => $playerData['jersey_number'],
                            'is_team_captain' => false,
                            'player_type' => 'amateur',
                            'experience_level' => 'beginner',
                            'status' => 'registered',
                            'verification_status' => 'pending',
                            'otp_verified' => false,
                        ]);
                        // Also send an invitation/notification of assignment
                        // $playerUser->notify(new \App\Notifications\User\PlayerInvitationNotification($team, $tournament));
                    }
                } else {
                    // Set temporary registration data for the observer
                    // \App\Observers\UserObserver::setTempRegistrationData($password, $playerData['name']);
                    
                    // Create a temporary user account for the player and create player record
                    $playerUser = User::create([
                        'name' => $playerData['name'],
                        'email' => $playerData['email'],
                        'phone' => $playerData['phone'],
                        'password' => Hash::make($password), // Temporary password
                        'user_type' => 'player',
                        'profile_complete' => false,
                        'payment_status' => 'pending',
                        'must_change_password' => true,
                        'terms_accepted_at' => now(),
                    ]);

                    $playerUser->players()->create([
                        'tournament_id' => $tournament->id,
                        'team_id' => $team->id,
                        'display_name' => $playerData['name'],
                        'phone' => $playerData['phone'],
                        'football_position' => $playerData['position'],
                        'jersey_number' => $playerData['jersey_number'],
                        'is_team_captain' => false,
                        'player_type' => 'amateur',
                        'experience_level' => 'beginner',
                        'status' => 'registered',
                        'verification_status' => 'pending',
                        'otp_verified' => false,
                    ]);
                }
            }
        }
        
        // Create team officials (invite existing users; allow multiple roles per user)
        foreach ($request->officials as $officialData) {
            $existingOfficialUser = User::where('email', $officialData['email'])->first();

            if ($existingOfficialUser) {
                // If user exists, send invitation and create official record for this role
                // $existingOfficialUser->notify(new \App\Notifications\User\OfficialInvitationNotification($team, $tournament, $officialData['role']));
            }

            TeamOfficial::create([
                'team_id' => $team->id,
                'name' => $officialData['name'],
                'email' => $officialData['email'],
                'phone' => $officialData['phone'],
                'role' => $officialData['role'],
                'qualifications' => $officialData['qualifications'] ?? null,
                'otp_verified' => false,
            ]);
        }
        
        DB::commit();
        
        // event(new Registered($captain));

        // Log the user in
        Auth::login($captain);

        return redirect()->route('profile.complete')->with('success', [
            'message' => 'Team registration successful! OTP codes have been sent to all team members.',
        ]);
        
    
    }
    
    /**
     * Handle an incoming user registration request (spectator).
     */
    public function storeUser(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|lowercase|email|max:255|unique:'.User::class,
            'phone' => 'required|string|max:20',
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'terms_accepted' => 'required|accepted',
        ]);

        // Set temporary registration data for the observer
        // \App\Observers\UserObserver::setTempRegistrationData($request->password, $request->name);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            'user_type' => 'spectator',
            'role' => 'spectator',
            'profile_complete' => true, // Spectators don't need additional profile completion
            'payment_status' => 'not_required', // Spectators don't need to pay
            'terms_accepted_at' => now(),
        ]);

        // event(new Registered($user));

        Auth::login($user);

        return redirect()->route('dashboard');
    }
    
    /**
     * Handle individual user registration (non-team)
     */
    public function storeIndividual(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|lowercase|email|max:255|unique:'.User::class,
            'phone' => 'required|string|max:20',
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'terms_accepted' => 'required|accepted',
        ]);

        // Set temporary registration data for the observer
        // \App\Observers\UserObserver::setTempRegistrationData($request->password, $request->name);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            'user_type' => 'individual',
            'profile_complete' => false,
            'payment_status' => 'pending',
            'terms_accepted_at' => now(),
        ]);

        // event(new Registered($user));

        Auth::login($user);

        return redirect()->route('profile.complete');
    }
}
