<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use App\Services\OtpService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Inertia\Inertia;
use Inertia\Response;

class AuthenticatedSessionController extends Controller
{
    protected $otpService;
    
    public function __construct(OtpService $otpService)
    {
        $this->otpService = $otpService;
    }
    
    /**
     * Show the login page.
     */
    public function create(Request $request): Response
    {
        return Inertia::render('frontend/auth/Login', [
            'canResetPassword' => Route::has('password.request'),
            'status' => $request->session()->get('status'),
        ]);
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        $request->authenticate();

        $request->session()->regenerate();
        
        $user = Auth::user();

        // If  user is not verified, send verification email
        if (!$user->hasVerifiedEmail()) {
            $user->sendEmailVerificationNotification();
            return redirect()->route('verification.notice')->with('message', 
                'Please verify your email to continue.'
            );
        }
        
        // Check if user must change password (new team members)
        // if ($user->must_change_password) {
        //     // Send OTP for password change verification
        //     $user->sendEmailVerificationNotification();     
        //     return redirect()->route('password.change.verify')->with('message', 
        //         'For security reasons, you must change your password. An OTP has been sent to your email.'
        //     );
        // }
        
        // Check if profile is incomplete
        if (!$user->profile_complete) {
            return redirect()->route('profile.complete')->with('message', 
                'Please complete your profile to continue.'
            );
        }
        
        // Check payment status
        if ($user->payment_status !== 'completed' && $user->payment_status !== 'not_required') {
            return redirect()->route('payment.required')->with('message', 
                'Payment is required to access the tournament system.'
            );
        }

        // All checks passed, redirect to dashboard
        return redirect()->intended(route('admin.dashboard', absolute: false));
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }
    
    /**
     * Show password change verification form
     */
    public function showPasswordChangeVerify(): Response
    {
        $user = Auth::user();
        
        if (!$user->must_change_password) {
            return redirect()->route('admin.dashboard');
        }
        
        return Inertia::render('frontend/auth/PasswordChangeVerify', [
            'user' => $user
        ]);
    }
    
    /**
     * Verify OTP and show password change form
     */
    public function verifyPasswordChangeOtp(Request $request): RedirectResponse
    {
        $request->validate([
            'otp' => 'required|string|size:6'
        ]);
        
        $user = Auth::user();
        
        if ($this->otpService->verifyUserOtp($user, $request->otp)) {
            return redirect()->route('password.change.form')->with('message', 
                'OTP verified successfully. Please set your new password.'
            );
        }
        
        return back()->withErrors(['otp' => 'Invalid or expired OTP code.']);
    }
    
    /**
     * Show new password form
     */
    public function showPasswordChangeForm(): Response
    {
        $user = Auth::user();
        
        if ($user->must_change_password && $user->otp_code) {
            return redirect()->route('password.change.verify');
        }
        
        return Inertia::render('frontend/auth/PasswordChangeForm', [
            'user' => $user
        ]);
    }
    
    /**
     * Update password
     */
    public function updatePassword(Request $request): RedirectResponse
    {
        $request->validate([
            'password' => ['required', 'confirmed', \Illuminate\Validation\Rules\Password::defaults()],
        ]);
        
        $user = Auth::user();
        
        $user->update([
            'password' => \Illuminate\Support\Facades\Hash::make($request->password),
            'must_change_password' => false,
        ]);
        
        return redirect()->route('profile.complete')->with('success', 
            'Password updated successfully. Please complete your profile.'
        );
    }
}
