<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class PaymentController extends Controller
{
    /**
     * Initiate a payment
     */
    public function initiate(Request $request)
    {
        return $request->all();
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0',
            'currency' => 'required|string|in:USD,KES,EUR,GBP',
            'team_data' => 'required|array',
            'payment_method' => 'required|string|in:card,bank_transfer,paypal,mpesa',
            'card_details' => 'sometimes|array',
        ], [
            'amount.required' => 'The amount field is required.',
            'amount.numeric' => 'The amount must be a number.',
            'amount.min' => 'The amount must be greater than 0.',
            'currency.required' => 'The currency field is required.',
            'currency.string' => 'The currency must be a string.',
            'currency.in' => 'The currency must be one of the following: USD, KES, EUR, GBP.',
            'team_data.required' => 'The team data field is required.',
            'payment_method.required' => 'The payment method field is required.',
            'payment_method.string' => 'The payment method must be a string.',
            'payment_method.in' => 'The payment method must be one of the following: card, bank_transfer, paypal, mpesa.',
            'card_details.array' => 'The card details must be an array.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Generate a unique transaction ID
            $transactionId = 'TXN_' . strtoupper(Str::random(12)) . '_' . time();

            // Store payment data in cache for simulation
            $paymentData = [
                'transaction_id' => $transactionId,
                'amount' => $validated['amount'],
                'currency' => $validated['currency'],
                'status' => 'processing',
                'payment_method' => $validated['payment_method'],
                'team_data' => $validated['team_data'],
                'created_at' => now()->toISOString(),
                'updated_at' => now()->toISOString(),
            ];

            // Store in cache for 1 hour
            Cache::put('payment_' . $transactionId, $paymentData, 3600);

            // Log payment initiation
            Log::info('Payment initiated', [
                'transaction_id' => $transactionId,
                'amount' => $validated['amount'],
                'payment_method' => $validated['payment_method']
            ]);

            // Simulate payment processing delay
            // In real implementation, this would be handled by payment gateway
            $this->simulatePaymentProcessing($transactionId);

            return response()->json([
                'success' => true,
                'message' => 'Payment initiated successfully',
                'transaction_id' => $transactionId,
                'status' => 'processing'
            ]);

        } catch (\Exception $e) {
            Log::error('Payment initiation failed', [
                'error' => $e->getMessage(),
                'amount' => $validated['amount']
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Payment initiation failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check payment status
     */
    public function status($transactionId)
    {
        try {
            $paymentData = Cache::get('payment_' . $transactionId);

            if (!$paymentData) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment not found'
                ], 404);
            }

            // Check if payment should be completed (simulation)
            $createdAt = new \DateTime($paymentData['created_at']);
            $now = new \DateTime();
            $secondsElapsed = $now->getTimestamp() - $createdAt->getTimestamp();

            // Simulate payment completion after 10-15 seconds
            if ($secondsElapsed >= 10 && $paymentData['status'] === 'processing') {
                // Simulate 95% success rate
                $success = mt_rand(1, 100) <= 95;
                
                $paymentData['status'] = $success ? 'completed' : 'failed';
                $paymentData['updated_at'] = now()->toISOString();
                
                if (!$success) {
                    $paymentData['failure_reason'] = 'Insufficient funds';
                }

                // Update cache
                Cache::put('payment_' . $transactionId, $paymentData, 3600);

                Log::info('Payment status updated', [
                    'transaction_id' => $transactionId,
                    'status' => $paymentData['status']
                ]);
            }

            return response()->json([
                'success' => true,
                'transaction_id' => $transactionId,
                'status' => $paymentData['status'],
                'amount' => $paymentData['amount'],
                'currency' => $paymentData['currency'],
                'message' => $this->getStatusMessage($paymentData['status']),
                'failure_reason' => $paymentData['failure_reason'] ?? null,
                'created_at' => $paymentData['created_at'],
                'updated_at' => $paymentData['updated_at']
            ]);

        } catch (\Exception $e) {
            Log::error('Payment status check failed', [
                'transaction_id' => $transactionId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Status check failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all payments (for admin)
     */
    public function index(Request $request)
    {
        // In real implementation, this would query from database
        // For simulation, we'll return cached payments
        
        $payments = [];
        $cacheKeys = Cache::get('payment_keys', []);
        
        foreach ($cacheKeys as $key) {
            $payment = Cache::get($key);
            if ($payment) {
                $payments[] = $payment;
            }
        }

        // Sort by created_at desc
        usort($payments, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });

        return response()->json([
            'success' => true,
            'payments' => $payments,
            'total' => count($payments)
        ]);
    }

    /**
     * Refund a payment
     */
    public function refund(Request $request, $transactionId)
    {
        $validated = $request->validate([
            'amount' => 'sometimes|numeric|min:0',
            'reason' => 'required|string|max:255'
        ]);

        try {
            $paymentData = Cache::get('payment_' . $transactionId);

            if (!$paymentData) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment not found'
                ], 404);
            }

            if ($paymentData['status'] !== 'completed') {
                return response()->json([
                    'success' => false,
                    'message' => 'Can only refund completed payments'
                ], 400);
            }

            $refundAmount = $validated['amount'] ?? $paymentData['amount'];

            // Create refund record
            $refundId = 'REF_' . strtoupper(Str::random(12)) . '_' . time();
            $refundData = [
                'refund_id' => $refundId,
                'original_transaction_id' => $transactionId,
                'amount' => $refundAmount,
                'reason' => $validated['reason'],
                'status' => 'completed',
                'created_at' => now()->toISOString()
            ];

            Cache::put('refund_' . $refundId, $refundData, 3600);

            // Update original payment
            $paymentData['status'] = 'refunded';
            $paymentData['refund_id'] = $refundId;
            $paymentData['refund_amount'] = $refundAmount;
            $paymentData['updated_at'] = now()->toISOString();

            Cache::put('payment_' . $transactionId, $paymentData, 3600);

            Log::info('Payment refunded', [
                'transaction_id' => $transactionId,
                'refund_id' => $refundId,
                'amount' => $refundAmount
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Refund processed successfully',
                'refund_id' => $refundId,
                'refund_amount' => $refundAmount
            ]);

        } catch (\Exception $e) {
            Log::error('Refund failed', [
                'transaction_id' => $transactionId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Refund failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Simulate payment processing
     */
    private function simulatePaymentProcessing($transactionId)
    {
        // In real implementation, this would trigger actual payment processing
        // For simulation, we'll just log it
        Log::info('Simulating payment processing', [
            'transaction_id' => $transactionId
        ]);
    }

    /**
     * Get status message
     */
    private function getStatusMessage($status)
    {
        switch ($status) {
            case 'pending':
                return 'Payment is pending';
            case 'processing':
                return 'Payment is being processed';
            case 'completed':
                return 'Payment completed successfully';
            case 'failed':
                return 'Payment failed';
            case 'refunded':
                return 'Payment has been refunded';
            default:
                return 'Unknown status';
        }
    }
}