<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class UserAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:users.view')->only(['index', 'show', 'export']);
        $this->middleware('permission:users.create')->only(['create', 'store']);
        $this->middleware('permission:users.edit')->only(['edit', 'update']);
        $this->middleware('permission:users.delete')->only(['destroy', 'bulkDelete']);
        $this->middleware('permission:roles.assign')->only(['assignRoles', 'removeRoles']);
        $this->middleware('permission:users.impersonate')->only(['impersonate']);
    }

    public function index(Request $request)
    {
        $query = User::with(['roles']);

        // Filtering
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('role') && $request->role !== 'all') {
            $query->whereHas('roles', function($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            if ($request->status === 'verified') {
                $query->whereNotNull('email_verified_at');
            } else {
                $query->whereNull('email_verified_at');
            }
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $users = $query->paginate($request->get('per_page', 15))->withQueryString();

        // Get roles for filtering
        $roles = Role::active()->get();

        return Inertia::render('admin/users/index', [
            'users' => $users,
            'roles' => $roles,
            'filters' => $request->only(['search', 'role', 'status', 'sort_by', 'sort_order']),
        ]);
    }

    public function show(User $user)
    {
        $user->load(['roles', 'players', 'teams']);
        
        return Inertia::render('admin/users/show', [
            'user' => $user,
        ]);
    }

    public function create()
    {
        // Exclude super_admin role from available roles for creation
        $roles = Role::active()->where('name', '!=', 'super_admin')->get();
        
        return Inertia::render('admin/users/create', [
            'roles' => $roles,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'roles' => 'nullable|array',
            'roles.*' => 'exists:roles,id',
            'must_change_password' => 'nullable|boolean',
            'payment_status' => 'nullable|string|in:pending,paid,failed,refunded,not_required',
            'user_type' => 'nullable|string|in:team,player,official',
            // KYC fields (all optional)
            'phone' => 'nullable|string|max:20',
            'id_number' => 'nullable|string|max:20',
            'passport_number' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|string|in:male,female,other',
            'nationality' => 'nullable|string|max:100',
            'county' => 'nullable|string|max:100',
            'sub_county' => 'nullable|string|max:100',
            'ward' => 'nullable|string|max:100',
            'address' => 'nullable|string',
            'postal_code' => 'nullable|string|max:10',
            'occupation' => 'nullable|string|max:100',
            'employer' => 'nullable|string|max:100',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relationship' => 'nullable|string|max:100',
            'bio' => 'nullable|string',
            'kyc_status' => 'nullable|string|in:pending,verified,rejected,not_submitted',
            'kyc_notes' => 'nullable|string',
        ]);

        try {
            // Set temporary registration data for the observer
            // \App\Observers\UserObserver::setTempRegistrationData($validated['password'], $validated['name']);
            
            $userData = [
                'name' => $validated['name'],
                'email' => $validated['email'],
                'password' => Hash::make($validated['password']),
                'email_verified_at' => now(), // Auto-verify admin created users
                'must_change_password' => $validated['must_change_password'] ?? false,
                'payment_status' => $validated['payment_status'] ?? 'pending',
                'user_type' => $validated['user_type'] ?? 'player',
                'profile_complete' => false,
                'terms_accepted_at' => now(),
                // KYC fields
                'phone' => $validated['phone'] ?? null,
                'id_number' => $validated['id_number'] ?? null,
                'passport_number' => $validated['passport_number'] ?? null,
                'date_of_birth' => $validated['date_of_birth'] ?? null,
                'gender' => $validated['gender'] ?? null,
                'nationality' => $validated['nationality'] ?? null,
                'county' => $validated['county'] ?? null,
                'sub_county' => $validated['sub_county'] ?? null,
                'ward' => $validated['ward'] ?? null,
                'address' => $validated['address'] ?? null,
                'postal_code' => $validated['postal_code'] ?? null,
                'occupation' => $validated['occupation'] ?? null,
                'employer' => $validated['employer'] ?? null,
                'emergency_contact_name' => $validated['emergency_contact_name'] ?? null,
                'emergency_contact_phone' => $validated['emergency_contact_phone'] ?? null,
                'emergency_contact_relationship' => $validated['emergency_contact_relationship'] ?? null,
                'bio' => $validated['bio'] ?? null,
                'kyc_status' => $validated['kyc_status'] ?? 'not_submitted',
                'kyc_notes' => $validated['kyc_notes'] ?? null,
            ];

            $user = User::create($userData);

            if (!empty($validated['roles'])) {
                foreach ($validated['roles'] as $roleId) {
                    $role = Role::find($roleId);
                    if ($role) {
                        $user->assignRole($role);
                    }
                }
            }

            // Return JSON response for AJAX requests (like from player creation)
            if ($request->expectsJson() || $request->header('Accept') === 'application/json') {
                return response()->json([
                    'success' => true,
                    'message' => 'User created successfully!',
                    'user' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                    ]
                ]);
            }

            // Return redirect for regular form submissions
            return redirect()->route('admin.users.index')->with('success', 'User created successfully!');

        } catch (\Throwable $th) {
            if ($request->expectsJson() || $request->header('Accept') === 'application/json') {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create user',
                    'error' => $th->getMessage()
                ], 422);
            }
            
            return redirect()->back()->withErrors(['general' => 'Failed to create user: ' . $th->getMessage()])->withInput();
        }
    }

    public function edit(User $user)
    {
        // Prevent editing super admin users
        if ($user->hasRole('super_admin')) {
            return redirect()->route('admin.users.index')->with('error', 'Super Administrator accounts cannot be edited through the interface.');
        }

        $user->load('roles');
        // Exclude super_admin role from available roles for editing
        $roles = Role::active()->where('name', '!=', 'super_admin')->get();
        
        return Inertia::render('admin/users/edit', [
            'user' => $user,
            'roles' => $roles,
        ]);
    }

    public function update(Request $request, User $user)
    {
        // Prevent updating super admin users
        if ($user->hasRole('super_admin')) {
            return redirect()->route('admin.users.index')->with('error', 'Super Administrator accounts cannot be updated through the interface.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'password' => 'nullable|string|min:8|confirmed',
            'roles' => 'array',
            'roles.*' => 'exists:roles,id',
            'email_verified_at' => 'nullable|boolean',
            // KYC fields (all optional)
            'phone' => 'nullable|string|max:20',
            'id_number' => 'nullable|string|max:20',
            'passport_number' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|string|in:male,female,other',
            'nationality' => 'nullable|string|max:100',
            'county' => 'nullable|string|max:100',
            'sub_county' => 'nullable|string|max:100',
            'ward' => 'nullable|string|max:100',
            'address' => 'nullable|string',
            'postal_code' => 'nullable|string|max:10',
            'occupation' => 'nullable|string|max:100',
            'employer' => 'nullable|string|max:100',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relationship' => 'nullable|string|max:100',
            'bio' => 'nullable|string',
            'kyc_status' => 'nullable|string|in:pending,verified,rejected,not_submitted',
            'kyc_notes' => 'nullable|string',
        ]);

        // Prevent assigning super_admin role
        if (!empty($validated['roles'])) {
            $superAdminRole = Role::where('name', 'super_admin')->first();
            if ($superAdminRole && in_array($superAdminRole->id, $validated['roles'])) {
                return redirect()->back()->withErrors(['roles' => 'Super Administrator role cannot be assigned through the interface.'])->withInput();
            }
        }

        $updateData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            // KYC fields
            'phone' => $validated['phone'] ?? $user->phone,
            'id_number' => $validated['id_number'] ?? $user->id_number,
            'passport_number' => $validated['passport_number'] ?? $user->passport_number,
            'date_of_birth' => $validated['date_of_birth'] ?? $user->date_of_birth,
            'gender' => $validated['gender'] ?? $user->gender,
            'nationality' => $validated['nationality'] ?? $user->nationality,
            'county' => $validated['county'] ?? $user->county,
            'sub_county' => $validated['sub_county'] ?? $user->sub_county,
            'ward' => $validated['ward'] ?? $user->ward,
            'address' => $validated['address'] ?? $user->address,
            'postal_code' => $validated['postal_code'] ?? $user->postal_code,
            'occupation' => $validated['occupation'] ?? $user->occupation,
            'employer' => $validated['employer'] ?? $user->employer,
            'emergency_contact_name' => $validated['emergency_contact_name'] ?? $user->emergency_contact_name,
            'emergency_contact_phone' => $validated['emergency_contact_phone'] ?? $user->emergency_contact_phone,
            'emergency_contact_relationship' => $validated['emergency_contact_relationship'] ?? $user->emergency_contact_relationship,
            'bio' => $validated['bio'] ?? $user->bio,
            'kyc_status' => $validated['kyc_status'] ?? $user->kyc_status,
            'kyc_notes' => $validated['kyc_notes'] ?? $user->kyc_notes,
        ];

        // Update password if provided
        if (!empty($validated['password'])) {
            $updateData['password'] = Hash::make($validated['password']);
        }

        // Update email verification status
        if (isset($validated['email_verified_at'])) {
            $updateData['email_verified_at'] = $validated['email_verified_at'] ? now() : null;
        }

        $user->update($updateData);

        // Update roles
        if (isset($validated['roles'])) {
            $user->roles()->detach();
            foreach ($validated['roles'] as $roleId) {
                $role = Role::find($roleId);
                if ($role) {
                    $user->assignRole($role);
                }
            }
        }

        return redirect()->route('admin.users.index')->with('success', 'User updated successfully!');
    }

    public function destroy(User $user)
    {
        // Prevent deleting the current user
        if ($user->id === auth()->id()) {
            return redirect()->back()->with('error', 'You cannot delete your own account.');
        }

        // Prevent deleting super admin users
        if ($user->hasRole('super_admin')) {
            return redirect()->back()->with('error', 'Super Administrator accounts cannot be deleted through the interface.');
        }

        $user->delete();

        return redirect()->route('admin.users.index')->with('success', 'User deleted successfully!');
    }

    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:users,id',
        ]);

        // Remove current user and super admin users from deletion list
        $userIds = array_filter($validated['ids'], function($id) {
            if ($id == auth()->id()) return false;
            
            $user = User::find($id);
            return !$user || !$user->hasRole('super_admin');
        });

        User::whereIn('id', $userIds)->delete();

        return redirect()->back()->with('success', count($userIds) . ' users deleted successfully!');
    }

    public function assignRoles(Request $request, User $user)
    {
        $validated = $request->validate([
            'role_ids' => 'required|array',
            'role_ids.*' => 'exists:roles,id',
        ]);

        foreach ($validated['role_ids'] as $roleId) {
            $role = Role::find($roleId);
            if ($role && !$user->hasRole($role->name)) {
                $user->assignRole($role);
            }
        }

        return redirect()->back()->with('success', 'Roles assigned successfully!');
    }

    public function removeRoles(Request $request, User $user)
    {
        $validated = $request->validate([
            'role_ids' => 'required|array',
            'role_ids.*' => 'exists:roles,id',
        ]);

        foreach ($validated['role_ids'] as $roleId) {
            $role = Role::find($roleId);
            if ($role) {
                $user->removeRole($role);
            }
        }

        return redirect()->back()->with('success', 'Roles removed successfully!');
    }

    public function impersonate(User $user)
    {
        // Store the original user ID and start time in session
        session([
            'impersonate_original_user' => auth()->id(),
            'impersonation_started_at' => now()->toISOString()
        ]);
        
        // Login as the target user
        auth()->login($user);

        return redirect()->route('dashboard')->with('success', "You are now impersonating {$user->name}");
    }

    public function stopImpersonating()
    {
        $originalUserId = session('impersonate_original_user');
        
        if (!$originalUserId) {
            return redirect()->route('dashboard')->with('error', 'No active impersonation session found.');
        }

        $originalUser = User::find($originalUserId);
        if (!$originalUser) {
            session()->forget(['impersonate_original_user', 'impersonation_started_at']);
            return redirect()->route('dashboard')->with('error', 'Original user not found. Impersonation session cleared.');
        }

        // Log back in as the original user
        auth()->login($originalUser);
        session()->forget(['impersonate_original_user', 'impersonation_started_at']);
        
        return redirect()->route('admin.users.index')->with('success', 'Impersonation stopped successfully.');
    }

    public function export(Request $request)
    {
        $query = User::with(['roles']);

        // Apply same filters as index
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('role') && $request->role !== 'all') {
            $query->whereHas('roles', function($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        $users = $query->get();

        $csvData = [];
        $csvData[] = ['ID', 'Name', 'Email', 'Roles', 'Email Verified', 'Created At'];

        foreach ($users as $user) {
            $csvData[] = [
                $user->id,
                $user->name,
                $user->email,
                $user->roles->pluck('display_name')->join(', '),
                $user->email_verified_at ? 'Yes' : 'No',
                $user->created_at->format('Y-m-d H:i:s'),
            ];
        }

        $filename = 'users_export_' . now()->format('Y_m_d_H_i_s') . '.csv';
        
        $handle = fopen('php://output', 'w');
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        foreach ($csvData as $row) {
            fputcsv($handle, $row);
        }

        fclose($handle);
        exit;
    }
}
