<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;
use Inertia\Response;
use Illuminate\Support\Facades\Validator;

class TournamentSettingsController extends Controller
{
    /**
     * Display the tournament settings page
     */
    public function index(): Response
    {
        $settings = $this->getCurrentSettings();
        
        // Get active tournament for comparison
        $activeTournament = \App\Models\Tournament::where('status', 'active')->first();
        
        // Get users with approver roles
        $firstApprovers = User::whereHas('roles', function($query) {
            $query->whereIn('name', ['first_approver', 'organizer', 'admin', 'super_admin']);
        })->select('id', 'name', 'email')->get();
        
        $secondApprovers = User::whereHas('roles', function($query) {
            $query->whereIn('name', ['second_approver', 'organizer', 'admin', 'super_admin']);
        })->select('id', 'name', 'email')->get();
        
        // Get current default approvers from environment
        $currentFirstApproverId = env('TOURNAMENT_DEFAULT_FIRST_APPROVER_ID');
        $currentSecondApproverId = env('TOURNAMENT_DEFAULT_SECOND_APPROVER_ID');
        
        return Inertia::render('admin/tournament-settings/Index', [
            'settings' => $settings,
            'active_tournament' => $activeTournament ? [
                'id' => $activeTournament->id,
                'name' => $activeTournament->name,
                'entry_fee' => $activeTournament->entry_fee,
                'currency' => $activeTournament->currency ?? 'KES'
            ] : null,
            'firstApprovers' => $firstApprovers,
            'secondApprovers' => $secondApprovers,
            'currentFirstApproverId' => $currentFirstApproverId,
            'currentSecondApproverId' => $currentSecondApproverId,
        ]);
    }

    /**
     * Update tournament settings
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'registration_fee' => 'required|numeric|min:0|max:999999',
            'max_players_per_team' => 'required|integer|min:1|max:50',
            'min_players_per_team' => 'required|integer|min:1|max:50',
            'max_officials_per_team' => 'required|integer|min:0|max:20',
            'registration_start' => 'nullable|date',
            'registration_end' => 'nullable|date|after:registration_start',
            'tournament_start' => 'nullable|date',
            'tournament_end' => 'nullable|date|after:tournament_start',
            'allow_jersey_number_change' => 'boolean',
            'require_team_logo' => 'boolean',
            'payment_deadline_days' => 'required|integer|min:1|max:90',
            'allow_partial_payments' => 'boolean',
            'otp_expiry_hours' => 'required|integer|min:1|max:168',
            'max_otp_resends' => 'required|integer|min:1|max:10',
            'default_first_approver_id' => 'nullable|exists:users,id',
            'default_second_approver_id' => 'nullable|exists:users,id',
            'available_countries' => 'required|string|min:3',
            'available_counties' => 'required|string|min:3',
            'category_locations' => 'nullable|array',
            'category_locations.*.restrict_locations' => 'nullable|boolean',
            'category_locations.*.locations' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

            $validated = $validator->validated();

        try {
            // Update environment variables
            $this->updateEnvironmentVariables($validated);
            
            return redirect()->back()->with('success', 'Tournament settings updated successfully!');
        } catch (\Exception $e) {
            Log::error('Failed to update tournament settings: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to update settings. Please try again.');
        }
    }

    /**
     * Get current tournament settings
     */
    private function getCurrentSettings(): array
    {
        return [
            'registration_fee' => config('tournament.registration_fee', 1000),
            'max_players_per_team' => config('tournament.max_players_per_team', 23),
            'min_players_per_team' => config('tournament.min_players_per_team', 11),
            'max_officials_per_team' => config('tournament.max_officials_per_team', 10),
            'registration_start' => config('tournament.registration_start'),
            'registration_end' => config('tournament.registration_end'),
            'tournament_start' => config('tournament.tournament_start'),
            'tournament_end' => config('tournament.tournament_end'),
            'terms_version' => config('tournament.terms_version', '1.0'),
            'terms_last_updated' => config('tournament.terms_last_updated'),
            'allow_jersey_number_change' => config('tournament.allow_jersey_number_change', false),
            'require_team_logo' => config('tournament.require_team_logo', false),
            'payment_deadline_days' => config('tournament.payment_deadline_days', 7),
            'allow_partial_payments' => config('tournament.allow_partial_payments', false),
            'otp_expiry_hours' => config('tournament.otp_expiry_hours', 24),
            'max_otp_resends' => config('tournament.max_otp_resends', 3),
            'default_first_approver_id' => config('tournament.default_first_approver_id'),
            'default_second_approver_id' => config('tournament.default_second_approver_id'),
            'available_countries' => config('tournament.available_countries', 'Kenya'),
            'available_counties' => config('tournament.available_counties', 'Kakamega County,Nairobi County,Mombasa County'),
            'category_locations' => config('tournament.category_locations', []),
            'categories' => config('tournament.categories', []),
        ];
    }

    /**
     * Update environment variables
     */
    private function updateEnvironmentVariables(array $settings): void
    {
        $envFile = base_path('.env');
        
        if (!file_exists($envFile)) {
            throw new \Exception('.env file not found');
        }

        $envContent = file_get_contents($envFile);
        
        // Handle category_locations separately (update config file)
        if (isset($settings['category_locations'])) {
            $this->updateConfigFile($settings['category_locations']);
        }

        $envMappings = [
            'registration_fee' => 'TOURNAMENT_REGISTRATION_FEE',
            'max_players_per_team' => 'TOURNAMENT_MAX_PLAYERS',
            'min_players_per_team' => 'TOURNAMENT_MIN_PLAYERS',
            'max_officials_per_team' => 'TOURNAMENT_MAX_OFFICIALS',
            'registration_start' => 'TOURNAMENT_REGISTRATION_START',
            'registration_end' => 'TOURNAMENT_REGISTRATION_END',
            'tournament_start' => 'TOURNAMENT_START_DATE',
            'tournament_end' => 'TOURNAMENT_END_DATE',
            'allow_jersey_number_change' => 'TOURNAMENT_ALLOW_JERSEY_CHANGE',
            'require_team_logo' => 'TOURNAMENT_REQUIRE_LOGO',
            'payment_deadline_days' => 'TOURNAMENT_PAYMENT_DEADLINE',
            'allow_partial_payments' => 'TOURNAMENT_ALLOW_PARTIAL',
            'otp_expiry_hours' => 'TOURNAMENT_OTP_EXPIRY',
            'max_otp_resends' => 'TOURNAMENT_MAX_OTP_RESENDS',
            'available_countries' => 'TOURNAMENT_COUNTRIES',
            'available_counties' => 'TOURNAMENT_COUNTIES',
            'default_first_approver_id' => 'TOURNAMENT_DEFAULT_FIRST_APPROVER_ID',
            'default_second_approver_id' => 'TOURNAMENT_DEFAULT_SECOND_APPROVER_ID',
        ];

        foreach ($settings as $key => $value) {
            if (isset($envMappings[$key])) {
                $envKey = $envMappings[$key];
                
                // Convert boolean values
                if (is_bool($value)) {
                    $value = $value ? 'true' : 'false';
                }
                
                // Handle null values
                if (is_null($value)) {
                    $value = '';
                }
                
                // Update or add the environment variable
                $pattern = "/^{$envKey}=.*$/m";
                $replacement = "{$envKey}=\"{$value}\"";
                
                if (preg_match($pattern, $envContent)) {
                    $envContent = preg_replace($pattern, $replacement, $envContent);
                } else {
                    $envContent .= "\n{$replacement}";
                }
            }
        }

        file_put_contents($envFile, $envContent);
        
        // Clear and rebuild config cache to reflect changes
        $this->updateConfigCache();
    }

    /**
     * Update config file with category locations
     */
    private function updateConfigFile(array $categoryLocations): void
    {
        $configFile = config_path('tournament.php');
        
        if (!file_exists($configFile)) {
            throw new \Exception('tournament.php config file not found');
        }

        $configContent = file_get_contents($configFile);
        
        // Build the category_locations PHP array string
        $locationsArray = var_export($categoryLocations, true);
        
        // Replace the category_locations array in the config file
        $pattern = "/'category_locations'\s*=>\s*\[[\s\S]*?\],\s*\n\s*\/\//";
        $replacement = "'category_locations' => " . $locationsArray . ",\n\n    //";
        
        $configContent = preg_replace($pattern, $replacement, $configContent);
        
        file_put_contents($configFile, $configContent);
        
        Log::info('Config file updated with new category locations');
        
        // Clear and rebuild config cache after updating config file
        $this->updateConfigCache();
    }

    /**
     * Update config cache after environment changes
     */
    private function updateConfigCache(): void
    {
        // Clear config cache
        \Artisan::call('config:clear');
        
        // Rebuild config cache for better performance
        \Artisan::call('config:cache');
        
        // Also reset OPcache if available
        if (function_exists('opcache_reset')) {
            opcache_reset();
        }
        
        Log::info('Config cache updated after tournament settings change');
    }
}