<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Tournament;
use App\Models\Team;
use App\Models\GameMatch;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Validator;

class TournamentAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:tournaments.view')->only(['index', 'show', 'statistics']);
        $this->middleware('permission:tournaments.create')->only(['create', 'store']);
        $this->middleware('permission:tournaments.edit')->only(['edit', 'update', 'publish', 'complete', 'cancel', 'updateMultiple']);
        $this->middleware('permission:tournaments.delete')->only(['destroy', 'destroyMultiple']);
        $this->middleware('permission:tournaments.categories.manage')->only(['update', 'store']);
    }

    public function index(Request $request)
    {
        $query = Tournament::query();

        // Basic filters
        if ($request->filled('search')) {
            $query->where(function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%')
                  ->orWhere('location', 'like', '%' . $request->search . '%')
                  ->orWhere('organizer', 'like', '%' . $request->search . '%');
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->filled('type') && $request->type !== 'all') {
            $query->where('type', $request->type);
        }

        if ($request->filled('format') && $request->format !== 'all') {
            $query->where('format', $request->format);
        }

        if ($request->filled('visibility') && $request->visibility !== 'all') {
            $query->where('visibility', $request->visibility);
        }

        if ($request->filled('registration_status') && $request->registration_status !== 'all') {
            $query->where('registration_status', $request->registration_status);
        }

        if ($request->filled('game_title') && $request->game_title !== 'all') {
            $query->where('game_title', $request->game_title);
        }

        if ($request->filled('location') && $request->location !== 'all') {
            $query->where('location', $request->location);
        }

        // Date filters
        if ($request->filled('start_date_from')) {
            $query->where('start_date', '>=', $request->start_date_from);
        }

        if ($request->filled('start_date_to')) {
            $query->where('start_date', '<=', $request->start_date_to);
        }

        // Prize pool filters
        if ($request->filled('min_prize_pool')) {
            $query->where('prize_pool', '>=', $request->min_prize_pool);
        }

        if ($request->filled('max_prize_pool')) {
            $query->where('prize_pool', '<=', $request->max_prize_pool);
        }

        // Participant filters
        if ($request->filled('min_participants')) {
            $query->where('total_participants', '>=', $request->min_participants);
        }

        if ($request->filled('max_participants')) {
            $query->where('total_participants', '<=', $request->max_participants);
        }

        // Featured filter
        if ($request->filled('is_featured') && $request->is_featured !== 'all') {
            $query->where('is_featured', $request->is_featured === 'true');
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        
        $allowedSortFields = [
            'name', 'status', 'type', 'format', 'start_date', 'end_date', 
            'total_participants', 'prize_pool', 'created_at', 'updated_at'
        ];
        
        if (in_array($sortBy, $allowedSortFields)) {
            $query->orderBy($sortBy, $sortOrder);
        } else {
            $query->orderByDesc('created_at');
        }

        $tournaments = $query->paginate(15)->withQueryString();

        // Get filter options
        $locations = Tournament::whereNotNull('location')->distinct()->pluck('location');
        $gameTitles = Tournament::whereNotNull('game_title')->distinct()->pluck('game_title');
        $formats = ['individual', 'team', 'mixed'];
        $types = ['single_elimination', 'double_elimination', 'round_robin'];
        $statuses = ['draft', 'active', 'completed', 'cancelled'];
        $visibilities = ['public', 'private', 'unlisted'];
        $registrationStatuses = ['open', 'closed', 'full'];

        return Inertia::render('admin/tournaments/index', [
            'tournaments' => $tournaments,
            'filters' => $request->only([
                'search', 'status', 'type', 'format', 'visibility', 'registration_status',
                'game_title', 'location', 'start_date_from', 'start_date_to',
                'min_prize_pool', 'max_prize_pool', 'min_participants', 'max_participants',
                'is_featured', 'sort_by', 'sort_order'
            ]),
            'filterOptions' => [
                'locations' => $locations,
                'gameTitles' => $gameTitles,
                'formats' => $formats,
                'types' => $types,
                'statuses' => $statuses,
                'visibilities' => $visibilities,
                'registrationStatuses' => $registrationStatuses,
            ],
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/tournaments/create');
    }

    public function store(Request $request)
    {
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:tournaments,slug',
            'description' => 'nullable|string',
            'location' => 'nullable|string|max:255',
            'venue' => 'nullable|string|max:255',
            'organizer' => 'nullable|string|max:255',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:20',
            'website' => 'nullable|url|max:255',
            'type' => 'required|in:knockout,elimination,round-robin,group-knockout,double-elimination',
            'status' => 'required|in:draft,active,completed,cancelled',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'registration_start' => 'nullable|date',
            'registration_end' => 'nullable|date|after_or_equal:registration_start',
            'min_participants' => 'nullable|integer|min:1|max:1000',
            'max_participants' => 'nullable|integer|min:1|max:1000|gte:min_participants',
            'min_players_per_team' => 'nullable|integer|min:1|max:100',
            'max_players_per_team' => 'nullable|integer|min:1|max:100|gte:min_players_per_team',
            'auto_accept_registrations' => 'boolean',
            'registration_status' => 'required|in:open,closed,full',
            'game_title' => 'nullable|string|max:255',
            'game_version' => 'nullable|string|max:100',
            'format' => 'required|in:individual,team,mixed',
            'team_size' => 'nullable|integer|min:1|max:50',
            'rounds' => 'nullable|integer|min:1|max:100',
            'matches_per_round' => 'nullable|integer|min:1|max:1000',
            'bracket_settings' => 'nullable|array',
            'stream_url' => 'nullable|url|max:255',
            'youtube_url' => 'nullable|url|max:255',
            'twitch_url' => 'nullable|url|max:255',
            'is_streamed' => 'boolean',
            'is_featured' => 'boolean',
            'entry_fee' => 'nullable|numeric|min:0|max:99999.99',
            'prize_pool' => 'nullable|numeric|min:0|max:9999999.99',
            'prize_distribution' => 'nullable|array',
            'currency' => 'nullable|string|size:3',
            'sponsor_info' => 'nullable|array',
            'rewards' => 'nullable|array',
            'banner_image' => 'nullable|string|max:255',
            'logo_image' => 'nullable|string|max:255',
            'social_links' => 'nullable|array',
            'announcement' => 'nullable|string',
            'visibility' => 'required|in:public,private,unlisted',
            'allow_spectators' => 'boolean',
            'require_approval' => 'boolean',
            'enable_categories' => 'boolean',
            'category_fees' => 'nullable|array',
            'category_fees.men' => 'nullable|numeric|min:0',
            'category_fees.women' => 'nullable|numeric|min:0',
            'category_fees.youth_male' => 'nullable|numeric|min:0',
            'category_fees.youth_female' => 'nullable|numeric|min:0',
            'category_fees.children' => 'nullable|numeric|min:0',
            'available_categories' => 'nullable|array',
            'available_categories.*' => 'in:men,women,youth_male,youth_female,children',
            'admin_notes' => 'nullable|array',
            'metadata' => 'nullable|array',
            'rules' => 'nullable|array',
            'settings' => 'nullable|array',
        ]);

        // Set default values
        if (empty($validated['slug'])) {
            $validated['slug'] = \Illuminate\Support\Str::slug($validated['name']);
        }

        // // Set default approvers from environment if two-tier approval is enabled
        // if ($validated['enable_two_tier_approval'] ?? false) {
        //     if (empty($validated['first_approver_id']) && env('TOURNAMENT_DEFAULT_FIRST_APPROVER_ID')) {
        //         $validated['first_approver_id'] = (int) env('TOURNAMENT_DEFAULT_FIRST_APPROVER_ID');
        //     }
        //     if (empty($validated['second_approver_id']) && env('TOURNAMENT_DEFAULT_SECOND_APPROVER_ID')) {
        //         $validated['second_approver_id'] = (int) env('TOURNAMENT_DEFAULT_SECOND_APPROVER_ID');
        //     }
        // }

        Tournament::create($validated);
        return redirect()->route('admin.tournaments.index')->with('success', 'Tournament created successfully!');
    }

    public function edit(Tournament $tournament)
    {
        // Load tournament with approvers and teams count
        $tournament->load(['firstApprover', 'secondApprover']);
        $tournament->loadCount('teams');
        
        // Get user details for the tournament's current approvers or environment defaults
        $firstApprovers = [];
        $secondApprovers = [];
        
        // Get first approver (tournament's or environment default)
        $firstApproverId = $tournament->first_approver_id ?: env('TOURNAMENT_DEFAULT_FIRST_APPROVER_ID');
        if ($firstApproverId) {
            $firstApprover = User::find($firstApproverId);
            if ($firstApprover) {
                $firstApprovers = [['id' => $firstApprover->id, 'name' => $firstApprover->name, 'email' => $firstApprover->email]];
            }
        }
        
        // Get second approver (tournament's or environment default)
        $secondApproverId = $tournament->second_approver_id ?: env('TOURNAMENT_DEFAULT_SECOND_APPROVER_ID');
        if ($secondApproverId) {
            $secondApprover = User::find($secondApproverId);
            if ($secondApprover) {
                $secondApprovers = [['id' => $secondApprover->id, 'name' => $secondApprover->name, 'email' => $secondApprover->email]];
            }
        }
        
        return Inertia::render('admin/tournaments/edit', [
            'tournament' => $tournament,
            'firstApprovers' => $firstApprovers,
            'secondApprovers' => $secondApprovers,
        ]);
    }

    /**
     * Get users by role for dropdown selection
     */
    public function getUsersByRole(Request $request)
    {
        $role = $request->get('role');
        
        if (!$role) {
            return response()->json([]);
        }
        
        $users = User::whereHas('roles', function($query) use ($role) {
            $query->whereIn('name', [$role, 'admin', 'super_admin']);
        })->select('id', 'name', 'email')->get();
        
        return response()->json($users);
    }

    public function update(Request $request, Tournament $tournament)
    {
        // Set default approvers from environment if two-tier approval is enabled
        // if ($request->enable_two_tier_approval ?? false) {
        //     if (empty($request->first_approver_id) && env('TOURNAMENT_DEFAULT_FIRST_APPROVER_ID')) {
        //         $request->merge(['first_approver_id' => (int) env('TOURNAMENT_DEFAULT_FIRST_APPROVER_ID')]);
        //     }
        //     if (empty($request->second_approver_id) && env('TOURNAMENT_DEFAULT_SECOND_APPROVER_ID')) {
        //         $request->merge(['second_approver_id' => (int) env('TOURNAMENT_DEFAULT_SECOND_APPROVER_ID')]);
        //     }
        // }

        $validated = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:tournaments,slug,' . $tournament->id,
            'description' => 'nullable|string',
            'location' => 'nullable|string|max:255',
            'venue' => 'nullable|string|max:255',
            'organizer' => 'nullable|string|max:255',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:20',
            'website' => 'nullable|url|max:255',
            'type' => 'required|in:knockout,elimination,round-robin,group-knockout,double-elimination',
            'status' => 'required|in:draft,active,completed,cancelled',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'registration_start' => 'nullable|date',
            'registration_end' => 'nullable|date|after_or_equal:registration_start',
            'min_participants' => 'nullable|integer|min:1|max:1000',
            'max_participants' => 'nullable|integer|min:1|max:1000|gte:min_participants',
            'min_players_per_team' => 'nullable|integer|min:1|max:100',
            'max_players_per_team' => 'nullable|integer|min:1|max:100|gte:min_players_per_team',
            'auto_accept_registrations' => 'boolean',
            'registration_status' => 'required|in:open,closed,full',
            'game_title' => 'nullable|string|max:255',
            'game_version' => 'nullable|string|max:100',
            'format' => 'required|in:individual,team,mixed',
            'team_size' => 'nullable|integer|min:1|max:50',
            'rounds' => 'nullable|integer|min:1|max:100',
            'matches_per_round' => 'nullable|integer|min:1|max:1000',
            'bracket_settings' => 'nullable|array',
            'stream_url' => 'nullable|url|max:255',
            'youtube_url' => 'nullable|url|max:255',
            'twitch_url' => 'nullable|url|max:255',
            'is_streamed' => 'boolean',
            'is_featured' => 'boolean',
            'entry_fee' => 'nullable|numeric|min:0|max:99999.99',
            'prize_pool' => 'nullable|numeric|min:0|max:9999999.99',
            'prize_distribution' => 'nullable|array',
            'currency' => 'nullable|string|size:3',
            'sponsor_info' => 'nullable|array',
            'rewards' => 'nullable|array',
            'banner_image' => 'nullable|string|max:255',
            'logo_image' => 'nullable|string|max:255',
            'social_links' => 'nullable|array',
            'announcement' => 'nullable|string',
            'visibility' => 'required|in:public,private,unlisted',
            'allow_spectators' => 'boolean',
            'require_approval' => 'boolean',
            'enable_two_tier_approval' => 'boolean',
            'first_approver_id' => 'nullable|exists:users,id',
            'second_approver_id' => 'nullable|exists:users,id',
            'enable_categories' => 'boolean',
            'category_fees' => 'nullable|array',
            'category_fees.men' => 'nullable|numeric|min:0',
            'category_fees.women' => 'nullable|numeric|min:0',
            'category_fees.youth_male' => 'nullable|numeric|min:0',
            'category_fees.youth_female' => 'nullable|numeric|min:0',
            'category_fees.children' => 'nullable|numeric|min:0',
            'available_categories' => 'nullable|array',
            'available_categories.*' => 'in:men,women,youth_male,youth_female,children',
            'admin_notes' => 'nullable|array',
            'metadata' => 'nullable|array',
            'rules' => 'nullable|array',
            'settings' => 'nullable|array',
        ]);

        // Check if category changes affect existing teams
        if ($tournament->enable_categories && $tournament->teams()->count() > 0) {
            // Validate category changes
            if ($request->enable_categories === false) {
                return back()->withErrors([
                    'enable_categories' => 'Cannot disable categories when teams are registered. Contact support if needed.'
                ]);
            }
            
            // Warn about fee changes
            if ($request->category_fees != $tournament->category_fees) {
                session()->flash('warning', 'Category fee changes may require payment adjustments for registered teams.');
            }
        }

        $tournament->update($validated->validated());

        // Update statistics if needed
        $tournament->updateParticipantCount();
        $tournament->updateMatchStats();

        return redirect()->route('admin.tournaments.index')->with('success', 'Tournament updated successfully!');
    }

    public function show(Tournament $tournament)
    {
        $tournament->load([
            'teams.captain',
            'teams' => function ($query) {
                $query->withPivot('status', 'registered_at', 'approved_at');
            }
        ]);

        $teamsByStatus = [
            'registered' => $tournament->getRegisteredTeams(),
            'active' => $tournament->getActiveTeams(),
            'eliminated' => $tournament->getEliminatedTeams(),
            'withdrawn' => $tournament->getWithdrawnTeams(),
        ];

        return Inertia::render('admin/tournaments/show', [
            'tournament' => $tournament,
            'teamsByStatus' => $teamsByStatus,
        ]);
    }

    public function destroy(Tournament $tournament)
    {
        $tournament->delete();
        return redirect()->route('admin.tournaments.index')->with('success', 'Tournament deleted successfully!');
    }

    public function destroyMultiple(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:tournaments,id'
        ]);

        Tournament::whereIn('id', $validated['ids'])->delete();
        return redirect()->route('admin.tournaments.index')->with('success', 'Tournaments deleted successfully!');
    }

    // Additional advanced methods
    // public function statistics()
    // {
    //     $stats = [
    //         'total_tournaments' => Tournament::count(),
    //         'active_tournaments' => Tournament::where('status', 'active')->count(),
    //         'completed_tournaments' => Tournament::where('status', 'completed')->count(),
    //         'featured_tournaments' => Tournament::where('is_featured', true)->count(),
    //         'total_prize_pool' => Tournament::sum('prize_pool'),
    //         'average_participants' => Tournament::avg('total_participants'),
    //         'upcoming_tournaments' => Tournament::upcoming()->count(),
    //         'ongoing_tournaments' => Tournament::ongoing()->count(),
    //     ];

    //     return response()->json($stats);
    // }

    public function publish(Tournament $tournament)
    {
        $tournament->publish();
        return redirect()->route('admin.tournaments.show', $tournament)
            ->with('success', 'Tournament published successfully!');
    }

    public function complete(Tournament $tournament)
    {
        $tournament->complete();
        return redirect()->route('admin.tournaments.show', $tournament)
            ->with('success', 'Tournament completed successfully!');
    }

    public function cancel(Tournament $tournament)
    {
        $tournament->cancel();
        return redirect()->route('admin.tournaments.show', $tournament)
            ->with('success', 'Tournament cancelled successfully!');
    }

    public function updateMultiple(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:tournaments,id',
            'action' => 'required|in:publish,complete,cancel,feature,unfeature',
        ]);

        $tournaments = Tournament::whereIn('id', $validated['ids'])->get();

        foreach ($tournaments as $tournament) {
            switch ($validated['action']) {
                case 'publish':
                    $tournament->publish();
                    break;
                case 'complete':
                    $tournament->complete();
                    break;
                case 'cancel':
                    $tournament->cancel();
                    break;
                case 'feature':
                    $tournament->update(['is_featured' => true]);
                    break;
                case 'unfeature':
                    $tournament->update(['is_featured' => false]);
                    break;
            }
        }

        return redirect()->route('admin.tournaments.index')->with('success', 'Tournaments updated successfully!');
    }

    /**
     * Show the team registration page for a tournament
     */
    public function registerTeams(Tournament $tournament)
    {
        // Get all teams that are not registered for this tournament
        $availableTeams = Team::with(['captain', 'tournaments'])
            ->whereDoesntHave('tournaments', function ($query) use ($tournament) {
                $query->where('tournament_id', $tournament->id);
            })
            ->get();

        // Get teams already registered for this tournament
        $registeredTeams = Team::with(['captain', 'tournaments'])
            ->whereHas('tournaments', function ($query) use ($tournament) {
                $query->where('tournament_id', $tournament->id);
            })
            ->get();

        return Inertia::render('admin/tournaments/register-teams', [
            'tournament' => $tournament,
            'available_teams' => $availableTeams,
            'registered_teams' => $registeredTeams,
            'errors' => session('errors', []),
        ]);
    }

    /**
     * Register teams for a tournament
     */
    public function registerTeamsStore(Request $request, Tournament $tournament)
    {
        $validated = $request->validate([
            'team_ids' => 'required|array|min:1',
            'team_ids.*' => 'exists:teams,id',
            'registration_notes' => 'nullable|string',
        ]);

        $teams = Team::whereIn('id', $validated['team_ids'])->get();

        foreach ($teams as $team) {
            // Check if team is already registered
            if (!$team->isRegisteredForTournament($tournament)) {
                $team->registerForTournament($tournament, [
                    'status' => 'registered',
                    'registration_notes' => $validated['registration_notes'] ?? null,
                ]);
            }
        }

        return redirect()->back()->with('success', count($teams) . ' team(s) registered successfully.');
    }

    /**
     * Unregister a team from a tournament
     */
    public function unregisterTeam(Tournament $tournament, Team $team)
    {
        if ($team->isRegisteredForTournament($tournament)) {
            $team->withdrawFromTournament($tournament);
            return redirect()->back()->with('success', 'Team unregistered successfully.');
        }

        return redirect()->back()->with('error', 'Team is not registered for this tournament.');
    }

    /**
     * Show tournament participants management page
     */
    public function participants(Tournament $tournament, Request $request)
    {
        $query = $tournament->teams()->with([
            'captain',
            'players.user',
            'officials.user'
        ]);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('teams.name', 'like', "%{$search}%")
                  ->orWhereHas('captain', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->wherePivot('status', $request->status);
        }

        if ($request->filled('team_status') && $request->team_status !== 'all') {
            $query->where('teams.status', $request->team_status);
        }

        $participants = $query->get();

        // Calculate stats
        $stats = [
            'total' => $participants->count(),
            'registered' => $participants->where('pivot.status', 'registered')->count(),
            'active' => $participants->where('pivot.status', 'active')->count(),
            'eliminated' => $participants->where('pivot.status', 'eliminated')->count(),
            'withdrawn' => $participants->where('pivot.status', 'withdrawn')->count(),
            'pending_approval' => $participants->where('pivot.status', 'pending_approval')->count(),
            'inactive' => $participants->where('pivot.status', 'inactive')->count(),
            // New approval stats
            'pending_first_approval' => $participants->where('pivot.approval_status', 'pending')->count(),
            'first_approved' => $participants->where('pivot.approval_status', 'first_approved')->count(),
            'second_approved' => $participants->where('pivot.approval_status', 'second_approved')->count(),
            'rejected' => $participants->where('pivot.approval_status', 'rejected')->count(),
        ];

        // Load tournament with approvers
        $tournament->load(['firstApprover', 'secondApprover']);

        return Inertia::render('admin/tournaments/participants', [
            'tournament' => $tournament,
            'participants' => $participants,
            'filters' => $request->only(['search', 'status', 'team_status']),
            'stats' => $stats,
            'userPermissions' => auth()->user()->getAllPermissions(),
            'userRoles' => auth()->user()->roles->pluck('name')->toArray(),
        ]);
    }

    /**
     * Update participant status
     */
    public function updateParticipantStatus(Tournament $tournament, Team $team, Request $request)
    {
        $validated = $request->validate([
            'status' => 'required|in:registered,active,eliminated,withdrawn',
        ]);

        $team->updateTournamentStatus($tournament, $validated['status']);

        if ($validated['status'] === 'active') {
            $team->updateTournamentStatus($tournament, 'active', [
                'approved_at' => now(),
            ]);
        }

        return redirect()->back()->with('success', 'Participant status updated successfully.');
    }

    /**
     * Bulk actions on participants
     */
    public function bulkActionParticipants(Tournament $tournament, Request $request)
    {
        $validated = $request->validate([
            'participant_ids' => 'required|array',
            'participant_ids.*' => 'exists:teams,id',
            'action' => 'required|in:approve,eliminate,withdraw',
        ]);

        $teams = Team::whereIn('id', $validated['participant_ids'])->get();
        $action = $validated['action'];

        foreach ($teams as $team) {
            if ($team->isRegisteredForTournament($tournament)) {
                switch ($action) {
                    case 'approve':
                        $team->updateTournamentStatus($tournament, 'active', [
                            'approved_at' => now(),
                        ]);
                        break;
                    case 'eliminate':
                        $team->updateTournamentStatus($tournament, 'eliminated');
                        break;
                    case 'withdraw':
                        $team->updateTournamentStatus($tournament, 'withdrawn');
                        break;
                }
            }
        }

        $actionText = ucfirst($action);
        return redirect()->back()->with('success', count($teams) . " participant(s) {$actionText} successfully.");
    }

    /**
     * Show participant detail page
     */
    public function participantDetail(Tournament $tournament, Team $team)
    {
        $participant = $tournament->teams()
            ->with(['captain', 'players.user', 'officials.user'])
            ->where('teams.id', $team->id)
            ->first();

        if (!$participant) {
            return redirect()->route('admin.tournaments.participants', $tournament)
                ->with('error', 'Team is not registered for this tournament.');
        }

        return Inertia::render('admin/tournaments/participant-detail', [
            'tournament' => $tournament,
            'participant' => $participant,
        ]);
    }

    /**
     * First approve a participant
     */
    public function firstApproveParticipant(Tournament $tournament, Team $team, Request $request)
    {
        $validated = $request->validate([
            'notes' => 'nullable|string|max:1000',
        ]);

        // Check if user can first approve
        if (!$tournament->canUserFirstApprove(auth()->user())) {
            return redirect()->back()->with('error', 'You do not have permission to perform first approval.');
        }

        // Check if team is registered and pending approval
        if (!$team->isRegisteredForTournament($tournament) || !$team->isPendingApproval($tournament)) {
            return redirect()->back()->with('error', 'Team is not eligible for first approval.');
        }

        $tournament->firstApproveTeam($team, auth()->user(), $validated['notes'] ?? null);

        return redirect()->back()->with('success', 'Team first approved successfully.');
    }

    /**
     * Second approve a participant
     */
    public function secondApproveParticipant(Tournament $tournament, Team $team, Request $request)
    {
        $validated = $request->validate([
            'notes' => 'nullable|string|max:1000',
        ]);

        // Check if user can second approve
        if (!$tournament->canUserSecondApprove(auth()->user())) {
            return redirect()->back()->with('error', 'You do not have permission to perform second approval.');
        }

        // Check if team is first approved
        if (!$team->isFirstApproved($tournament)) {
            return redirect()->back()->with('error', 'Team must be first approved before second approval.');
        }

        $tournament->secondApproveTeam($team, auth()->user(), $validated['notes'] ?? null);

        return redirect()->back()->with('success', 'Team second approved successfully.');
    }

    /**
     * Reject a participant
     */
    public function rejectParticipant(Tournament $tournament, Team $team, Request $request)
    {
        $validated = $request->validate([
            'reason' => 'required|string|max:1000',
        ]);

        // Check if user can approve (any level)
        if (!$tournament->canUserApprove(auth()->user())) {
            return redirect()->back()->with('error', 'You do not have permission to reject participants.');
        }

        // Check if team is registered and not already rejected
        if (!$team->isRegisteredForTournament($tournament) || $team->isRejected($tournament)) {
            return redirect()->back()->with('error', 'Team is not eligible for rejection.');
        }

        $tournament->rejectTeamApproval($team, auth()->user(), $validated['reason']);

        return redirect()->back()->with('success', 'Team rejected successfully.');
    }

    /**
     * Show tournament matches management page
     */
    public function matches(Tournament $tournament, Request $request)
    {
        $query = $tournament->matches()->with(['team1', 'team2', 'winner']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->whereHas('team1', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                })->orWhereHas('team2', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->filled('match_type') && $request->match_type !== 'all') {
            $query->where('match_type', $request->match_type);
        }

        $matches = $query->get();

        // Calculate stats
        $stats = [
            'total' => $matches->count(),
            'scheduled' => $matches->where('status', 'scheduled')->count(),
            'in_progress' => $matches->where('status', 'in_progress')->count(),
            'completed' => $matches->where('status', 'completed')->count(),
            'cancelled' => $matches->where('status', 'cancelled')->count(),
        ];

        return Inertia::render('admin/tournaments/matches', [
            'tournament' => $tournament,
            'matches' => $matches,
            'filters' => $request->only(['search', 'status', 'match_type']),
            'stats' => $stats,
        ]);
    }

    /**
     * Update match status
     */
    public function updateMatchStatus(Tournament $tournament, GameMatch $match, Request $request)
    {
        $validated = $request->validate([
            'status' => 'required|in:scheduled,in_progress,completed,cancelled',
        ]);

        $match->update(['status' => $validated['status']]);

        if ($validated['status'] === 'in_progress') {
            $match->update(['started_at' => now()]);
        } elseif ($validated['status'] === 'completed') {
            $match->update(['completed_at' => now()]);
        }

        return redirect()->back()->with('success', 'Match status updated successfully.');
    }

    /**
     * Bulk actions on matches
     */
    public function bulkActionMatches(Tournament $tournament, Request $request)
    {
        $validated = $request->validate([
            'match_ids' => 'required|array',
            'match_ids.*' => 'exists:game_matches,id',
            'action' => 'required|in:start,complete,cancel',
        ]);

        $matches = GameMatch::whereIn('id', $validated['match_ids'])->get();
        $action = $validated['action'];

        foreach ($matches as $match) {
            switch ($action) {
                case 'start':
                    $match->update(['status' => 'in_progress', 'started_at' => now()]);
                    break;
                case 'complete':
                    $match->update(['status' => 'completed', 'completed_at' => now()]);
                    break;
                case 'cancel':
                    $match->update(['status' => 'cancelled']);
                    break;
            }
        }

        $actionText = ucfirst($action);
        return redirect()->back()->with('success', count($matches) . " match(es) {$actionText} successfully.");
    }

    /**
     * Show tournament brackets management page
     */
    public function brackets(Tournament $tournament)
    {
        // For now, we'll return empty brackets until the bracket system is implemented
        $brackets = collect([]);
        
        $stats = [
            'total_brackets' => 0,
            'active_brackets' => 0,
            'completed_brackets' => 0,
            'total_matches' => $tournament->matches()->count(),
            'completed_matches' => $tournament->matches()->where('status', 'completed')->count(),
        ];

        return Inertia::render('admin/tournaments/brackets', [
            'tournament' => $tournament,
            'brackets' => $brackets,
            'stats' => $stats,
        ]);
    }

    /**
     * Generate bracket for tournament
     */
    public function generateBracket(Tournament $tournament)
    {
        // Placeholder for bracket generation logic
        return redirect()->back()->with('success', 'Bracket generation feature coming soon.');
    }

    /**
     * Auto advance winners in brackets
     */
    public function autoAdvanceBracket(Tournament $tournament)
    {
        // Placeholder for auto-advance logic
        return redirect()->back()->with('success', 'Auto-advance feature coming soon.');
    }

    /**
     * Show tournament statistics page
     */
    public function statistics(Tournament $tournament, Request $request)
    {
        // Calculate comprehensive statistics
        $stats = [
            'overview' => [
                'total_participants' => $tournament->teams()->count(),
                'total_matches' => $tournament->matches()->count(),
                'completed_matches' => $tournament->matches()->where('status', 'completed')->count(),
                'average_match_duration' => $tournament->matches()->where('status', 'completed')->avg('duration') ?? 0,
                'total_prize_pool' => $tournament->prize_pool ?? 0,
                'registration_rate' => 85, // Placeholder
            ],
            'participants' => [
                'by_status' => [
                    'active' => $tournament->teams()->where('teams.status', 'active')->count(),
                    'inactive' => $tournament->teams()->where('teams.status', 'inactive')->count(),
                ],
                'by_team_size' => [
                    'small' => 5,
                    'medium' => 8,
                    'large' => 3,
                ],
                'registration_timeline' => [
                    ['date' => '2024-01-01', 'count' => 5],
                    ['date' => '2024-01-02', 'count' => 8],
                    ['date' => '2024-01-03', 'count' => 12],
                ],
            ],
            'matches' => [
                'by_status' => [
                    'scheduled' => $tournament->matches()->where('status', 'scheduled')->count(),
                    'in_progress' => $tournament->matches()->where('status', 'in_progress')->count(),
                    'completed' => $tournament->matches()->where('status', 'completed')->count(),
                    'cancelled' => $tournament->matches()->where('status', 'cancelled')->count(),
                ],
                'by_type' => [
                    'group' => $tournament->matches()->where('match_type', 'group')->count(),
                    'knockout' => $tournament->matches()->where('match_type', 'knockout')->count(),
                    'final' => $tournament->matches()->where('match_type', 'final')->count(),
                ],
                'average_scores' => [
                    'team1' => 15,
                    'team2' => 12,
                ],
                'duration_distribution' => [
                    'short' => 5,
                    'medium' => 8,
                    'long' => 3,
                ],
            ],
            'performance' => [
                'top_teams' => [
                    ['team' => 'Team Alpha', 'wins' => 5, 'losses' => 1, 'win_rate' => 83],
                    ['team' => 'Team Beta', 'wins' => 4, 'losses' => 2, 'win_rate' => 67],
                    ['team' => 'Team Gamma', 'wins' => 3, 'losses' => 3, 'win_rate' => 50],
                ],
                'most_active_teams' => [
                    ['team' => 'Team Alpha', 'matches_played' => 6],
                    ['team' => 'Team Beta', 'matches_played' => 6],
                    ['team' => 'Team Gamma', 'matches_played' => 6],
                ],
                'longest_matches' => [
                    ['match' => 'Match 1', 'duration' => 120, 'teams' => 'Team A vs Team B'],
                    ['match' => 'Match 2', 'duration' => 95, 'teams' => 'Team C vs Team D'],
                ],
            ],
            'trends' => [
                'daily_matches' => [
                    ['date' => '2024-01-01', 'count' => 3],
                    ['date' => '2024-01-02', 'count' => 5],
                    ['date' => '2024-01-03', 'count' => 4],
                ],
                'weekly_participation' => [
                    ['week' => 'Week 1', 'participants' => 16],
                    ['week' => 'Week 2', 'participants' => 18],
                    ['week' => 'Week 3', 'participants' => 20],
                ],
                'monthly_growth' => [
                    ['month' => 'January', 'growth' => 15],
                    ['month' => 'February', 'growth' => 12],
                    ['month' => 'March', 'growth' => 18],
                ],
            ],
        ];

        return Inertia::render('admin/tournaments/statistics', [
            'tournament' => $tournament,
            'stats' => $stats,
            'filters' => $request->only(['period', 'view']),
        ]);
    }

    /**
     * Generate and download certificate of participation for a team
     */
    public function generateCertificate(Tournament $tournament, Team $team)
    {
        // Check if team is registered for this tournament
        if (!$tournament->isTeamRegistered($team)) {
            return redirect()->back()->with('error', 'Team is not registered for this tournament.');
        }

        // Check if team is approved/active
        $teamStatus = $tournament->getTeamStatus($team);
        if (!in_array($teamStatus, ['active', 'eliminated', 'withdrawn'])) {
            return redirect()->back()->with('error', 'Certificate can only be generated for teams that have participated in the tournament.');
        }

        $data = [
            'tournament' => $tournament,
            'team' => $team->load(['captain', 'members']),
            'generated_at' => now()->format('F j, Y'),
            'certificate_id' => 'CERT-' . $tournament->id . '-' . $team->id . '-' . now()->format('Ymd'),
        ];

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('certificates.participation', $data);
        $pdf->setPaper('a4', 'landscape');
        
        $filename = 'Certificate_' . $tournament->name . '_' . $team->name . '_' . now()->format('Y-m-d') . '.pdf';
        $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
        
        return $pdf->download($filename);
    }

    /**
     * Bulk generate certificates for all active participants
     */
    public function bulkGenerateCertificates(Tournament $tournament, Request $request)
    {
        $validated = $request->validate([
            'status_filter' => 'nullable|in:all,active,eliminated,withdrawn',
        ]);

        $statusFilter = $validated['status_filter'] ?? 'active';
        
        if ($statusFilter === 'all') {
            $teams = $tournament->teams()->wherePivotIn('status', ['active', 'eliminated', 'withdrawn'])->get();
        } else {
            $teams = $tournament->teams()->wherePivot('status', $statusFilter)->get();
        }

        if ($teams->isEmpty()) {
            return redirect()->back()->with('error', 'No teams found matching the criteria.');
        }

        // For now, we'll generate a single PDF with all certificates
        // In a production environment, you might want to create a ZIP file with individual PDFs
        $data = [
            'tournament' => $tournament,
            'teams' => $teams->load(['captain', 'members']),
            'generated_at' => now()->format('F j, Y'),
            'bulk_generation' => true,
        ];

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('certificates.bulk-participation', $data);
        $pdf->setPaper('a4', 'landscape');
        
        $filename = 'Certificates_' . $tournament->name . '_' . $teams->count() . '_teams_' . now()->format('Y-m-d') . '.pdf';
        $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
        
        return $pdf->download($filename);
    }
} 