<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Team;
use App\Models\Player;
use App\Models\Payment;
use App\Models\Tournament;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class TeamAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:teams.view')->only(['index', 'show']);
        $this->middleware('permission:teams.create')->only(['create', 'store']);
        $this->middleware('permission:teams.edit')->only(['edit', 'update']);
        $this->middleware('permission:teams.delete')->only(['destroy', 'destroyMultiple']);
        $this->middleware('permission:teams.approve')->only(['approve', 'reject', 'firstApprove', 'secondApprove']);
        $this->middleware('permission:teams.manage_payments')->only(['addManualPayment']);
    }

    public function index(Request $request)
    {
            $query = Team::with(['manager', 'tournaments', 'players', 'payments' => function($q) {
                $q->where('status', 'completed')->latest();
            }]);

        // Filter by tournament if specified
        if ($request->has('tournament_id') && $request->tournament_id) {
            $tournament = Tournament::find($request->tournament_id);
            if ($tournament) {
                $query->forTournamentWithStatus($tournament, 'active');
            }
        }

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('manager', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        $query->orderBy('id', 'desc');
        $teams = $query->paginate(15);
        
        // Get tournament-specific settings for each tournament
        $tournamentsWithSettings = Tournament::select('id', 'name', 'team_size', 'settings', 'entry_fee', 'currency')->get()->map(function($tournament) {
            $settings = is_string($tournament->settings) ? json_decode($tournament->settings, true) : $tournament->settings;
            
            return [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'entry_fee' => $tournament->entry_fee,
                'currency' => $tournament->currency ?? 'KES',
                'min_players_per_team' => $settings['min_players_per_team'] ?? config('tournament.min_players_per_team', 11),
                'max_players_per_team' => $settings['max_players_per_team'] ?? $tournament->team_size ?? config('tournament.max_players_per_team', 23),
            ];
        });
        
        return Inertia::render('admin/teams/index', [
            'teams' => $teams,
            'filters' => $request->only(['search', 'tournament_id']),
            'tournaments' => $tournamentsWithSettings,
            'users' => User::select('id', 'name', 'email')->get(),
            'tournamentSettings' => [
                'min_players_per_team' => config('tournament.min_players_per_team', 11),
                'max_players_per_team' => config('tournament.max_players_per_team', 23),
            ],
        ]);
    }

    public function create()
    {
        $users = User::select('id', 'name', 'email')->get();
        $existingTeams = Team::select('id', 'name', 'description')->get();
        
        // Get tournaments with settings
        $tournamentsWithSettings = $this->getTournamentsWithSettings();
        
        // Get the active tournament for default registration fee
        $activeTournament = Tournament::where('status', 'active')->first();
        $registrationFee = $activeTournament ? $activeTournament->entry_fee : config('tournament.registration_fee', 1000);
        
        return Inertia::render('admin/teams/create', [
            'tournaments' => $tournamentsWithSettings,
            'users' => $users,
            'existing_teams' => $existingTeams,
            'registration_fee' => $registrationFee,
            'tournamentSettings' => [
                'min_players_per_team' => config('tournament.min_players_per_team', 11),
                'max_players_per_team' => config('tournament.max_players_per_team', 23),
            ],
            'active_tournament' => $activeTournament ? [
                'id' => $activeTournament->id,
                'name' => $activeTournament->name,
                'entry_fee' => $activeTournament->entry_fee,
                'currency' => $activeTournament->currency ?? 'KES'
            ] : null
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'required|in:active,eliminated,withdrawn',
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'manager_id' => 'required|integer|exists:users,id',
            // Payment related fields
            'payment_status' => 'nullable|string|in:pending,processing,completed,failed',
            'transaction_id' => 'nullable|string|max:255',
            'payment_amount' => 'nullable|numeric|min:0',
            'payment_method' => 'nullable|string|in:mpesa,card,bank_transfer,paypal',
            'phone_number' => 'nullable|string|max:20',
            // Players data
            'players' => 'nullable|array|min:11|max:23',
            'players.*.name' => 'required_with:players|string|max:255',
            'players.*.email' => 'required_with:players|string|email|max:255',
            'players.*.phone' => 'required_with:players|string|max:20',
            'players.*.position' => 'nullable|string|max:100',
            'players.*.jersey_number' => 'nullable|string|max:3',
            'players.*.existing_user_id' => 'nullable|integer|exists:users,id',
            // Additional team settings
            'logo' => 'nullable|string|max:500',
            'max_players' => 'nullable|integer|min:1|max:50',
            'notes' => 'nullable|string',
            'notify_match_updates' => 'nullable|boolean',
            'notify_tournament_updates' => 'nullable|boolean',
            // Team registration options
            'use_existing_team' => 'nullable|boolean',
            'existing_team_id' => 'nullable|integer|exists:teams,id',
        ]);

        try {
            DB::beginTransaction();

            $tournament = Tournament::findOrFail($validated['tournament_id']);

            // Check if using existing team
            if ($validated['use_existing_team'] && $validated['existing_team_id']) {
                $team = Team::findOrFail($validated['existing_team_id']);
                
                // Check if team is already registered for this tournament
                if ($team->isRegisteredForTournament($tournament)) {
                    return back()->withErrors(['team' => 'This team is already registered for this tournament']);
                }
                
                // Register existing team for tournament
                $team->registerForTournament($tournament, [
                    'notes' => $validated['notes'] ?? 'Registered by admin',
                    'tournament_data' => [
                        'registration_method' => 'admin',
                        'admin_notes' => $validated['notes'],
                    ]
                ]);
                
            } else {
                // Create new team
                $team = Team::create([
                    'name' => $validated['name'],
                    'description' => $validated['description'],
                    'status' => $validated['status'],
                    'manager_id' => $validated['manager_id'],
                    'logo' => $validated['logo'] ?? null,
                    'max_players' => $validated['max_players'] ?? 23,
                    'notes' => $validated['notes'] ?? null,
                    'notify_match_updates' => $validated['notify_match_updates'] ?? true,
                    'notify_tournament_updates' => $validated['notify_tournament_updates'] ?? true,
                ]);

                // Register new team for tournament
                $team->registerForTournament($tournament, [
                    'notes' => $validated['notes'] ?? 'Created by admin',
                    'tournament_data' => [
                        'registration_method' => 'admin',
                        'admin_notes' => $validated['notes'],
                    ]
                ]);
            }

            // Create players if provided
            if (!empty($validated['players'])) {
                foreach ($validated['players'] as $playerData) {
                    // Skip empty players
                    if (empty($playerData['name']) || empty($playerData['email'])) {
                        continue;
                    }

                    Player::create([
                        'team_id' => $team->id,
                        'user_id' => $playerData['existing_user_id'] ?? null,
                        'tournament_id' => $validated['tournament_id'],
                        'display_name' => $playerData['name'],
                        'phone' => $playerData['phone'],
                        'football_position' => $playerData['position'] ?? 'Midfielder',
                        'jersey_number' => $playerData['jersey_number'] ?? null,
                        'status' => 'registered',
                        'verification_status' => 'unverified',
                        'registered_at' => now(),
                    ]);
                }
            }

            // Handle payment information
            if (!empty($validated['payment_status']) && $validated['payment_status'] === 'completed') {
                // Create payment record
                Payment::create([
                    'user_id' => $validated['manager_id'],
                    'team_id' => $team->id,
                    'payment_method' => $validated['payment_method'] ?? 'mpesa',
                    'amount' => $validated['payment_amount'] ?? 5000,
                    'currency' => 'KES',
                    'reference_number' => $validated['transaction_id'] ?? uniqid('TEAM_'),
                    'description' => "Tournament Registration - {$team->name}",
                    'status' => 'completed',
                ]);

                // Update team captain payment status
                $captain = User::find($validated['manager_id']);
                if ($captain) {
                    $captain->update(['payment_status' => 'completed']);
                }
            }

            DB::commit();

            // Log team creation with payment info if provided
            if ($validated['payment_status'] === 'completed') {
                Log::info('Team created with payment', [
                    'team_id' => $team->id,
                    'team_name' => $team->name,
                    'tournament_id' => $tournament->id,
                    'transaction_id' => $validated['transaction_id'],
                    'payment_amount' => $validated['payment_amount'],
                    'players_count' => count($validated['players'] ?? [])
                ]);
            }

            return redirect()->route('admin.teams.show', $team)->with('success', 'Team created successfully!');

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Enhanced team creation failed: ' . $e->getMessage());
            return back()->withErrors(['creation' => 'Team creation failed. Please try again.'])->withInput();
        }
    }

    public function edit(Team $team)
    {
        // Load team relationships
        $team->load(['players', 'payments' => function($q) {
            $q->where('status', 'completed')->latest();
        }, 'tournaments']);
        
        $users = User::select('id', 'name', 'email')->get();
        
        // Get tournaments with settings
        $tournamentsWithSettings = $this->getTournamentsWithSettings();
        
        return Inertia::render('admin/teams/edit', [
            'team' => $team,
            'tournaments' => $tournamentsWithSettings,
            'users' => $users,
            'tournamentSettings' => [
                'min_players_per_team' => config('tournament.min_players_per_team', 11),
                'max_players_per_team' => config('tournament.max_players_per_team', 23),
            ],
        ]);
    }

    public function update(Request $request, Team $team)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'required|in:active,eliminated,withdrawn',
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'manager_id' => 'required|integer|exists:users,id',
        ]);
        $team->update($validated);
        return redirect()->route('admin.teams.index')->with('success', 'Team updated!');
    }

    public function show(Team $team)
    {
        $team->load([
            'manager',
            'tournaments',
            'teamPlayers.user',
            'teamOfficials',
            'players',
            'payments' => function($q) {
                $q->where('status', 'completed')->latest();
            },
            'tournaments' => function ($query) {
                $query->withPivot('status', 'registered_at', 'approved_at', 'approval_status',
                                 'first_approved_by', 'first_approved_at', 'second_approved_by',
                                 'second_approved_at', 'rejection_reason');
            }
        ]);

        // Get tournaments with settings
        $tournamentsWithSettings = $this->getTournamentsWithSettings();

        return Inertia::render('admin/teams/show', [
            'team' => $team,
            'tournaments' => $tournamentsWithSettings,
            'tournamentSettings' => [
                'min_players_per_team' => config('tournament.min_players_per_team', 11),
                'max_players_per_team' => config('tournament.max_players_per_team', 23),
            ],
        ]);
    }

    public function destroy(Team $team)
    {
        $team->delete();
        return redirect()->route('admin.teams.index')->with('success', 'Team deleted!');
    }

    public function destroyMultiple(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:teams,id'
        ]);
        Team::whereIn('id', $validated['ids'])->delete();
        return redirect()->route('admin.teams.index')->with('success', 'Teams deleted!');
    }

    /**
     * Approve a team for tournament participation
     */
    public function approve(Request $request, Team $team)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'notes' => 'nullable|string|max:1000',
        ]);

        $tournament = Tournament::findOrFail($validated['tournament_id']);
        $user = auth()->user();

        // Check if user can approve this team
        $approvalCheck = $tournament->canUserApproveTeam($user, $team);
        
        if (!$approvalCheck['can_approve'] && !$approvalCheck['is_super_admin']) {
            return back()->withErrors(['approval' => $approvalCheck['reason']]);
        }

        // Check team completeness
        $completenessCheck = $team->canBeApprovedForTournament($tournament);
        if (!$completenessCheck['can_approve']) {
            return back()->withErrors(['approval' => implode(', ', $completenessCheck['reasons'])]);
        }

        try {
            if ($approvalCheck['is_super_admin']) {
                // Super admin can approve directly
                $tournament->secondApproveTeam($team, $user, $validated['notes']);
                $message = "Team {$team->name} approved successfully by super admin!";
            } else {
                // Regular approval based on tournament settings
                if ($tournament->enable_two_tier_approval) {
                    $tournament->firstApproveTeam($team, $user, $validated['notes']);
                    $message = "Team {$team->name} first approved successfully!";
                } else {
                    $tournament->secondApproveTeam($team, $user, $validated['notes']);
                    $message = "Team {$team->name} approved successfully!";
                }
            }

            return back()->with('success', $message);
        } catch (\Exception $e) {
            \Log::error('Team approval failed: ' . $e->getMessage());
            return back()->withErrors(['approval' => 'Team approval failed. Please try again.']);
        }
    }

    /**
     * Reject a team for tournament participation
     */
    public function reject(Request $request, Team $team)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'reason' => 'required|string|max:1000',
        ]);

        $tournament = Tournament::findOrFail($validated['tournament_id']);
        $user = auth()->user();

        // Check if user can reject this team
        $approvalCheck = $tournament->canUserApproveTeam($user, $team);
        
        if (!$approvalCheck['can_reject'] && !$approvalCheck['is_super_admin']) {
            return back()->withErrors(['rejection' => $approvalCheck['reason']]);
        }

        try {
            $tournament->rejectTeamApproval($team, $user, $validated['reason']);
            $message = "Team {$team->name} rejected successfully!";
            
            return back()->with('success', $message);
        } catch (\Exception $e) {
            \Log::error('Team rejection failed: ' . $e->getMessage());
            return back()->withErrors(['rejection' => 'Team rejection failed. Please try again.']);
        }
    }

    /**
     * First approve a team (for two-tier approval system)
     */
    public function firstApprove(Request $request, Team $team)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'notes' => 'nullable|string|max:1000',
        ]);

        $tournament = Tournament::findOrFail($validated['tournament_id']);
        $user = auth()->user();

        // Check if user can first approve this team
        $approvalCheck = $tournament->canUserApproveTeam($user, $team);
        
        if (!$approvalCheck['can_first_approve'] && !$approvalCheck['is_super_admin']) {
            return back()->withErrors(['approval' => $approvalCheck['reason']]);
        }

        // Check team completeness
        $completenessCheck = $team->canBeApprovedForTournament($tournament);
        if (!$completenessCheck['can_approve']) {
            return back()->withErrors(['approval' => implode(', ', $completenessCheck['reasons'])]);
        }

        try {
            $tournament->firstApproveTeam($team, $user, $validated['notes']);
            $message = "Team {$team->name} first approved successfully!";
            
            return back()->with('success', $message);
        } catch (\Exception $e) {
            \Log::error('Team first approval failed: ' . $e->getMessage());
            return back()->withErrors(['approval' => 'Team first approval failed. Please try again.']);
        }
    }

    /**
     * Second approve a team (for two-tier approval system)
     */
    public function secondApprove(Request $request, Team $team)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'notes' => 'nullable|string|max:1000',
        ]);

        $tournament = Tournament::findOrFail($validated['tournament_id']);
        $user = auth()->user();

        // Check if user can second approve this team
        $approvalCheck = $tournament->canUserApproveTeam($user, $team);
        
        if (!$approvalCheck['can_second_approve'] && !$approvalCheck['is_super_admin']) {
            return back()->withErrors(['approval' => $approvalCheck['reason']]);
        }

        // Check if team is first approved (unless super admin)
        if ($team->getApprovalStatus($tournament) !== 'first_approved' && !$approvalCheck['is_super_admin']) {
            return back()->withErrors(['approval' => 'Team must be first approved before second approval']);
        }

        try {
            $tournament->secondApproveTeam($team, $user, $validated['notes']);
            $message = "Team {$team->name} second approved successfully!";
            
            return back()->with('success', $message);
        } catch (\Exception $e) {
            \Log::error('Team second approval failed: ' . $e->getMessage());
            return back()->withErrors(['approval' => 'Team second approval failed. Please try again.']);
        }
    }

    /**
     * Bulk approve teams
     */
    public function bulkApprove(Request $request)
    {
        $validated = $request->validate([
            'team_ids' => 'required|array',
            'team_ids.*' => 'exists:teams,id',
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'notes' => 'nullable|string|max:1000',
        ]);

        $tournament = Tournament::findOrFail($validated['tournament_id']);
        $teams = Team::whereIn('id', $validated['team_ids'])->get();
        $approvedCount = 0;
        $errors = [];

        foreach ($teams as $team) {
            try {
                if (!$team->isRegisteredForTournament($tournament)) {
                    $errors[] = "Team {$team->name} is not registered for this tournament";
                    continue;
                }

                if ($team->getTournamentStatus($tournament) === 'active') {
                    $errors[] = "Team {$team->name} is already approved";
                    continue;
                }

                if ($tournament->enable_two_tier_approval) {
                    $tournament->firstApproveTeam($team, auth()->user(), $validated['notes']);
                } else {
                    $team->updateTournamentStatus($tournament, 'active');
                }
                $approvedCount++;
            } catch (\Exception $e) {
                $errors[] = "Failed to approve team {$team->name}: " . $e->getMessage();
            }
        }

        $message = "Successfully approved {$approvedCount} team(s)";
        if (!empty($errors)) {
            $message .= ". Errors: " . implode(', ', $errors);
        }

        return back()->with('success', $message);
    }

    /**
     * Show detailed team approval view with player information
     */
    public function approvalView(Request $request, Team $team)
    {
        $tournamentId = $request->get('tournament_id');
        
        if (!$tournamentId) {
            return back()->withErrors(['tournament' => 'Tournament ID is required']);
        }

        $tournament = Tournament::findOrFail($tournamentId);
        $user = auth()->user();
        
        // Load team with all necessary relationships
        $team->load([
            'captain',
            'players.user',
            'teamOfficials.user',
            'payments' => function($q) {
                $q->latest();
            },
            'tournaments' => function ($query) use ($tournamentId) {
                $query->where('tournament_id', $tournamentId)
                      ->withPivot('status', 'registered_at', 'approved_at', 'approval_status', 
                                 'first_approved_by', 'first_approved_at', 'second_approved_by', 
                                 'second_approved_at', 'first_approval_notes', 'second_approval_notes', 
                                 'rejection_reason', 'rejected_by', 'rejected_at');
            }
        ]);

        // Get approval summary
        $approvalSummary = $team->getApprovalSummary($tournament);
        
        // Get user's approval permissions
        $userApprovalPermissions = $tournament->canUserApproveTeam($user, $team);

        // Get all tournaments for dropdown with settings
        $tournamentsWithSettings = $this->getTournamentsWithSettings();

        return Inertia::render('admin/teams/approval-view', [
            'team' => $team,
            'tournament' => [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'entry_fee' => $tournament->entry_fee,
                'currency' => $tournament->currency ?? 'KES',
                'min_players_per_team' => $tournament->min_players_per_team,
                'max_players_per_team' => $tournament->max_players_per_team,
                'enable_two_tier_approval' => $tournament->enable_two_tier_approval,
                'require_approval' => $tournament->require_approval,
            ],
            'approvalSummary' => $approvalSummary,
            'userApprovalPermissions' => $userApprovalPermissions,
            'tournaments' => $tournamentsWithSettings,
            'tournamentSettings' => [
                'min_players_per_team' => $tournament->min_players_per_team ?? config('tournament.min_players_per_team', 11),
                'max_players_per_team' => $tournament->max_players_per_team ?? config('tournament.max_players_per_team', 23),
            ],
        ]);
    }

    /**
     * Approve individual player
     */
    public function approvePlayer(Request $request, Team $team, Player $player)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'notes' => 'nullable|string|max:1000',
        ]);

        $tournament = Tournament::findOrFail($validated['tournament_id']);

        // Check if player belongs to team
        if ($player->team_id !== $team->id) {
            return back()->withErrors(['player' => 'Player does not belong to this team']);
        }

        // Check player completeness
        $completeness = $team->checkPlayerCompleteness($player);
        if (!$completeness['is_complete']) {
            return back()->withErrors(['player' => 'Player profile is incomplete. Missing: ' . implode(', ', $completeness['missing_required'])]);
        }

        try {
            $player->update([
                'status' => 'active',
                'verification_status' => 'verified',
                'verified_at' => now(),
            ]);

            return back()->with('success', "Player {$player->display_name} approved successfully!");
        } catch (\Exception $e) {
            \Log::error('Player approval failed: ' . $e->getMessage());
            return back()->withErrors(['approval' => 'Player approval failed. Please try again.']);
        }
    }

    /**
     * Reject individual player
     */
    public function rejectPlayer(Request $request, Team $team, Player $player)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'reason' => 'required|string|max:1000',
        ]);

        $tournament = Tournament::findOrFail($validated['tournament_id']);

        // Check if player belongs to team
        if ($player->team_id !== $team->id) {
            return back()->withErrors(['player' => 'Player does not belong to this team']);
        }

        try {
            $player->update([
                'verification_status' => 'rejected',
                'status' => 'eliminated',
            ]);

            return back()->with('success', "Player {$player->display_name} rejected successfully!");
        } catch (\Exception $e) {
            \Log::error('Player rejection failed: ' . $e->getMessage());
            return back()->withErrors(['rejection' => 'Player rejection failed. Please try again.']);
        }
    }

    /**
     * Bulk approve all complete players in a team
     */
    public function bulkApprovePlayers(Request $request, Team $team)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'notes' => 'nullable|string|max:1000',
        ]);

        $tournament = Tournament::findOrFail($validated['tournament_id']);
        $completeness = $team->getPlayerCompletenessStatus($tournament);
        
        $approvedCount = 0;
        $errors = [];

        foreach ($completeness['incomplete_players'] as $incompletePlayer) {
            $player = $incompletePlayer['player'];
            try {
                $player->update([
                    'status' => 'active',
                    'verification_status' => 'verified',
                    'verified_at' => now(),
                ]);
                $approvedCount++;
            } catch (\Exception $e) {
                $errors[] = "Failed to approve player {$player->display_name}: " . $e->getMessage();
            }
        }

        $message = "Successfully approved {$approvedCount} player(s)";
        if (!empty($errors)) {
            $message .= ". Errors: " . implode(', ', $errors);
        }

        return back()->with('success', $message);
    }

    /**
     * Manually add payment for a team
     */
    public function addManualPayment(Request $request, Team $team)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|exists:tournaments,id',
            'amount' => 'required|numeric|min:0',
            'currency' => 'nullable|string|max:3',
            'payment_method' => 'required|in:mpesa,cash,bank_transfer,card,other',
            'reference_number' => 'nullable|string|max:255',
            'transaction_id' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:500',
            'paid_at' => 'nullable|date',
        ]);

        try {
            DB::beginTransaction();

            // Get tournament for currency
            $tournament = Tournament::find($validated['tournament_id']);
            $currency = $validated['currency'] ?? $tournament->currency ?? 'KES';

            // Create payment record
            $payment = Payment::create([
                'user_id' => auth()->id(),
                'team_id' => $team->id,
                'tournament_id' => $validated['tournament_id'],
                'amount' => $validated['amount'],
                'currency' => $currency,
                'payment_method' => $validated['payment_method'],
                'reference_number' => $validated['reference_number'] ?? 'MANUAL-' . time(),
                'status' => 'completed',
                'description' => $validated['description'] ?? 'Manual payment added by admin',
                'paid_at' => $validated['paid_at'] ?? now(),
            ]);

            // Update team payment status
            $team->update([
                'payment_status' => 'completed',
                'payment_amount' => $validated['amount'],
                'payment_method' => $validated['payment_method'],
                'payment_completed_at' => $validated['paid_at'] ?? now(),
                'transaction_id' => $validated['transaction_id'] ?? $payment->reference_number,
            ]);

            DB::commit();

            Log::info('Manual payment added for team', [
                'team_id' => $team->id,
                'payment_id' => $payment->id,
                'amount' => $validated['amount'],
                'added_by' => auth()->id(),
            ]);

            return back()->with('success', 'Payment added successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Manual payment failed: ' . $e->getMessage());
            return back()->withErrors(['payment' => 'Failed to add payment. Please try again.']);
        }
    }

    /**
     * Helper method to get tournaments with settings
     */
    private function getTournamentsWithSettings()
    {
        return Tournament::select('id', 'name', 'team_size', 'settings', 'entry_fee', 'currency', 'min_players_per_team', 'max_players_per_team')->get()->map(function($tournament) {
            return [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'entry_fee' => $tournament->entry_fee,
                'currency' => $tournament->currency ?? 'KES',
                'min_players_per_team' => $tournament->min_players_per_team ?? config('tournament.min_players_per_team', 11),
                'max_players_per_team' => $tournament->max_players_per_team ?? $tournament->team_size ?? config('tournament.max_players_per_team', 23),
            ];
        });
    }
} 