<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Helpers\EnvHelper;
use App\Helpers\SettingsValidator;
use App\Services\EmailSettingsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Inertia\Inertia;

class SettingsController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:settings.view')->only(['index', 'getEmailStatus']);
        $this->middleware('permission:settings.edit')->only(['update', 'sendTestEmail', 'testSmtpConnection']);
        $this->middleware('permission:settings.system')->only(['runCommand']);
    }

    public function index()
    {
        // Get current settings from config and .env
        $settings = [
            // General Settings
            'site_name' => config('app.name'),
            'site_description' => config('app.description'),
            'contact_email' => config('mail.from.address'),
            'timezone' => config('app.timezone'),
            'maintenance_mode' => app()->isDownForMaintenance(),
            
            // SEO Settings
            'meta_title' => config('app.meta_title'),
            'meta_description' => config('app.meta_description'),
            'meta_keywords' => config('app.meta_keywords'),
            'og_image' => config('app.og_image'),
            
            // Branding
            'logo' => config('app.logo'),
            'favicon' => config('app.favicon'),
            
            // Environment
            'app_debug' => config('app.debug'),
            'app_env' => config('app.env'),
            'mail_from_address' => config('mail.from.address'),
            'mail_from_name' => config('mail.from.name'),
            'database_connection' => config('database.default'),
            'contact_phone' => config('app.contact_phone'),
            
            // Queue Settings
            'queue_connection' => config('queue.default'),
            'queue_driver' => config('queue.default'),
            
            // Cache Settings
            'cache_driver' => config('cache.default'),
            'session_driver' => config('session.driver'),
            
            // Mail Settings
            'mail_driver' => config('mail.default'),
            'mail_host' => config('mail.mailers.smtp.host'),
            'mail_port' => config('mail.mailers.smtp.port'),
            'mail_username' => config('mail.mailers.smtp.username'),
            'mail_password' => config('mail.mailers.smtp.password'),
            'mail_encryption' => config('mail.mailers.smtp.encryption'),
            'mail_timeout' => config('mail.mailers.smtp.timeout', 60),
            'mail_verify_peer' => config('mail.mailers.smtp.verify_peer', true),
            
            // Storage Settings
            'filesystem_disk' => config('filesystems.default'),
            'storage_path' => config('filesystems.disks.public.root'),
        ];

        // Get validation results
        $validations = SettingsValidator::getAllValidations();
        $hasCriticalIssues = SettingsValidator::hasCriticalIssues();

        return Inertia::render('admin/Settings', [
            'settings' => $settings,
            'validations' => $validations,
            'hasCriticalIssues' => $hasCriticalIssues,
        ]);
    }

    public function update(Request $request)
    {
        // Log the incoming data for debugging
        \Log::info('Settings update request received', $request->all());
        
        $validated = $request->validate([
            // General Settings
            'site_name' => 'required|string|max:255',
            'site_description' => 'nullable|string|max:500',
            'contact_email' => 'required|email',
            'contact_phone' => 'required|string',
            'timezone' => 'required|string',
            'maintenance_mode' => 'boolean',
            
            // SEO Settings
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|string|max:500',
            'og_image' => 'nullable|string|max:500',
            
            // Branding
            'logo' => 'nullable|string|max:500',
            'favicon' => 'nullable|string|max:500',
            
            // Environment
            'app_debug' => 'boolean',
            'app_env' => 'required|string|in:production,staging,local',
            'mail_from_address' => 'required|email',
            'mail_from_name' => 'required|string|max:255',
            'database_connection' => 'required|string|in:mysql,pgsql,sqlite',
            
            // Queue Settings
            'queue_connection' => 'required|string|in:sync,database,redis,beanstalkd,sqs',
            'queue_driver' => 'required|string|in:sync,database,redis,beanstalkd,sqs',
            
            // Cache Settings
            'cache_driver' => 'required|string|in:file,database,redis,memcached',
            'session_driver' => 'required|string|in:file,database,redis,memcached',
            
            // Mail Settings
            'mail_driver' => 'required|string|in:smtp,mailgun,ses,sendmail,log,postmark,resend',
            'mail_host' => 'required_if:mail_driver,smtp|nullable|string|max:255',
            'mail_port' => 'required_if:mail_driver,smtp|nullable|integer|between:1,65535',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string|max:255',
            'mail_encryption' => 'nullable|string|in:tls,ssl,starttls,',
            'mail_timeout' => 'nullable|integer|between:10,300',
            'mail_verify_peer' => 'nullable|boolean',
            
            // Storage Settings
            'filesystem_disk' => 'required|string|in:local,s3,azure,digitalocean',
            'storage_path' => 'nullable|string|max:500',
        ]);

        \Log::info('Settings validation passed', $validated);

        try {
            // Update .env file using EnvHelper
            $this->updateEnvFile($validated);
            
            // Update config cache
            $this->updateConfigCache();
            
            // Handle maintenance mode
            if ($validated['maintenance_mode']) {
                \Artisan::call('down');
            } else {
                \Artisan::call('up');
            }

            return redirect()->back()->with('success', 'Settings updated successfully.');
        } catch (\Exception $e) {
            \Log::error('Settings update failed', ['error' => $e->getMessage()]);
            return redirect()->back()->with('error', 'Failed to update settings: ' . $e->getMessage());
        }
    }

    public function runCommand(Request $request)
    {
        $request->validate([
            'command' => 'required|string|max:255',
        ]);

        $command = $request->input('command');
        
        // List of allowed commands for security
        $allowedCommands = [
            'migrate', 'migrate:fresh', 'migrate:rollback', 'db:seed',
            'cache:clear', 'config:clear', 'config:cache', 'route:clear', 'view:clear',
            'queue:work', 'queue:restart', 'queue:failed', 'queue:retry', 'queue:prune-failed', 'queue:prune-batches',
            'down', 'up',
            'storage:link', 'storage:clear', 'storage:setup', 'storage:sync', 'storage:cleanup',
        ];

        if (!in_array($command, $allowedCommands)) {
            return response()->json([
                'success' => false,
                'error' => 'Command not allowed for security reasons.',
            ], 403);
        }

        try {
            // Capture command output
            $output = '';
            $returnCode = 0;
            
            // Execute the command
            \Artisan::call($command, [], $output);
            
            // Get the output
            $output = \Artisan::output();
            
            return response()->json([
                'success' => true,
                'output' => $output ?: 'Command executed successfully with no output.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Error executing command: ' . $e->getMessage(),
            ], 500);
        }
    }

    private function updateEnvFile($settings)
    {
        // Define the mappings for standard Laravel .env keys
        $standardMappings = [
            'APP_NAME' => $settings['site_name'],
            'APP_ENV' => $settings['app_env'],
            'APP_DEBUG' => $settings['app_debug'] ? 'true' : 'false',
            'APP_TIMEZONE' => $settings['timezone'],
            'MAIL_FROM_ADDRESS' => $settings['mail_from_address'],
            'MAIL_FROM_NAME' => $settings['mail_from_name'],
            'DB_CONNECTION' => $settings['database_connection'],
            'QUEUE_CONNECTION' => $settings['queue_connection'],
            'CACHE_DRIVER' => $settings['cache_driver'],
            'SESSION_DRIVER' => $settings['session_driver'],
            'MAIL_MAILER' => $settings['mail_driver'],
            'FILESYSTEM_DISK' => $settings['filesystem_disk'],
        ];

        // Define custom settings
        $customSettings = [
            'SITE_DESCRIPTION' => $settings['site_description'],
            'META_TITLE' => $settings['meta_title'],
            'META_DESCRIPTION' => $settings['meta_description'],
            'META_KEYWORDS' => $settings['meta_keywords'],
            'OG_IMAGE' => $settings['og_image'],
            'SITE_LOGO' => $settings['logo'],
            'SITE_FAVICON' => $settings['favicon'],
            'CONTACT_EMAIL' => $settings['contact_email'],
            'CONTACT_PHONE' => $settings['contact_phone'],
        ];

        // Add mail settings if SMTP is selected
        if ($settings['mail_driver'] === 'smtp') {
            $mailSettings = [
                'MAIL_HOST' => $settings['mail_host'],
                'MAIL_PORT' => $settings['mail_port'],
                'MAIL_USERNAME' => $settings['mail_username'],
                'MAIL_PASSWORD' => $settings['mail_password'],
                'MAIL_ENCRYPTION' => $settings['mail_encryption'],
                'MAIL_TIMEOUT' => $settings['mail_timeout'] ?? 60,
                'MAIL_VERIFY_PEER' => $settings['mail_verify_peer'] ?? true,
            ];
            $standardMappings = array_merge($standardMappings, $mailSettings);
        }

        // Update all settings
        EnvHelper::updateEnvKeys(array_merge($standardMappings, $customSettings));
    }

    /**
     * Send test email to verify email configuration
     */
    public function sendTestEmail(Request $request)
    {
        $request->validate([
            'test_email' => 'required|email|max:255',
            'test_name' => 'nullable|string|max:255',
            'use_current_settings' => 'boolean',
        ]);

        $emailService = new EmailSettingsService();
        
        try {
            $settings = [];
            
            // If not using current settings, validate and use provided settings
            if (!$request->get('use_current_settings', true)) {
                $providedSettings = $request->only([
                    'mail_driver', 'mail_host', 'mail_port', 'mail_username', 
                    'mail_password', 'mail_encryption', 'mail_from_address', 
                    'mail_from_name', 'mail_timeout'
                ]);
                
                $validation = $emailService->validateEmailSettings($providedSettings);
                if (!$validation['valid']) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Email settings validation failed.',
                        'errors' => $validation['errors']
                    ], 422);
                }
                
                $settings = $validation['normalized_settings'];
            }

            $result = $emailService->sendTestEmail(
                $request->input('test_email'),
                $request->input('test_name', 'Test User'),
                $settings
            );

            return response()->json($result);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send test email: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test SMTP connection without sending email
     */
    public function testSmtpConnection(Request $request)
    {
        $settings = $request->validate([
            'mail_host' => 'required|string|max:255',
            'mail_port' => 'required|integer|between:1,65535',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string|max:255',
            'mail_encryption' => 'nullable|string|in:tls,ssl,starttls',
            'mail_timeout' => 'nullable|integer|between:10,300',
        ]);

        $settings['mail_driver'] = 'smtp';
        
        $emailService = new EmailSettingsService();
        $result = $emailService->testSmtpConnection($settings);

        return response()->json($result);
    }

    /**
     * Get current email configuration status
     */
    public function getEmailStatus()
    {
        $emailService = new EmailSettingsService();
        $currentSettings = $emailService->getCurrentMailSettings();
        
        $status = [
            'configured' => !empty($currentSettings['mail_from_address']) && $currentSettings['mail_from_address'] !== 'hello@example.com',
            'driver' => $currentSettings['mail_driver'],
            'smtp_configured' => false,
            'test_enabled' => config('mail.test.enabled', true),
        ];

        if ($currentSettings['mail_driver'] === 'smtp') {
            $status['smtp_configured'] = !empty($currentSettings['mail_host']) && !empty($currentSettings['mail_port']);
        }

        return response()->json([
            'status' => $status,
            'settings' => $currentSettings
        ]);
    }

    /**
     * Get queue status and statistics
     */
    public function getQueueStatus()
    {
        try {
            $queueConnection = config('queue.default');
            $queueTable = config('queue.connections.database.table', 'jobs');
            $failedTable = config('queue.failed.table', 'failed_jobs');
            
            // Get queue statistics
            $pendingJobs = \DB::table($queueTable)->count();
            $failedJobs = \DB::table($failedTable)->count();
            
            // Get recent failed jobs
            $recentFailures = \DB::table($failedTable)
                ->orderBy('failed_at', 'desc')
                ->limit(5)
                ->get(['id', 'queue', 'failed_at', 'exception'])
                ->map(function ($job) {
                    return [
                        'id' => $job->id,
                        'queue' => $job->queue,
                        'failed_at' => $job->failed_at,
                        'exception' => substr($job->exception, 0, 100) . '...'
                    ];
                });

            $status = [
                'connection' => $queueConnection,
                'pending_jobs' => $pendingJobs,
                'failed_jobs' => $failedJobs,
                'recent_failures' => $recentFailures,
                'worker_status' => 'unknown', // Will be updated if we can check
            ];

            return response()->json([
                'success' => true,
                'status' => $status
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to get queue status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Process queue jobs manually (for immediate processing)
     */
    public function processQueue(Request $request)
    {
        $request->validate([
            'action' => 'required|string|in:work,retry-all,clear-failed,restart',
            'queue' => 'nullable|string|max:255',
        ]);

        $action = $request->input('action');
        $queue = $request->input('queue', 'default');

        try {
            switch ($action) {
                case 'work':
                    // Process jobs with stop-when-empty (safe for shared hosting)
                    $output = '';
                    \Artisan::call('queue:work', [
                        '--stop-when-empty' => true,
                        '--queue' => $queue,
                        '--timeout' => 60,
                        '--tries' => 3
                    ], $output);
                    $message = 'Queue processed successfully. Jobs processed: ' . \Artisan::output();
                    break;

                case 'retry-all':
                    \Artisan::call('queue:retry', ['id' => 'all']);
                    $message = 'All failed jobs have been retried.';
                    break;

                case 'clear-failed':
                    \Artisan::call('queue:flush');
                    $message = 'All failed jobs have been cleared.';
                    break;

                case 'restart':
                    \Artisan::call('queue:restart');
                    $message = 'Queue workers have been restarted.';
                    break;

                default:
                    throw new \Exception('Invalid action specified.');
            }

            return response()->json([
                'success' => true,
                'message' => $message
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to process queue: ' . $e->getMessage()
            ], 500);
        }
    }

    private function updateConfigCache()
    {
        // Clear config cache
        \Artisan::call('config:clear');
        
        // Rebuild config cache
        \Artisan::call('config:cache');
    }
}