<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\Permission;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;

class RoleAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:roles.view')->only(['index', 'show']);
        $this->middleware('permission:roles.edit')->only(['edit', 'update']);
        $this->middleware('permission:roles.assign')->only(['assignUsers', 'removeUsers']);
        $this->middleware('permission:permissions.assign')->only(['assignPermissions', 'removePermissions', 'syncPermissions']);
    }

    public function index(Request $request)
    {
        $query = Role::with('users');

        // Filtering
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('display_name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        if ($request->filled('is_active') && $request->is_active !== 'all') {
            $query->where('is_active', $request->is_active === 'true');
        }

        if ($request->filled('is_system') && $request->is_system !== 'all') {
            $query->where('is_system', $request->is_system === 'true');
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'priority');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $roles = $query->paginate($request->get('per_page', 15))->withQueryString();

        return Inertia::render('admin/roles/index', [
            'roles' => $roles,
            'filters' => $request->only(['search', 'is_active', 'is_system', 'sort_by', 'sort_order']),
        ]);
    }

    public function show(Role $role)
    {
        $role->load('users');
        
        return Inertia::render('admin/roles/show', [
            'role' => $role,
        ]);
    }

    public function create()
    {
        $allPermissions = Permission::active()->ordered()->get()->groupBy('category');
        
        return Inertia::render('admin/roles/create', [
            'permissions' => $allPermissions,
            'categories' => Permission::getCategories(),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:roles,name',
            'display_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_system' => 'boolean',
            'is_active' => 'boolean',
            'priority' => 'integer|min:0',
            'color' => 'nullable|string|max:255',
            'icon' => 'nullable|string|max:255',
            'permission_ids' => 'nullable|array',
            'permission_ids.*' => 'exists:permissions,id',
        ]);

        // Create the role
        $role = Role::create([
            'name' => $validated['name'],
            'display_name' => $validated['display_name'],
            'description' => $validated['description'],
            'is_system' => $validated['is_system'] ?? false,
            'is_active' => $validated['is_active'] ?? true,
            'priority' => $validated['priority'] ?? 0,
            'color' => $validated['color'],
            'icon' => $validated['icon'] ?? 'shield',
        ]);

        // Assign permissions
        if (isset($validated['permission_ids'])) {
            $role->permissions()->sync($validated['permission_ids']);
        }

        return redirect()->route('admin.roles.show', $role)
                        ->with('success', 'Role created successfully!');
    }

    public function edit(Role $role)
    {
        $role->load('permissions');
        $users = User::select('id', 'name', 'email')->get();
        $allPermissions = Permission::active()->ordered()->get()->groupBy('category');
        
        return Inertia::render('admin/roles/edit', [
            'role' => $role,
            'users' => $users,
            'permissions' => $allPermissions,
            'categories' => Permission::getCategories(),
        ]);
    }

    public function update(Request $request, Role $role)
    {
        $validated = $request->validate([
            'display_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'permissions' => 'nullable|array',
            'permission_ids' => 'nullable|array',
            'permission_ids.*' => 'exists:permissions,id',
            'is_active' => 'boolean',
            'color' => 'nullable|string|max:255',
            'icon' => 'nullable|string|max:255',
        ]);

        // Update role basic info
        $role->update([
            'display_name' => $validated['display_name'],
            'description' => $validated['description'],
            'permissions' => $validated['permissions'] ?? [],
            'is_active' => $validated['is_active'] ?? true,
            'color' => $validated['color'],
            'icon' => $validated['icon'],
        ]);

        // Sync permissions via relationships
        if (isset($validated['permission_ids'])) {
            $role->permissions()->sync($validated['permission_ids']);
        }

        return redirect()->route('admin.roles.index')->with('success', 'Role updated successfully!');
    }

    public function assignUsers(Request $request, Role $role)
    {
        $validated = $request->validate([
            'user_ids' => 'required|array',
            'user_ids.*' => 'exists:users,id',
            'is_primary' => 'boolean',
        ]);

        foreach ($validated['user_ids'] as $userId) {
            $role->assignToUser(User::find($userId), $validated['is_primary'] ?? false);
        }

        return redirect()->back()->with('success', 'Users assigned to role successfully!');
    }

    public function removeUsers(Request $request, Role $role)
    {
        $validated = $request->validate([
            'user_ids' => 'required|array',
            'user_ids.*' => 'exists:users,id',
        ]);

        foreach ($validated['user_ids'] as $userId) {
            $role->removeFromUser(User::find($userId));
        }

        return redirect()->back()->with('success', 'Users removed from role successfully!');
    }

    public function assignPermissions(Request $request, Role $role)
    {
        $validated = $request->validate([
            'permission_ids' => 'required|array',
            'permission_ids.*' => 'exists:permissions,id',
        ]);

        $role->permissions()->syncWithoutDetaching($validated['permission_ids']);

        return response()->json([
            'success' => true,
            'message' => 'Permissions assigned to role successfully!',
        ]);
    }

    public function removePermissions(Request $request, Role $role)
    {
        $validated = $request->validate([
            'permission_ids' => 'required|array',
            'permission_ids.*' => 'exists:permissions,id',
        ]);

        $role->permissions()->detach($validated['permission_ids']);

        return response()->json([
            'success' => true,
            'message' => 'Permissions removed from role successfully!',
        ]);
    }

    public function syncPermissions(Request $request, Role $role)
    {
        $validated = $request->validate([
            'permission_ids' => 'array',
            'permission_ids.*' => 'exists:permissions,id',
        ]);

        $role->permissions()->sync($validated['permission_ids'] ?? []);

        return response()->json([
            'success' => true,
            'message' => 'Role permissions synchronized successfully!',
        ]);
    }

    public function destroy(Role $role)
    {
        // Prevent deletion of system roles
        if ($role->is_system) {
            return redirect()->back()->with('error', 'Cannot delete system roles!');
        }

        // Check if role has users assigned
        if ($role->users()->count() > 0) {
            return redirect()->back()->with('error', 'Cannot delete role with assigned users!');
        }

        $role->delete();

        return redirect()->route('admin.roles.index')
                        ->with('success', 'Role deleted successfully!');
    }

    public function toggleStatus(Role $role)
    {
        $role->update(['is_active' => !$role->is_active]);

        $status = $role->is_active ? 'activated' : 'deactivated';
        return redirect()->back()->with('success', "Role {$status} successfully!");
    }

    public function bulkUpdate(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:roles,id',
            'action' => 'required|in:activate,deactivate,delete',
        ]);

        $roles = Role::whereIn('id', $validated['ids']);

        switch ($validated['action']) {
            case 'activate':
                $roles->update(['is_active' => true]);
                $message = 'Roles activated successfully!';
                break;
            case 'deactivate':
                $roles->where('is_system', false)->update(['is_active' => false]);
                $message = 'Custom roles deactivated successfully!';
                break;
            case 'delete':
                $roles->where('is_system', false)->whereDoesntHave('users')->delete();
                $message = 'Custom roles without users deleted successfully!';
                break;
        }

        return redirect()->back()->with('success', $message);
    }

    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:roles,id',
        ]);

        $roles = Role::whereIn('id', $validated['ids'])
                    ->where('is_system', false)
                    ->whereDoesntHave('users');

        $deletedCount = $roles->count();
        $roles->delete();

        return redirect()->back()->with('success', "{$deletedCount} roles deleted successfully!");
    }
}
