<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PrizeAward;
use App\Models\Tournament;
use Illuminate\Http\Request;
use Inertia\Inertia;

class PrizeAwardController extends Controller
{
    public function index(Request $request)
    {
        $filters = $request->only(['search', 'category', 'tournament', 'status']);

        $prizeAwards = PrizeAward::query()
            ->with('tournament')
            ->when($filters['search'] ?? null, function ($query, $search) {
                $query->where('title', 'like', '%' . $search . '%')
                      ->orWhere('description', 'like', '%' . $search . '%');
            })
            ->when($filters['category'] ?? null, function ($query, $category) {
                $query->where('category', $category);
            })
            ->when($filters['tournament'] ?? null, function ($query, $tournament) {
                $query->where('tournament_id', $tournament);
            })
            ->when($filters['status'] ?? null, function ($query, $status) {
                $query->where('is_active', $status === 'active');
            })
            ->ordered()
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('admin/prize-awards/Index', [
            'prizeAwards' => $prizeAwards,
            'filters' => $filters,
            'categories' => PrizeAward::getCategories(),
            'prizeTypes' => PrizeAward::getPrizeTypes(),
            'positions' => PrizeAward::getPositions(),
            'tournaments' => Tournament::select('id', 'name')->get(),
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/prize-awards/Create', [
            'categories' => PrizeAward::getCategories(),
            'prizeTypes' => PrizeAward::getPrizeTypes(),
            'positions' => PrizeAward::getPositions(),
            'tournaments' => Tournament::select('id', 'name')->get(),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|exists:tournaments,id',
            'category' => 'required|string|max:255',
            'position' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'prize_value' => 'nullable|numeric|min:0',
            'prize_type' => 'required|string|max:255',
            'additional_benefits' => 'nullable|string',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        PrizeAward::create($validated);

        return redirect()->route('admin.prize-awards.index')
            ->with('success', 'Prize award created successfully.');
    }

    public function show(PrizeAward $prizeAward)
    {
        $prizeAward->load('tournament');

        return Inertia::render('admin/prize-awards/Show', [
            'prizeAward' => $prizeAward,
            'categories' => PrizeAward::getCategories(),
            'prizeTypes' => PrizeAward::getPrizeTypes(),
            'positions' => PrizeAward::getPositions(),
        ]);
    }

    public function edit(PrizeAward $prizeAward)
    {
        return Inertia::render('admin/prize-awards/Edit', [
            'prizeAward' => $prizeAward,
            'categories' => PrizeAward::getCategories(),
            'prizeTypes' => PrizeAward::getPrizeTypes(),
            'positions' => PrizeAward::getPositions(),
            'tournaments' => Tournament::select('id', 'name')->get(),
        ]);
    }

    public function update(Request $request, PrizeAward $prizeAward)
    {
        $validated = $request->validate([
            'tournament_id' => 'required|exists:tournaments,id',
            'category' => 'required|string|max:255',
            'position' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'prize_value' => 'nullable|numeric|min:0',
            'prize_type' => 'required|string|max:255',
            'additional_benefits' => 'nullable|string',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        $prizeAward->update($validated);

        return redirect()->route('admin.prize-awards.index')
            ->with('success', 'Prize award updated successfully.');
    }

    public function destroy(PrizeAward $prizeAward)
    {
        $prizeAward->delete();

        return redirect()->route('admin.prize-awards.index')
            ->with('success', 'Prize award deleted successfully.');
    }

    public function toggleStatus(PrizeAward $prizeAward)
    {
        $prizeAward->update(['is_active' => !$prizeAward->is_active]);

        return redirect()->back()
            ->with('success', 'Prize award status updated successfully.');
    }

    public function bulkAction(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|in:activate,deactivate,delete',
            'prize_award_ids' => 'required|array|min:1',
            'prize_award_ids.*' => 'exists:prize_awards,id',
        ]);

        $prizeAwards = PrizeAward::whereIn('id', $validated['prize_award_ids']);

        switch ($validated['action']) {
            case 'activate':
                $prizeAwards->update(['is_active' => true]);
                $message = 'Selected prize awards have been activated.';
                break;
            case 'deactivate':
                $prizeAwards->update(['is_active' => false]);
                $message = 'Selected prize awards have been deactivated.';
                break;
            case 'delete':
                $prizeAwards->delete();
                $message = 'Selected prize awards have been deleted.';
                break;
        }

        return redirect()->back()->with('success', $message);
    }
}
