<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Player;
use App\Models\Tournament;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class PlayerAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:players.view')->only(['index', 'show', 'statistics', 'export', 'generateCertificate', 'printCertificate', 'downloadCertificate', 'generateNewCertificate']);
        $this->middleware('permission:players.create')->only(['create', 'store']);
        $this->middleware('permission:players.edit')->only(['edit', 'update', 'bulkUpdate']);
        $this->middleware('permission:players.delete')->only(['destroy', 'destroyMultiple']);
    }

    public function index(Request $request)
    {
        $query = Player::query();
        
        // Basic filters
        if ($request->filled('search')) {
            $query->where(function($q) use ($request) {
                $q->where('display_name', 'like', '%' . $request->search . '%')
                  ->orWhere('bio', 'like', '%' . $request->search . '%')
                  ->orWhereHas('user', function($userQuery) use ($request) {
                      $userQuery->where('name', 'like', '%' . $request->search . '%')
                               ->orWhere('email', 'like', '%' . $request->search . '%');
                  });
            });
        }
        
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        
        if ($request->filled('verification_status') && $request->verification_status !== 'all') {
            $query->where('verification_status', $request->verification_status);
        }
        
        if ($request->filled('tournament_id') && $request->tournament_id !== 'all') {
            $query->where('tournament_id', $request->tournament_id);
        }
        
        if ($request->filled('player_type') && $request->player_type !== 'all') {
            $query->where('player_type', $request->player_type);
        }
        
        if ($request->filled('experience_level') && $request->experience_level !== 'all') {
            $query->where('experience_level', $request->experience_level);
        }
        
        if ($request->filled('country') && $request->country !== 'all') {
            $query->where('country', $request->country);
        }
        
        if ($request->filled('gender') && $request->gender !== 'all') {
            $query->where('gender', $request->gender);
        }
        
        // Rating filters
        if ($request->filled('min_rating')) {
            $query->where('global_rating', '>=', $request->min_rating);
        }
        
        if ($request->filled('max_rating')) {
            $query->where('global_rating', '<=', $request->max_rating);
        }
        
        // Win rate filters
        if ($request->filled('min_win_rate')) {
            $query->where('tournament_win_rate', '>=', $request->min_win_rate);
        }
        
        if ($request->filled('max_win_rate')) {
            $query->where('tournament_win_rate', '<=', $request->max_win_rate);
        }
        
        // Date filters
        if ($request->filled('registered_from')) {
            $query->where('registered_at', '>=', $request->registered_from);
        }
        
        if ($request->filled('registered_to')) {
            $query->where('registered_at', '<=', $request->registered_to);
        }
        
        if ($request->filled('last_active_from')) {
            $query->where('last_active_at', '>=', $request->last_active_from);
        }
        
        if ($request->filled('last_active_to')) {
            $query->where('last_active_at', '<=', $request->last_active_to);
        }
        
        // Sorting
        $sortBy = $request->get('sort_by', 'id');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);
        
        $players = $query->with(['user', 'tournament'])->paginate(10)->withQueryString();
        
        // Get filter options
        $tournaments = Tournament::select('id', 'name')->get();
        $users = User::select('id', 'name', 'email')->get();
        $countries = Player::distinct()->pluck('country')->filter()->values();
        $playerTypes = ['amateur', 'semi_pro', 'professional', 'legend'];
        $experienceLevels = ['beginner', 'intermediate', 'advanced', 'expert'];
        $genders = ['male', 'female', 'other', 'prefer_not_to_say'];
        $statuses = ['registered', 'active', 'eliminated', 'withdrawn', 'suspended', 'banned'];
        $verificationStatuses = ['unverified', 'pending', 'verified', 'rejected'];
        
        return Inertia::render('admin/players/index', [
            'players' => $players,
            'filters' => $request->only([
                'search', 'status', 'verification_status', 'tournament_id', 
                'player_type', 'experience_level', 'country', 'gender',
                'min_rating', 'max_rating', 'min_win_rate', 'max_win_rate',
                'registered_from', 'registered_to', 'last_active_from', 'last_active_to',
                'sort_by', 'sort_order'
            ]),
            'tournaments' => $tournaments,
            'users' => $users,
            'filterOptions' => [
                'countries' => $countries,
                'playerTypes' => $playerTypes,
                'experienceLevels' => $experienceLevels,
                'genders' => $genders,
                'statuses' => $statuses,
                'verificationStatuses' => $verificationStatuses,
            ],
        ]);
    }

    public function create()
    {
        $tournaments = Tournament::select('id', 'name')->get();
        $users = User::select('id', 'name', 'email')->get();
        
        return Inertia::render('admin/players/create', [
            'tournaments' => $tournaments,
            'users' => $users,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'display_name' => 'required|string|max:255',
            'bio' => 'nullable|string',
            'avatar' => 'nullable|string',
            'country' => 'nullable|string|max:100',
            'city' => 'nullable|string|max:100',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other,prefer_not_to_say',
            'phone' => 'nullable|string|max:20',
            'discord_username' => 'nullable|string|max:100',
            'steam_id' => 'nullable|string|max:100',
            'twitch_username' => 'nullable|string|max:100',
            'youtube_channel' => 'nullable|string|max:100',
            'twitter_handle' => 'nullable|string|max:100',
            'instagram_handle' => 'nullable|string|max:100',
            'website' => 'nullable|url|max:255',
            'player_type' => 'required|in:amateur,semi_pro,professional,legend',
            'experience_level' => 'required|in:beginner,intermediate,advanced,expert',
            'primary_game' => 'nullable|string|max:100',
            'game_preferences' => 'nullable|array',
            'status' => 'required|in:registered,active,eliminated,withdrawn,suspended,banned',
            'verification_status' => 'required|in:unverified,pending,verified,rejected',
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'user_id' => 'required|integer|exists:users,id',
            'global_rating' => 'nullable|numeric|min:0|max:9999',
            'tournament_rating' => 'nullable|numeric|min:0|max:9999',
            'seeding_position' => 'nullable|integer|min:1',
            'is_public_profile' => 'boolean',
            'allow_messages' => 'boolean',
            'show_contact_info' => 'boolean',
        ]);
        
        // Set default values
        $validated['registered_at'] = now();
        $validated['last_active_at'] = now();
        
        Player::create($validated);
        return redirect()->route('admin.players.index')->with('success', 'Player created successfully!');
    }

    public function edit(Player $player)
    {
        $tournaments = Tournament::select('id', 'name')->get();
        $users = User::select('id', 'name', 'email')->get();
        
        return Inertia::render('admin/players/edit', [
            'player' => $player->load(['user', 'tournament']),
            'tournaments' => $tournaments,
            'users' => $users,
        ]);
    }

    public function update(Request $request, Player $player)
    {
        $validated = $request->validate([
            'display_name' => 'required|string|max:255',
            'bio' => 'nullable|string',
            'avatar' => 'nullable|string',
            'country' => 'nullable|string|max:100',
            'city' => 'nullable|string|max:100',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other,prefer_not_to_say',
            'phone' => 'nullable|string|max:20',
            'discord_username' => 'nullable|string|max:100',
            'steam_id' => 'nullable|string|max:100',
            'twitch_username' => 'nullable|string|max:100',
            'youtube_channel' => 'nullable|string|max:100',
            'twitter_handle' => 'nullable|string|max:100',
            'instagram_handle' => 'nullable|string|max:100',
            'website' => 'nullable|url|max:255',
            'player_type' => 'required|in:amateur,semi_pro,professional,legend',
            'experience_level' => 'required|in:beginner,intermediate,advanced,expert',
            'primary_game' => 'nullable|string|max:100',
            'game_preferences' => 'nullable|array',
            'status' => 'required|in:registered,active,eliminated,withdrawn,suspended,banned',
            'verification_status' => 'required|in:unverified,pending,verified,rejected',
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'user_id' => 'required|integer|exists:users,id',
            'global_rating' => 'nullable|numeric|min:0|max:9999',
            'tournament_rating' => 'nullable|numeric|min:0|max:9999',
            'seeding_position' => 'nullable|integer|min:1',
            'is_public_profile' => 'boolean',
            'allow_messages' => 'boolean',
            'show_contact_info' => 'boolean',
        ]);
        
        $player->update($validated);
        
        // Update rankings if ratings changed
        if ($request->has('global_rating')) {
            $player->updateGlobalRank();
        }
        if ($request->has('tournament_rating')) {
            $player->updateTournamentRank();
        }
        
        return redirect()->route('admin.players.index')->with('success', 'Player updated successfully!');
    }

    public function show(Player $player)
    {
        $player->load(['user', 'tournament', 'matchesAsPlayer1', 'matchesAsPlayer2', 'matchesAsWinner']);
        
        // Get advanced statistics
        $stats = $player->getDisplayStats();
        $performanceMetrics = $player->getPerformanceMetrics();
        $socialLinks = $player->getSocialLinks();
        $achievements = $player->achievements ?? [];
        
        return Inertia::render('admin/players/show', [
            'player' => $player,
            'stats' => $stats,
            'performanceMetrics' => $performanceMetrics,
            'socialLinks' => $socialLinks,
            'achievements' => $achievements,
        ]);
    }

    public function destroy(Player $player)
    {
        $player->delete();
        return redirect()->route('admin.players.index')->with('success', 'Player deleted successfully!');
    }

    public function destroyMultiple(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:players,id'
        ]);
        Player::whereIn('id', $validated['ids'])->delete();
        return redirect()->route('admin.players.index')->with('success', 'Players deleted successfully!');
    }

    // Additional advanced methods
    public function statistics()
    {
        $stats = [
            'total_players' => Player::count(),
            'active_players' => Player::where('status', 'active')->count(),
            'verified_players' => Player::where('verification_status', 'verified')->count(),
            'professional_players' => Player::whereIn('player_type', ['professional', 'legend'])->count(),
            'average_rating' => Player::avg('global_rating'),
            'average_win_rate' => Player::avg('tournament_win_rate'),
            'top_rated_player' => Player::orderBy('global_rating', 'desc')->first(),
            'most_active_player' => Player::orderBy('tournament_matches_played', 'desc')->first(),
        ];
        
        return response()->json($stats);
    }

    public function export(Request $request)
    {
        $players = Player::with(['user', 'tournament'])->get();
        
        $csvData = [];
        $csvData[] = [
            'ID', 'Display Name', 'User', 'Tournament', 'Status', 'Player Type', 
            'Experience Level', 'Country', 'City', 'Global Rating', 'Tournament Rating',
            'Matches Played', 'Matches Won', 'Win Rate', 'Total Earnings',
            'Verification Status', 'Registered At', 'Last Active'
        ];
        
        foreach ($players as $player) {
            $csvData[] = [
                $player->id,
                $player->display_name,
                $player->user->name,
                $player->tournament->name,
                $player->status,
                $player->player_type,
                $player->experience_level,
                $player->country,
                $player->city,
                $player->global_rating,
                $player->tournament_rating,
                $player->tournament_matches_played,
                $player->tournament_matches_won,
                $player->tournament_win_rate,
                $player->total_earnings,
                $player->verification_status,
                $player->registered_at,
                $player->last_active_at,
            ];
        }
        
        $filename = 'players_export_' . now()->format('Y-m-d_H-i-s') . '.csv';
        
        return response()->streamDownload(function () use ($csvData) {
            $output = fopen('php://output', 'w');
            foreach ($csvData as $row) {
                fputcsv($output, $row);
            }
            fclose($output);
        }, $filename);
    }

    /**
     * Generate Digital ID for a player
     */
    public function generateDigitalId(Player $player)
    {
        // Generate unique Digital ID
        $digitalId = 'PID-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id . time()), 0, 8));
        
        // Update player with Digital ID if not already set
        if (!$player->digital_id) {
            $player->update(['digital_id' => $digitalId]);
        }
        
        return response()->json([
            'success' => true,
            'digital_id' => $player->digital_id,
            'message' => 'Digital ID generated successfully!'
        ]);
    }

    /**
     * View Digital ID (front side)
     */
    public function viewDigitalId(Player $player)
    {
        $player->load(['user', 'tournament']);
        
        if (!$player->digital_id) {
            return redirect()->back()->with('error', 'Digital ID not generated yet. Please generate it first.');
        }

        $data = [
            'player' => $player,
            'digital_id' => $player->digital_id,
            'generated_at' => now()->format('F j, Y'),
            'valid_until' => now()->addYears(2)->format('F j, Y'),
            'qr_code' => $this->generateQRCode($player->digital_id),
            // Additional data for the ID card template
            'player_name' => $player->user->name,
            'player_email' => $player->user->email,
            'player_phone' => $player->phone ?: 'Not provided',
            'player_type' => ucfirst(str_replace('_', ' ', $player->player_type)),
            'tournament_edition' => $player->tournament->name ?? '1st Edition',
            'verification_status' => ucfirst($player->verification_status),
            'registration_status' => ucfirst($player->status),
            'player_id' => $player->id,
            'tournament_year' => now()->year,
            'barcode_text' => 'ID-' . str_pad($player->id, 3, '0', STR_PAD_LEFT) . '-' . now()->year,
            'generated_date' => now()->format('F j, Y'),
            'valid_until_date' => now()->addYears(2)->format('F j, Y'),
            'tournament' => $player->tournament->name ?? '1st Edition',
            'registration' => \Carbon\Carbon::parse($player->registered_at)->format('M j, Y'),
            'id' => $player->id,
            'validity_period' => '2 Years',
        ];

        return Inertia::render('admin/players/digital-id', $data);
    }

    /**
     * Print Digital ID as PDF
     */
    public function printDigitalId(Player $player)
    {
        $player->load(['user', 'tournament']);
        
        if (!$player->digital_id) {
            return redirect()->back()->with('error', 'Digital ID not generated yet. Please generate it first.');
        }

        $data = [
            'player' => $player,
            'digital_id' => $player->digital_id,
            'generated_at' => now()->format('F j, Y'),
            'valid_until' => now()->addYears(2)->format('F j, Y'),
            'qr_code' => $this->generateQRCode($player->digital_id),
            'player_name' => $player->user->name,
            'player_email' => $player->user->email,
            'player_phone' => $player->phone ?: 'Not provided',
            'player_type' => ucfirst(str_replace('_', ' ', $player->player_type)),
            'tournament_edition' => $player->tournament->name ?? '1st Edition',
            'verification_status' => ucfirst($player->verification_status),
            'registration_status' => ucfirst($player->status),
            'player_id' => $player->id,
            'tournament_year' => now()->year,
            'barcode_text' => 'ID-' . str_pad($player->id, 3, '0', STR_PAD_LEFT) . '-' . now()->year,
            'generated_date' => now()->format('F j, Y'),
            'valid_until_date' => now()->addYears(2)->format('F j, Y'),
            'tournament' => $player->tournament->name ?? '1st Edition',
            'registration' => \Carbon\Carbon::parse($player->registered_at)->format('M j, Y'),
            'id' => $player->id,
            'validity_period' => '2 Years',
            'columns' => 2,
            'rows' => 2,
            'photo' => $player->avatar ?? asset('player.jpg'),
        ];

        $html = view('players.id', $data)->render();

        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="digital-id-' . str_replace(' ', '-', strtolower($player->display_name ?? $player->user->name)) . '.html"')
            ->header('X-Inertia', 'false');
    }

    /**
     * View Player ID Card Template
     */
    public function viewIdCard(Player $player)
    {
        $player->load(['user', 'tournament']);
        
        $data = [
            'player' => $player,
            'digital_id' => $player->digital_id ?: 'PID-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id . time()), 0, 8)),
            'generated_at' => now()->format('F j, Y'),
            'valid_until' => now()->addYears(2)->format('F j, Y'),
            'qr_code' => $this->generateQRCode($player->digital_id ?: 'PID-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id . time()), 0, 8))),
            // Additional data for the template
            'player_name' => $player->user->name,
            'player_email' => $player->user->email,
            'player_phone' => $player->phone ?: 'Not provided',
            'player_type' => ucfirst(str_replace('_', ' ', $player->player_type)),
            'tournament_edition' => $player->tournament->name ?? '1st Edition',
            'verification_status' => ucfirst($player->verification_status),
            'registration_status' => ucfirst($player->status),
            'player_id' => $player->id,
            'tournament_year' => now()->year,
            'barcode_text' => 'ID-' . str_pad($player->id, 3, '0', STR_PAD_LEFT) . '-' . now()->year,
            'generated_date' => now()->format('F j, Y'),
            'valid_until_date' => now()->addYears(2)->format('F j, Y'),
            'tournament' => $player->tournament->name ?? '1st Edition',
            'registration' => \Carbon\Carbon::parse($player->registered_at)->format('M j, Y'),
            'id' => $player->id,
            'validity_period' => '2 Years',
            'columns' => 2,
            'rows' => 1,
            'photo' => $player->avatar ?? asset('player.jpg'),
        ];

        return view('players.id', $data);
    }

    /**
     * Generate QR Code for Digital ID
     */
    private function generateQRCode($digitalId)
    {
        // QR code generation using QR Server API (more reliable than Google Charts)
        $qrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=150x150&data=' . urlencode($digitalId) . '&format=png&margin=2';
        return $qrUrl;
    }

    /**
     * Resend Digital ID to player via email
     */
    public function resendDigitalId(Player $player)
    {
        $player->load(['user', 'tournament', 'team']);
        
        if (!$player->digital_id) {
            return redirect()->back()->with('error', 'Digital ID not generated yet. Please generate it first.');
        }

        try {
            \Log::info('Starting digital ID resend process', [
                'player_id' => $player->id,
                'player_name' => $player->user->name,
                'digital_id' => $player->digital_id,
                'user_email' => $player->user->email ?? 'No email'
            ]);

            // Generate PDF for email attachment
            $data = [
                'player' => $player,
                'digital_id' => $player->digital_id,
                'generated_at' => now()->format('F j, Y'),
                'valid_until' => now()->addYears(2)->format('F j, Y'),
                'qr_code' => $this->generateQRCode($player->digital_id),
            ];

            $pdf = Pdf::loadView('players.digital-id-pdf', $data);
            $pdf->setOptions([
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
            ]);
            $pdf->getDomPDF()->setHttpContext(stream_context_create([
                'ssl' => [
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                ],
            ]));
            $pdf->setPaper('a4', 'landscape');
            
            $pdfContent = $pdf->output();
            
            \Log::info('PDF generated successfully', [
                'player_id' => $player->id,
                'pdf_size' => strlen($pdfContent)
            ]);
            
            // Send notification with PDF attachment
            try {
                $player->user->notify(new \App\Notifications\User\PlayerDigitalIdGeneratedNotification($player, $pdfContent));
                
                \Log::info('Notification sent successfully', [
                    'player_id' => $player->id,
                    'user_id' => $player->user->id
                ]);
            } catch (\Exception $notificationError) {
                \Log::error('Failed to send notification', [
                    'player_id' => $player->id,
                    'error' => $notificationError->getMessage()
                ]);
                
                // Still return success since PDF was generated
                return redirect()->back()->with('success', 'Digital ID PDF generated successfully, but notification failed to send.');
            }
            
            return redirect()->back()->with('success', 'Digital ID has been resent to the player via email with PDF attachment!');
            
        } catch (\Exception $e) {
            \Log::error('Failed to resend digital ID', [
                'player_id' => $player->id,
                'error' => $e->getMessage()
            ]);
            
            return redirect()->back()->with('error', 'Failed to resend digital ID. Please try again.');
        }
    }

    /**
     * Generate Certificate for a player (New Design)
     */
    public function generateCertificate(Player $player)
    {
        $player->load(['user', 'tournament']);
        
        // Generate unique Certificate ID
        $certificateId = 'CERT-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id . time()), 0, 8));
        
        $data = [
            'player_name' => $player->user->name,
            'team_name' => $player->team_name ?? '',
            'position' => $player->position ?? 'Participant',
            'jersey_number' => $player->jersey_number ?? null,
            'tournament_matches_played' => $player->tournament_matches_played ?? 0,
            'tournament_win_rate' => $player->tournament_win_rate ?? 0,
            'global_rating' => $player->global_rating ?? 0,
            'digital_id' => $certificateId,
            'issued_date' => now()->format('F j, Y'),
            'tournament_name' => $player->tournament->name ?? 'The Milimus Tournament Cup',
            'description' => 'has successfully participated in the ' . $player->tournament->name . ' edition tournament held from ' . Carbon::parse($player->tournament->start_date)->format('F j, Y') . ' to ' . Carbon::parse($player->tournament->end_date)->format('F j, Y') . ' and has demonstrated exceptional sportsmanship, skill, and dedication throughout the competition.',
            'qr_code_url' => 'https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=https://example.com/verify/' . $certificateId,
            'verification_status' => $player->verification_status ?? 'verified',
        ];

        return view('certificates.player-participation', $data);
    }

    /**
     * Generate New Certificate for a player (Same as generateCertificate but with different route)
     */
    public function generateNewCertificate(Player $player)
    {
        return $this->generateCertificate($player);
    }

    /**
     * Print Certificate as PDF (New Design)
     */
    public function printCertificate(Player $player)
    {
        $player->load(['user', 'tournament']);
        
        // Generate unique Certificate ID
        $certificateId = 'CERT-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id . time()), 0, 8));
        
        $data = [
            'player_name' => $player->user->name,
            'team_name' => $player->team_name ?? '',
            'position' => $player->position ?? 'Participant',
            'jersey_number' => $player->jersey_number ?? null,
            'tournament_matches_played' => $player->tournament_matches_played ?? 0,
            'tournament_win_rate' => $player->tournament_win_rate ?? 0,
            'global_rating' => $player->global_rating ?? 0,
            'digital_id' => $certificateId,
            'issued_date' => now()->format('F j, Y'),
            'tournament_name' => $player->tournament->name ?? 'The Milimus Tournament Cup',
            'description' => 'has successfully participated in the ' . $player->tournament->name . ' edition tournament held from ' . Carbon::parse($player->tournament->start_date)->format('F j, Y') . ' to ' . Carbon::parse($player->tournament->end_date)->format('F j, Y') . ' and has demonstrated exceptional sportsmanship, skill, and dedication throughout the competition.',
            'qr_code_url' => 'https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=https://example.com/verify/' . $certificateId,
            'verification_status' => $player->verification_status ?? 'verified',
        ];

        try {
            $pdf = Pdf::loadView('certificates.player-participation', $data);
            
            // Ensure remote assets (QR Server API) load correctly
            $pdf->setOptions([
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'isPhpEnabled' => true,
                'isFontSubsettingEnabled' => false,
            ]);
            
            // Relax SSL for remote image fetching if needed
            $pdf->getDomPDF()->setHttpContext(stream_context_create([
                'ssl' => [
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                ],
                'http' => [
                    'timeout' => 30,
                ],
            ]));

            $pdf->setPaper('a4', 'landscape');
            
            $filename = 'Certificate_' . ($player->user->name) . '_' . now()->format('Y-m-d') . '.pdf';
            $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
            
            return $pdf->download($filename);
        } catch (\Exception $e) {
            \Log::error('PDF Generation Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate PDF: ' . $e->getMessage()], 500);
        }
    }

    public function downloadCertificate(Player $player)
    {
        $player->load(['user', 'tournament']);
        
        // Generate unique Certificate ID
        $certificateId = 'CERT-' . str_pad($player->id, 6, '0', STR_PAD_LEFT) . '-' . strtoupper(substr(md5($player->id . time()), 0, 8));
        
        $data = [
            'player_name' => $player->user->name,
            'team_name' => $player->team_name ?? '',
            'position' => $player->position ?? 'Participant',
            'jersey_number' => $player->jersey_number ?? null,
            'tournament_matches_played' => $player->tournament_matches_played ?? 0,
            'tournament_win_rate' => $player->tournament_win_rate ?? 0,
            'global_rating' => $player->global_rating ?? 0,
            'digital_id' => $certificateId,
            'issued_date' => now()->format('F j, Y'),
            'tournament_name' => $player->tournament->name ?? 'The Milimus Tournament Cup',
            'description' => 'has successfully participated in the ' . $player->tournament->name . ' edition tournament held from ' . Carbon::parse($player->tournament->start_date)->format('F j, Y') . ' to ' . Carbon::parse($player->tournament->end_date)->format('F j, Y') . ' and has demonstrated exceptional sportsmanship, skill, and dedication throughout the competition.',
            'qr_code_url' => 'https://api.qrserver.com/v1/create-qr-code/?size=100x100&data=https://example.com/verify/' . $certificateId,
            'verification_status' => $player->verification_status ?? 'verified',
        ];

        try {
            $pdf = Pdf::loadHTML(view('players.certificate', $data)->render());
            // Ensure remote assets (QR Server API) load correctly
            $pdf->setOptions([
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'isPhpEnabled' => true,
                'isFontSubsettingEnabled' => false,
            ]);
            
            // Relax SSL for remote image fetching if needed
            $pdf->getDomPDF()->setHttpContext(stream_context_create([
                'ssl' => [
                    'verify_peer' => false,
                    'verify_peer_name' => false,
                    'allow_self_signed' => true,
                ],
                'http' => [
                    'timeout' => 30,
                ],
            ]));

            $pdf->setPaper('a4', 'landscape');
            
            $filename = 'Certificate_' . ($player->user->name ?? 'Player') . '_' . now()->format('Y-m-d') . '.pdf';
            $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
            
            return $pdf->download($filename);
        } catch (\Exception $e) {
            \Log::error('PDF Generation Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate PDF: ' . $e->getMessage()], 500);
        }
    }

    public function bulkUpdate(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:players,id',
            'updates' => 'required|array',
        ]);
        
        $updates = $validated['updates'];
        $allowedFields = [
            'status', 'verification_status', 'player_type', 'experience_level',
            'global_rating', 'tournament_rating', 'seeding_position'
        ];
        
        $filteredUpdates = array_intersect_key($updates, array_flip($allowedFields));
        
        Player::whereIn('id', $validated['ids'])->update($filteredUpdates);
        
        return redirect()->route('admin.players.index')->with('success', 'Players updated successfully!');
    }
} 