<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Permission;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Inertia\Response;

class PermissionAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:permissions.view')->only(['index', 'show']);
        $this->middleware('permission:permissions.create')->only(['create', 'store']);
        $this->middleware('permission:permissions.edit')->only(['edit', 'update']);
        $this->middleware('permission:permissions.delete')->only(['destroy', 'bulkDelete']);
        $this->middleware('permission:permissions.manage')->only(['bulkUpdate', 'assignToRole', 'removeFromRole']);
    }

    public function index(Request $request): Response
    {
        $query = Permission::query();

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('display_name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Category filter
        if ($request->filled('category') && $request->category !== 'all') {
            $query->where('category', $request->category);
        }

        // Status filter
        if ($request->filled('status') && $request->status !== 'all') {
            if ($request->status === 'active') {
                $query->where('is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            } elseif ($request->status === 'system') {
                $query->where('is_system', true);
            } elseif ($request->status === 'custom') {
                $query->where('is_system', false);
            }
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'category');
        $sortOrder = $request->get('sort_order', 'asc');
        
        if ($sortBy === 'category') {
            $query->orderBy('category')->orderBy('sort_order')->orderBy('name');
        } else {
            $query->orderBy($sortBy, $sortOrder);
        }

        $permissions = $query->paginate($request->get('per_page', 20))->withQueryString();

        // Get statistics
        $stats = [
            'total' => Permission::count(),
            'active' => Permission::where('is_active', true)->count(),
            'inactive' => Permission::where('is_active', false)->count(),
            'system' => Permission::where('is_system', true)->count(),
            'custom' => Permission::where('is_system', false)->count(),
            'by_category' => Permission::select('category', DB::raw('count(*) as count'))
                ->groupBy('category')
                ->orderBy('category')
                ->get(),
        ];

        return Inertia::render('admin/permissions/index', [
            'permissions' => $permissions,
            'stats' => $stats,
            'categories' => Permission::getCategories(),
            'filters' => $request->only(['search', 'category', 'status', 'sort_by', 'sort_order']),
        ]);
    }

    public function create(): Response
    {
        return Inertia::render('admin/permissions/create', [
            'categories' => Permission::getCategories(),
            'actions' => Permission::getActions(),
        ]);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:permissions,name',
            'display_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|string|max:50',
            'is_system' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
            'metadata' => 'nullable|array',
        ]);

        $permission = Permission::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Permission created successfully',
            'permission' => $permission,
        ]);
    }

    public function show(Permission $permission): Response
    {
        $permission->load('roles');

        // Get usage statistics
        $stats = [
            'roles_count' => $permission->roles()->count(),
            'users_count' => DB::table('role_user')
                ->whereIn('role_id', $permission->roles()->pluck('id'))
                ->distinct('user_id')
                ->count(),
        ];

        return Inertia::render('admin/permissions/show', [
            'permission' => $permission,
            'stats' => $stats,
        ]);
    }

    public function edit(Permission $permission): Response
    {
        return Inertia::render('admin/permissions/edit', [
            'permission' => $permission,
            'categories' => Permission::getCategories(),
            'actions' => Permission::getActions(),
        ]);
    }

    public function update(Request $request, Permission $permission): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:permissions,name,' . $permission->id,
            'display_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|string|max:50',
            'is_system' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
            'metadata' => 'nullable|array',
        ]);

        $permission->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Permission updated successfully',
            'permission' => $permission,
        ]);
    }

    public function destroy(Permission $permission): JsonResponse
    {
        if ($permission->is_system) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete system permissions',
            ], 403);
        }

        $permission->delete();

        return response()->json([
            'success' => true,
            'message' => 'Permission deleted successfully',
        ]);
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:permissions,id',
        ]);

        $count = Permission::whereIn('id', $request->ids)
            ->where('is_system', false)
            ->delete();

        return response()->json([
            'success' => true,
            'message' => "Deleted {$count} permissions successfully",
        ]);
    }

    public function bulkUpdate(Request $request): JsonResponse
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:permissions,id',
            'action' => 'required|in:activate,deactivate,change_category',
            'value' => 'nullable|string',
        ]);

        $query = Permission::whereIn('id', $request->ids);

        switch ($request->action) {
            case 'activate':
                $query->update(['is_active' => true]);
                $message = 'Permissions activated successfully';
                break;
            case 'deactivate':
                $query->update(['is_active' => false]);
                $message = 'Permissions deactivated successfully';
                break;
            case 'change_category':
                $query->update(['category' => $request->value]);
                $message = 'Permission category updated successfully';
                break;
            default:
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid bulk action',
                ], 400);
        }

        return response()->json([
            'success' => true,
            'message' => $message,
        ]);
    }

    public function assignToRole(Request $request): JsonResponse
    {
        $request->validate([
            'permission_ids' => 'required|array',
            'permission_ids.*' => 'exists:permissions,id',
            'role_id' => 'required|exists:roles,id',
        ]);

        $role = Role::findOrFail($request->role_id);
        $permissions = Permission::whereIn('id', $request->permission_ids)->get();

        foreach ($permissions as $permission) {
            $role->permissions()->syncWithoutDetaching([$permission->id]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Permissions assigned to role successfully',
        ]);
    }

    public function removeFromRole(Request $request): JsonResponse
    {
        $request->validate([
            'permission_ids' => 'required|array',
            'permission_ids.*' => 'exists:permissions,id',
            'role_id' => 'required|exists:roles,id',
        ]);

        $role = Role::findOrFail($request->role_id);
        $role->permissions()->detach($request->permission_ids);

        return response()->json([
            'success' => true,
            'message' => 'Permissions removed from role successfully',
        ]);
    }
}