<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\User;
use App\Models\Team;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Carbon\Carbon;

class PaymentAdminController extends Controller
{
    public function index(Request $request)
    {
        // Get completed payments with relationships
        $query = Payment::with(['user', 'team'])
            ->where('status', 'completed');

        // Filtering
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('reference_number', 'like', "%{$search}%")
                  ->orWhere('checkout_request_id', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('user', function($subQ) use ($search) {
                      $subQ->where('name', 'like', "%{$search}%")
                           ->orWhere('email', 'like', "%{$search}%");
                  })
                  ->orWhereHas('team', function($subQ) use ($search) {
                      $subQ->where('name', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('payment_method') && $request->payment_method !== 'all') {
            $query->where('payment_method', $request->payment_method);
        }

        if ($request->filled('date_range')) {
            switch ($request->date_range) {
                case 'today':
                    $query->whereDate('paid_at', Carbon::today());
                    break;
                case 'week':
                    $query->whereBetween('paid_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                    break;
                case 'month':
                    $query->whereMonth('paid_at', Carbon::now()->month)
                          ->whereYear('paid_at', Carbon::now()->year);
                    break;
                case 'year':
                    $query->whereYear('paid_at', Carbon::now()->year);
                    break;
            }
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'paid_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $payments = $query->paginate($request->get('per_page', 15))->withQueryString();

        // Calculate statistics
        $stats = $this->getPaymentStatistics($request);

        // Get payment methods for filtering
        $paymentMethods = Payment::where('status', 'completed')
            ->distinct()
            ->pluck('payment_method')
            ->filter()
            ->values();

        return Inertia::render('admin/payments/index', [
            'payments' => $payments,
            'stats' => $stats,
            'paymentMethods' => $paymentMethods,
            'filters' => $request->only(['search', 'payment_method', 'date_range', 'sort_by', 'sort_order']),
        ]);
    }

    public function show(Payment $payment)
    {
        $payment->load(['user', 'team']);
        
        return Inertia::render('admin/payments/show', [
            'payment' => $payment,
        ]);
    }

    protected function getPaymentStatistics($request = null)
    {
        $baseQuery = Payment::where('status', 'completed');

        // Apply same date filtering for stats
        if ($request && $request->filled('date_range')) {
            switch ($request->date_range) {
                case 'today':
                    $baseQuery->whereDate('paid_at', Carbon::today());
                    break;
                case 'week':
                    $baseQuery->whereBetween('paid_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                    break;
                case 'month':
                    $baseQuery->whereMonth('paid_at', Carbon::now()->month)
                              ->whereYear('paid_at', Carbon::now()->year);
                    break;
                case 'year':
                    $baseQuery->whereYear('paid_at', Carbon::now()->year);
                    break;
            }
        }

        $stats = [
            // Basic stats
            'total_payments' => $baseQuery->count(),
            'total_amount' => $baseQuery->sum('amount'),
            'average_amount' => $baseQuery->avg('amount'),
            
            // Payment methods breakdown
            'payment_methods' => $baseQuery->select('payment_method', DB::raw('COUNT(*) as count'), DB::raw('SUM(amount) as total'))
                ->groupBy('payment_method')
                ->get(),
            
            // Daily stats for last 7 days
            'daily_stats' => $baseQuery->select(
                    DB::raw('DATE(paid_at) as date'),
                    DB::raw('COUNT(*) as count'),
                    DB::raw('SUM(amount) as total')
                )
                ->where('paid_at', '>=', Carbon::now()->subDays(7))
                ->groupBy(DB::raw('DATE(paid_at)'))
                ->get(),
                
            // Monthly stats for current year
            'monthly_stats' => $baseQuery->select(
                    DB::raw('MONTH(paid_at) as month'),
                    DB::raw('YEAR(paid_at) as year'),
                    DB::raw('COUNT(*) as count'),
                    DB::raw('SUM(amount) as total')
                )
                ->whereYear('paid_at', Carbon::now()->year)
                ->groupBy(DB::raw('MONTH(paid_at)'), DB::raw('YEAR(paid_at)'))
                ->orderBy('month', 'asc')
                ->get(),
        ];

        // Calculate period comparisons
        if (!$request || !$request->filled('date_range')) {
            // Compare with previous month if no filter applied
            $currentMonth = Payment::where('status', 'completed')
                ->whereMonth('paid_at', Carbon::now()->month)
                ->whereYear('paid_at', Carbon::now()->year);
                
            $previousMonth = Payment::where('status', 'completed')
                ->whereMonth('paid_at', Carbon::now()->subMonth()->month)
                ->whereYear('paid_at', Carbon::now()->subMonth()->year);

            $stats['current_month'] = [
                'count' => $currentMonth->count(),
                'amount' => $currentMonth->sum('amount'),
            ];

            $stats['previous_month'] = [
                'count' => $previousMonth->count(),
                'amount' => $previousMonth->sum('amount'),
            ];

            // Calculate growth percentages
            $stats['growth'] = [
                'count' => $stats['previous_month']['count'] > 0 
                    ? (($stats['current_month']['count'] - $stats['previous_month']['count']) / $stats['previous_month']['count']) * 100 
                    : 0,
                'amount' => $stats['previous_month']['amount'] > 0 
                    ? (($stats['current_month']['amount'] - $stats['previous_month']['amount']) / $stats['previous_month']['amount']) * 100 
                    : 0,
            ];
        }

        return $stats;
    }
}