<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\NotificationSetting;
use App\Services\NotificationRegistryService;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class NotificationSettingsController extends Controller
{
    protected NotificationRegistryService $registryService;

    public function __construct(NotificationRegistryService $registryService)
    {
        $this->registryService = $registryService;
    }

    /**
     * Display notification settings
     */
    public function index(): Response
    {
        $notifications = NotificationSetting::orderBy('category')
            ->orderBy('notification_type')
            ->get()
            ->groupBy('category');

        $categories = NotificationSetting::getCategoriesWithCounts();
        $categoryCounts = $this->registryService->getCategoryCounts();

        return Inertia::render('admin/settings/notifications/index', [
            'notifications' => $notifications,
            'categories' => $categories,
            'categoryCounts' => $categoryCounts,
        ]);
    }

    /**
     * Update a specific notification setting
     */
    public function update(Request $request, NotificationSetting $setting)
    {
        $validated = $request->validate([
            'enabled' => 'required|boolean',
        ]);

        $setting->update($validated);

        return back()->with('success', 'Notification setting updated successfully.');
    }

    /**
     * Bulk update notification settings
     */
    public function bulkUpdate(Request $request)
    {
        $validated = $request->validate([
            'updates' => 'required|array',
            'updates.*.id' => 'required|exists:notification_settings,id',
            'updates.*.enabled' => 'required|boolean',
        ]);

        foreach ($validated['updates'] as $update) {
            NotificationSetting::where('id', $update['id'])->update([
                'enabled' => $update['enabled'],
            ]);
        }

        // Clear cache after bulk update
        NotificationSetting::clearCache();

        $count = count($validated['updates']);
        return back()->with('success', "{$count} notification settings updated successfully.");
    }

    /**
     * Toggle an entire category
     */
    public function toggleCategory(Request $request, string $category)
    {
        $validated = $request->validate([
            'enabled' => 'required|boolean',
        ]);

        $count = NotificationSetting::where('category', $category)
            ->update(['enabled' => $validated['enabled']]);

        // Clear cache after category toggle
        NotificationSetting::clearCache();

        $status = $validated['enabled'] ? 'enabled' : 'disabled';
        return back()->with('success', "{$count} notifications in '{$category}' category {$status} successfully.");
    }

    /**
     * Sync notifications from code to database
     */
    public function sync()
    {
        $result = $this->registryService->syncNotifications();

        return back()->with('success', "Synced {$result['total']} notifications. Created: {$result['created']}, Updated: {$result['updated']}");
    }

    /**
     * Get notification statistics
     */
    public function stats()
    {
        $total = NotificationSetting::count();
        $enabled = NotificationSetting::where('enabled', true)->count();
        $disabled = NotificationSetting::where('enabled', false)->count();
        $categories = NotificationSetting::distinct('category')->count('category');

        return response()->json([
            'total' => $total,
            'enabled' => $enabled,
            'disabled' => $disabled,
            'categories' => $categories,
        ]);
    }
}

