<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Media;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Storage;

class MediaAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:media.view')->only(['index', 'show']);
        $this->middleware('permission:media.edit')->only(['edit', 'update']);
        $this->middleware('permission:media.delete')->only(['destroy', 'bulkDestroy']);
    }

    /**
     * Display the media management page
     */
    public function index(Request $request)
    {
        $query = Media::with('user')->latest();

        // Apply search filter
        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('file_name', 'like', '%' . $request->search . '%')
                  ->orWhere('alt_text', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
            });
        }

        // Apply type filter
        if ($request->filled('type') && $request->type !== 'all') {
            switch ($request->type) {
                case 'images':
                    $query->images();
                    break;
                case 'videos':
                    $query->videos();
                    break;
                case 'documents':
                    $query->documents();
                    break;
                case 'audio':
                    $query->audios();
                    break;
            }
        }

        // Apply collection filter
        if ($request->filled('collection') && $request->collection !== 'all') {
            $query->where('collection_name', $request->collection);
        }

        $media = $query->paginate(24)->withQueryString();

        // Get collections for filter dropdown
        $collections = Media::select('collection_name')
            ->distinct()
            ->whereNotNull('collection_name')
            ->pluck('collection_name')
            ->map(function ($collection) {
                return [
                    'value' => $collection,
                    'label' => ucfirst(str_replace('-', ' ', $collection)),
                    'count' => Media::where('collection_name', $collection)->count()
                ];
            });

        return Inertia::render('admin/media/index', [
            'media' => $media,
            'collections' => $collections,
            'filters' => $request->only(['search', 'type', 'collection']),
            'stats' => [
                'total' => Media::count(),
                'images' => Media::images()->count(),
                'videos' => Media::videos()->count(),
                'documents' => Media::documents()->count(),
                'audio' => Media::audios()->count(),
                'size' => $this->formatBytes(Media::sum('size'))
            ]
        ]);
    }

    /**
     * Show the media details page
     */
    public function show(Media $media)
    {
        $media->load('user');
        
        return Inertia::render('admin/media/show', [
            'media' => $media
        ]);
    }

    /**
     * Show the edit form
     */
    public function edit(Media $media)
    {
        $media->load('user');
        
        // Get available collections
        $collections = Media::select('collection_name')
            ->distinct()
            ->whereNotNull('collection_name')
            ->pluck('collection_name')
            ->sort()
            ->values();

        return Inertia::render('admin/media/edit', [
            'media' => $media,
            'collections' => $collections
        ]);
    }

    /**
     * Update media details
     */
    public function update(Request $request, Media $media)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'alt_text' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:1000',
            'collection_name' => 'nullable|string|max:255',
        ]);

        $media->update($validated);

        return redirect()
            ->route('admin.media.show', $media)
            ->with('success', 'Media updated successfully.');
    }

    /**
     * Delete media file
     */
    public function destroy(Media $media)
    {
        try {
            $media->delete();
            
            return redirect()
                ->route('admin.media.index')
                ->with('success', 'Media deleted successfully.');
        } catch (\Exception $e) {
            return redirect()
                ->route('admin.media.index')
                ->with('error', 'Failed to delete media: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete media files
     */
    public function bulkDestroy(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'integer|exists:media,id'
        ]);

        try {
            $media = Media::whereIn('id', $request->ids)->get();
            $count = $media->count();
            
            foreach ($media as $item) {
                $item->delete();
            }

            return redirect()
                ->route('admin.media.index')
                ->with('success', "{$count} media files deleted successfully.");
        } catch (\Exception $e) {
            return redirect()
                ->route('admin.media.index')
                ->with('error', 'Failed to delete media files: ' . $e->getMessage());
        }
    }

    /**
     * Format bytes to human readable format
     */
    private function formatBytes($bytes, $precision = 2)
    {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');

        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, $precision) . ' ' . $units[$i];
    }
} 